import bip.base.bipelt

class _XrefTypes(object):
    """
        Static class allowing to get the type of the xref. Internall class.
    """
    # data xref types
    dr_O  = 0x01 #: Offset
    dr_W  = 0x02 #: Write access
    dr_R  = 0x03 #: Read access
    dr_T  = 0x04 #: Text, forced operands only
    dr_I  = 0x05 #: Informational

    # code xref types
    fl_CF = 0x10 #: Call Far, function at the reference location
    fl_CN = 0x11 #: Call Near, function at the reference location
    fl_JF = 0x12 #: Jump Far
    fl_JN = 0x13 #: Jump Near
    fl_USobsolete = 0x14 #: obsolete ??
    fl_F  = 0x15 #: Ordinary flow

    # other flags in the type field
    XREF_USER = 0x20 # User specified xref
    XREF_TAIL = 0x40 # Reference to tail byte in extrn symbols.
    XREF_BASE = 0x80 # Reference to the base part of an offset.
    XREF_MASK = 0x1F # Mask to get xref type.
    XREF_PASTEND = 0x100

class BipXref(object):
    """
        Base class for representing an xref between two elements.

        .. todo:: classmethod allowing to create xref

        .. todo:: classmethod allowing to destroy xref
    """

    def __init__(self, xref):
        """
            Constructor for an xref object. There is few reason to directly
            use this constructor, properties of :class:`BipBaseElt` and of
            :class:`BipFunction` should allow to directly get the xref.

            :param xref: The xref object used in IDA.
            :type xref: ``idautils._xref`` .
        """
        self._xref = xref #: Internal object representing the xref in IDA

    @property
    def _type(self):
        """
            Return the type of the xref with the mask applied.
        """
        return self._xref.type & _XrefTypes.XREF_MASK

    @property
    def is_userdef(self):
        """
            Return True if the xref was added by a user.
        """
        return self._xref.type & _XrefTypes.XREF_USER != 0

    @property
    def src_ea(self):
        """
            Source address (or member id for xref on struct) of the XRef:
            the object which created it. This is equivalent to the *from* in
            the standard IDA api.
        """
        return self._xref.frm

    @property
    def src(self):
        """
            Property which allow to get an object representing the element at
            the source address.

            :return: An object representing the element at the source address.
            :rtypes: An :class:`BipBaseElt` or one of its subclasses. See
                :func:`GetElt`.
        """
        return bip.base.bipelt.GetElt(self.src_ea)

    @property
    def dst_ea(self):
        """
            Destination address (or member id for xref on struct) of the XRef.
            This is equivalent to the *to* in the standard IDA api.
        """
        return self._xref.to

    @property
    def dst(self):
        """
            Property which allow to get an object representing the element at
            the destination address or id.

            :return: An object representing the element at the destination
                address.
            :rtypes: An :class:`BipBaseElt` or one of its subclasses. See
                :func:`GetElt`.
        """
        return bip.base.bipelt.GetElt(self.dst_ea)

    ######### PROPERTY FOR CODE FLOW XREF ###########

    @property
    def is_codepath(self):
        """
            Property indicating if the xref indicate a path for the control
            flow of the program. This can be because of a jmp, a call or simply
            a normal path.
        """
        return self._type >= 0x10 # see _XrefTypes

    @property
    def is_call(self):
        """
            Property indicating if this xref was generated by a call
            intruction: the source is the call instruction and the destination
            is the beginning of a function.
        """
        return self._type in [_XrefTypes.fl_CF, _XrefTypes.fl_CN]

    @property
    def is_jmp(self):
        """
            Property indicating if this xref was generated by a jmp
            instruction.
        """
        return self._type in [_XrefTypes.fl_JF, _XrefTypes.fl_JN]

    @property
    def is_ordinaryflow(self):
        """
            Property indicating if this xref indicate an ordinary flow (not
            jump or call) between two instructions.
        """
        return self._type == _XrefTypes.fl_F

    ########### PROPERTY FOR DATA XREF ###################

    @property
    def is_offset(self):
        """
            The xref is created by an offset, it can be a ``lea`` instruction
            or a reference on a address in an array.
        """
        return self._type == _XrefTypes.dr_O

    @property
    def is_write_access(self):
        """
            The xref is created by a write access.
        """
        return self._type == _XrefTypes.dr_W

    @property
    def is_read_access(self):
        """
            The xref is created by a read access.
        """
        return self._type == _XrefTypes.dr_R

    ########## PROPERTY OF DST & SRC ################

    @property
    def is_dst_code(self):
        """
            Property which indicate if the destination of the xref is code.

            :return: True or False.
        """
        return self.dst.is_code

    @property
    def is_src_code(self):
        """
            Property which indicate if the source of the xref is code.

            :return: True or False.
        """
        return self.src.is_code

