import idc
import ida_idaapi
import ida_name
import ida_funcs
import ida_xref
import ida_bytes
import ida_kernwin
import ida_graph
import ida_lines
import ida_moves
import ida_auto
import ida_ua
import idautils

import re
import inspect
import os
import sys
import codecs
import tempfile
import traceback
import time

# import internal libraries
import get_func_relation
import tinfo
#import jump
#import config_base
import cto_base
import icon
import syncui
import xor_loop_detector
ida_idaapi.require("get_func_relation")
ida_idaapi.require("tinfo")
#ida_idaapi.require("jump")
#ida_idaapi.require("config_base")
ida_idaapi.require("cto_base")
ida_idaapi.require("icon")
ida_idaapi.require("syncui")
ida_idaapi.require("xor_loop_detector")

g_ida_view = "IDA View-A"
g_max_recursive = 10

FT_UNK = get_func_relation.FT_UNK
FT_GEN = get_func_relation.FT_GEN
FT_LIB = get_func_relation.FT_LIB
FT_API = get_func_relation.FT_API
FT_MEM = get_func_relation.FT_MEM
FT_VAR = get_func_relation.FT_VAR
FT_STR = get_func_relation.FT_STR

if not hasattr(ida_kernwin, "WOPN_NOT_CLOSED_BY_ESC"):
    setattr(ida_kernwin, "WOPN_NOT_CLOSED_BY_ESC", 0x100) # 7.5 lacks the definition

if not hasattr(ida_kernwin, "CVNF_LAZY"):
    setattr(ida_kernwin, "CVNF_LAZY", 0x1) # 7.5 lacks the definition
if not hasattr(ida_kernwin, "CVNF_JUMP"):
    setattr(ida_kernwin, "CVNF_JUMP", 0x2) # 7.5 lacks the definition
if not hasattr(ida_kernwin, "CVNF_ACT"):
    setattr(ida_kernwin, "CVNF_ACT", 0x4) # 7.5 lacks the definition

class CallTreeOverviewer(cto_base.cto_base, ida_graph.GraphViewer):
    orig_title = "CTO"
    DO_NOT_SKIP = 0
    SKIP_CHILDREN = 1
    SKIP_PARENTS = 2
    #icon_data = g_icon_data
    def __init__(self, start_ea, end_ea=ida_idaapi.BADADDR, max_depth=1, cto_data=None, close_open=True, title_postfix="", parent=None, skip=DO_NOT_SKIP, debug=False, skip_lib=True, skip_api=True):
        # generate title
        self.title = self.orig_title
        if title_postfix:
            self.title += title_postfix
        
        # init super class
        ida_graph.GraphViewer.__init__(self, self.title, close_open=close_open)
        
        self.ida_view = g_ida_view
        self.skip_children = False
        self.skip_parents = False
        if skip == self.SKIP_CHILDREN:
            self.skip_children = True
        elif skip == self.SKIP_PARENTS:
            self.skip_parents = True
        self.skip_api = True
        if not skip_api:
            self.skip_api = False
        self.skip_lib = True
        if not skip_lib:
            self.skip_lib = False

        self.icon = icon.icon_handler(icon_data=icon.g_icon_data_ascii, hexify=True)
        
        # basic config
        f = ida_funcs.get_func(start_ea)
        if f:
            start_ea = f.start_ea
        self.start_ea = start_ea
        
        self.end_ea = end_ea
        self.max_depth = max_depth
        #self.f = None
        self.sub_graphs = []
        self.node_types = {}
        
        # init cto base
        cto_base.cto_base.__init__(self, cto_data, debug)
        
        self.parent = None
        if parent and isinstance(parent, CallTreeOverviewer):
            self.parent = parent
        
        # for filtering out nodes
        self.max_recursive = 30
        self.clear_internal_caches()

        self.color_settings()
        
        self.remove_comment = True
        self.limit_depth = 10
        #self.max_nodes = 1000
        self.max_nodes = 300
        self.maximum_string_length = 20
        self.maximum_comment_length = 50
        self.canvas_margin = 0.07
        
        # node settings
        self.exceeded_node_symbol = '...'
        # see self.color_settings()
        
        self.rm_space_rule = re.compile(r' +')
        
        # Temoprary classes for UI and view hooks.
        # Their classes are only available in init function to avoid to forget unhooking and deleting.
        # Their instances are available while CallTreeOverviewer instance is available because they are generated at the end of init function below.
        class my_ui_hooks_t(syncui.my_ui_hooks_t):
            def _log(self, *msg):
                if self.v().config.debug:
                    self.v().dbg_print(">>> MyUiHook: %s" % " ".join([str(x) for x in msg]))
            
            def update_tif(self, ea, name=None):
                self.v().update_caller_tif(ea, name)
                
            def clear_history(self):
                pass
                #self.v().exec_ui_action("EmptyStack")
                
            def chk_dark_mode(self):
                refresh_flag = False
                if self.v().is_dark_mode_with_main():
                    if not self.v().config.dark_mode:
                        self.v().config.dark_mode = True
                        refresh_flag = True
                        self.v().change_widget_icon(bg_change=self.v().config.dark_mode)
                else:
                    if self.v().config.dark_mode:
                        self.v().config.dark_mode = False
                        refresh_flag = True
                        self.v().change_widget_icon(bg_change=self.v().config.dark_mode)
                return refresh_flag
            
        # observing "IDA View-A" window
        class my_view_hooks_t(syncui.my_view_hooks_t):
            def _log(self, *msg):
                if self.v().config.debug:
                    self.v().dbg_print(">>> MyViewHook: %s" % " ".join([str(x) for x in msg]))
                
            def view_loc_change_on_widget_b(self, now, was):
                now_node = now.renderer_info().pos.node
                was_node = was.renderer_info().pos.node
                if self.v().config.debug:
                    self._log("Click on the node id %d, was on %d" % (now_node, was_node))

                if now_node >= 0 and now_node != was_node:
                            
                    if now_node in self.v().node_ids:
                        if self.v().config.debug:
                            self._log("Click on the node id %d in node_ids list" % (now_node))
                                    
                        # jump to the ea corresponded to the now_node in IDA View-A.
                        # this jump leads to another view_loc_changed event for the CTO widget that will call self.update_widget_b() above.
                        self.v().jumpto(self.v().node_ids[now_node])
                        self.v().get_focus(self.v().GetWidget())
            
            def update_widget_b_ea(self, now_ea, was_ea):
                # Make sure we are in the same function
                if now_ea in self.v().nodes:
                    nid = self.v().nodes[now_ea]
                    if nid < len(self.v()._nodes):
                        if self.v().config.center_node and not self.v().is_node_in_canvas(nid):
                            self.v().do_center_node(nid)
                        ni = ida_graph.node_info_t()
                        ni.frame_color = self.v().selected_frame_color
                        ni.bg_color    = self.v().selected_bg_color
                        self.v().SetNodeInfo(nid, ni, ida_graph.NIF_BG_COLOR|ida_graph.NIF_FRAME_COLOR)
                        
                # remove the previous node frame color
                if now_ea != was_ea:
                    if was_ea in self.v().nodes:
                        nid = self.v().nodes[was_ea]
                        if nid < len(self.v()._nodes):
                            self.v().color_node(nid)
                            
            def _view_click(self, w, ve):
                if self.v().GetWidget() == w:
                    item = None
                    if ve.rtype in [ida_kernwin.TCCRT_GRAPH, ida_kernwin.TCCRT_PROXIMITY]:
                        item = ve.location.item
                    #if item is not None:
                    #    # push now node to the locaction history on the CTO window
                    #    if item.is_node:
                    #        r = self.v().push_lochist_jump(w)
                            
        # wait until IDA gets ready
        r = ida_auto.auto_wait()
        
        # Note that this is still in the init method of the CallTreeOverviewer class
        # get the UI and View Hooks
        self.my_ui_hooks = my_ui_hooks_t(self)
        self.my_view_hooks = my_view_hooks_t(self)
        
        # show the graph
        self.show_graph()
        
        # remove past navigation history of this graph.
        #self.exec_ui_action("EmptyStack")
        
        #r = self.ph.save_data(self.ph.data)
        
    ###############################################################################
    
    # jump to a node with chooser
    class node_chooser_t(ida_kernwin.Choose):
        def __init__(self, title, g, flags=ida_kernwin.Choose.CH_MODAL):
            ida_kernwin.Choose.__init__(
                self,
                title,
                [
                    ["nid", 3],
                    ["Address", 10],
                    ["Name", 20],
                    ["Type", 8],
                    ["Has Exceeded Node", 4],
                ],
                flags=flags,
                embedded=False,
                width=60,
                height=20)

            import weakref
            self.v = weakref.ref(g)

            self.items = self.create_item_list()
            self.icon = 5

        def create_item_list(self):
            items = []
            for i, (text, color) in enumerate(self.v()._nodes):
                
                # for address
                address = "N/A"
                if i in self.v().node_ids and self.v().node_ids[i] != ida_idaapi.BADADDR:
                    address = hex(self.v().node_ids[i]).rstrip("L")
                
                # for has_excceded_nodes
                src = self.v().find_src_node_from_edges(i, text=self.v().exceeded_node_symbol)
                dst = self.v().find_dst_node_from_edges(i, text=self.v().exceeded_node_symbol)
                has_exceeded_node = ""
                if src >= 0 or dst >= 0:
                    has_exceeded_node = "True"
                
                # for text
                text = ida_lines.tag_remove(text)
                text_displayed = text
                if i in self.v().exceeded_node_ids:
                    src = self.v().find_src_node_from_edges(i)
                    dst = self.v().find_dst_node_from_edges(i)
                    if src >= 0:
                        text_displayed += " (parent [%d]: %x)" % (src, self.v().node_ids[src])
                    elif dst >= 0:
                        text_displayed += " (child [%d]: %x)" % (dst, self.v().node_ids[dst])

                # node type
                node_type = "Unknown"
                if i in self.v().node_types:
                    node_type = self.v().node_types[i]
                items.append((str(i), address, text_displayed, node_type, has_exceeded_node))
            
            return items
        
        def OnGetLine(self, n):
            #print("getline %d" % n)
            return self.items[n]
        
        def OnGetSize(self):
            n = len(self.items)
            return n
    
    class func_chooser_t(ida_kernwin.Choose):
        def __init__(self, title, g, flags=ida_kernwin.Choose.CH_MODAL):
            ida_kernwin.Choose.__init__(
                self,
                title,
                [
                    ["Address", 10],
                    ["Name", 20],
                ],
                flags=flags,
                embedded=False,
                width=60,
                height=20)

            import weakref
            self.v = weakref.ref(g)

            self.items = self.create_item_list()
            self.icon = 5
        
        def create_item_list(self):
            items = []
            for func_ea in self.v().func_relations:
                func_type = self.v().func_relations[func_ea]['func_type']
                
                func_name = self.v().get_callee_name(func_ea, func_type)
                if func_ea != ida_idaapi.BADADDR:
                    items.append((hex(func_ea).rstrip("L"), ida_lines.tag_remove(func_name)))
            
            return items
        
        def OnGetLine(self, n):
            #print("getline %d" % n)
            return self.items[n]
        
        def OnGetSize(self):
            n = len(self.items)
            return n
    
    class cref_chooser_t(ida_kernwin.Choose):
        def __init__(self, title, g, ea, flags=ida_kernwin.Choose.CH_MODAL):
            ida_kernwin.Choose.__init__(
                self,
                title,
                [
                    ["Address", 10],
                    ["Name", 20],
                ],
                flags=flags,
                embedded=False,
                width=60,
                height=20)

            import weakref
            self.v = weakref.ref(g)
            self.ea = ea

            self.items = self.create_item_list()
            self.icon = 5
        
        def create_item_list(self):
            items = []
            for ref in idautils.CodeRefsFrom(self.ea, False):
                if ref in self.v().func_relations:
                    func_type = self.v().func_relations[ref]['func_type']
                    func_name = self.v().get_callee_name(ref, func_type)
                    if ref != ida_idaapi.BADADDR:
                        items.append((hex(ref).rstrip("L"), ida_lines.tag_remove(func_name)))
            
            return items
        
        def OnGetLine(self, n):
            #print("getline %d" % n)
            return self.items[n]
        
        def OnGetSize(self):
            n = len(self.items)
            return n
    
    class next_node_chooser_t(ida_kernwin.Choose):
        def __init__(self, title, g, node_id, direction, flags=ida_kernwin.Choose.CH_MODAL):
            ida_kernwin.Choose.__init__(
                self,
                title,
                [
                    ["nid", 3],
                    ["Address", 10],
                    ["Name", 20],
                ],
                flags=flags,
                embedded=False,
                width=60,
                height=20)

            import weakref
            self.v = weakref.ref(g)
            self.node_id = node_id
            self.direction = direction
            
            self.items = self.create_item_list()
            self.icon = 5
        
        def create_item_list(self):
            items = []
            node_id = self.node_id
            
            if self.direction == 'parents':
                nid_iter = self.v().find_src_nodes_from_edges(node_id)
            else:
                nid_iter = self.v().find_dst_nodes_from_edges(node_id)
            
            for nid in nid_iter:
                name = self.v()._nodes[nid][0]
                address = "N/A"
                if nid in self.v().exceeded_node_ids:
                    next_node_ea = self.v().exceeded_node_ids[nid]
                    next_node_id = self.v().nodes[next_node_ea]
                    address += " (%s nid:%d @ %x)" % (self.direction, next_node_id, next_node_ea)
                elif nid in self.v().node_ids:
                    address = hex(self.v().node_ids[nid]).rstrip("L")
                
                items.append((str(nid), address, ida_lines.tag_remove(name)))
            
            return items
        
        def OnGetLine(self, n):
            #print("getline %d" % n)
            return self.items[n]
        
        def OnGetSize(self):
            n = len(self.items)
            return n
    
    # for right-click context menu
    class _base_graph_action_handler_t(ida_kernwin.action_handler_t):
        def __init__(self, g):
            ida_kernwin.action_handler_t.__init__(self)
            import weakref
            self.v = weakref.ref(g)
        
        def update(self, ctx):
            return ida_kernwin.AST_ENABLE_ALWAYS

    class expand_collapse_node(_base_graph_action_handler_t):
        def activate(self, ctx):
            r = self.v().get_selected_node()
            if r:
                if len(r) == 1:
                    self.v().OnDblClick(r[0])
            return 1
        
    class hint_printer(_base_graph_action_handler_t):
        def activate(self, ctx):
            x = self.v().get_node_hint()
            if x:
                ida_kernwin.msg(x)
            return 1
    
    class add_cref_from(_base_graph_action_handler_t):
        def activate(self, ctx):
            r = self.v().get_selected_node()
            node_ea = ida_idaapi.BADADDR
            if r:
                if len(r) == 1:
                    nid = r[0]
                    if nid in self.v().node_ids:
                        node_ea = self.v().node_ids[nid]
            if node_ea != ida_idaapi.BADADDR and self.v().add_cref(node_ea, CallTreeOverviewer.func_chooser_t):
                self.v().partial_cache_update(node_ea)
                #self.v().exec_ui_action("EmptyStack")
                self.v().refresh()
                ida_kernwin.msg("added the cref to the node." + os.linesep)
            return 1
    
    class del_cref_from(_base_graph_action_handler_t):
        def activate(self, ctx):
            r = self.v().get_selected_node()
            node_ea = ida_idaapi.BADADDR
            if r:
                if len(r) == 1:
                    nid = r[0]
                    if nid in self.v().node_ids:
                        node_ea = self.v().node_ids[nid]
            if node_ea != ida_idaapi.BADADDR and self.v().del_cref(node_ea, CallTreeOverviewer.cref_chooser_t):
                self.v().partial_cache_update(node_ea)
                #self.v().exec_ui_action("EmptyStack")
                self.v().refresh()
                ida_kernwin.msg("deleted the cref from the node." + os.linesep)
    
    class go_to_node(_base_graph_action_handler_t):
        def __init__(self, g, direction):
            CallTreeOverviewer._base_graph_action_handler_t.__init__(self, g)
            self.direction = direction
            
        def activate(self, ctx):
            r = self.v().get_selected_node()
            if r:
                if len(r) == 1:
                    nid = r[0]
                    nnc = CallTreeOverviewer.next_node_chooser_t("Choose the %s" % self.direction, self.v(), nid, self.direction)
                    selected = nnc.Show(modal=True)
                    if selected >= 0:
                        snid = nnc.items[selected][0]
                        nid = int(snid)
                        if self.v().config.center_node and not self.v().is_node_in_canvas(nid):
                            self.v().do_center_node(nid)
                        self.v().select(nid)
                        
            return 1

    class path_finder(_base_graph_action_handler_t):
        def __init__(self, g, skip, act_name):
            CallTreeOverviewer._base_graph_action_handler_t.__init__(self, g)
            self.skip = skip
            self.act_name = act_name
            
        def _activate(self, ctx):
            r = self.v().get_selected_node()
            if r:
                if len(r) == 1:
                    nid = r[0]
                    if nid in self.v().node_ids:
                        node_ea = self.v().node_ids[nid]
                        
                        parent = self.v()
                        if self.v().parent:
                            parent = self.v().parent
                        title = self.v().orig_title + "_%x" % node_ea
                        
                        flag = False
                        for sg in self.v().sub_graphs:
                            if sg.title == title:
                                flag = True
                                break
                        if parent.start_ea == node_ea:
                            flag = True
                        
                        if flag:
                            ida_kernwin.msg("%s is already displayed as a subgraph. Close it first.%s" % (title, os.linesep))
                        else:
                            dst_ea = ida_idaapi.BADADDR
                            depth = 1
                            if self.skip == CallTreeOverviewer.SKIP_CHILDREN:
                                depth = 3

                            exec_flag = True
                            src_ea = node_ea
                            skip_api = True
                            skip_lib = True
                            if self.act_name.startswith("path_finder_start_end") or self.act_name.startswith("path_finder_end_start") or self.act_name.startswith("path_finder_start_end_skip") or self.act_name.startswith("path_finder_end_start_skip"):
                                # choosing destination with a chooser
                                fc = CallTreeOverviewer.func_chooser_t("Choose the destination", self.v())
                                selected = fc.Show(modal=True)
                                if selected >= 0:
                                    end_ea, _ = fc.items[selected]
                                    dst_ea = int(end_ea, 16)
                                    depth = -1
                                    skip_api = False
                                    skip_lib = False
                                    if self.act_name.startswith("path_finder_end_start") or self.act_name.startswith("path_finder_end_start_skip"):
                                        _ = dst_ea
                                        dst_ea = src_ea
                                        src_ea = _
                                    if self.act_name.startswith("path_finder_start_end_skip") or self.act_name.startswith("path_finder_end_start_skip"):
                                        skip_api = True
                                        skip_lib = True
                                else:
                                    exec_flag = False
                                    ida_kernwin.msg("You did not select an address.%s" % (os.linesep))

                                
                            if exec_flag:
                                g = CallTreeOverviewer(src_ea, end_ea=dst_ea, max_depth=depth, cto_data=self.v().cto_data, close_open=True, title_postfix="_%x" % node_ea, parent=parent, skip=self.skip, skip_api=skip_api, skip_lib=skip_lib)
                                if g and g.parent:
                                    g.parent.sub_graphs.append(g)
        def activate(self, ctx):
            try:
                self._activate(ctx)
            except Exception as e:
                exc_type, exc_obj, tb = sys.exc_info()
                lineno = tb.tb_lineno
                ida_kernwin.msg("Got a unexpected error (%s: %s) (%d)%s" % (str(type(e)), e, lineno, os.linesep))
                traceback.print_exc()
                return 0
            return 1

    class change_primary_node(_base_graph_action_handler_t):
        def activate(self, ctx):
            r = self.v().get_selected_node()
            if r:
                if len(r) == 1 and r[0] in self.v().node_ids:
                    old_start_ea = self.v().start_ea
                    start_ea = self.v().node_ids[r[0]]
                    f = ida_funcs.get_func(start_ea)
                    if f:
                        start_ea = f.start_ea
                    self.v().start_ea = start_ea
                    self.v().force_reload()
                    ida_kernwin.msg("Change the primary node to %x from %x.%s" % (start_ea, old_start_ea, os.linesep))
            return 1

    # wrapper for Show()
    def show(self):
        r = False
        try:
            t1 = time.time()
            if self.config.debug: self.dbg_print("Showing...")
            
            r = self.Show()
            if r:
                w = self.GetWidget()
                if w is None:
                    r = False
            
            if self.config.debug: self.dbg_print("Showed!")
            
            t2 = time.time()
            if self.config.debug: self.dbg_print("show() time: %d" % (t2-t1))
            
        except Exception as e:
            exc_type, exc_obj, tb = sys.exc_info()
            lineno = tb.tb_lineno
            ida_kernwin.msg("Got a unexpected error (%s: %s) (%d)%s" % (str(type(e)), e, lineno, os.linesep))
            if self.config.debug:
                self.dbg_print(traceback.format_exc())
            else:
                traceback.print_exc()
        return r
    
    # wrapper for Refresh() that ends up to call OnRefresh() internally.
    def refresh(self, ea=ida_idaapi.BADADDR, center=False):
        if center:
            self.refresh_with_center_node(ea)
        else:
            self._refresh(ea)
            
    def _refresh(self, ea=ida_idaapi.BADADDR, center=False):
        try:
            t1 = time.time()
            if self.config.debug: self.dbg_print("Refreshing...")
            
            w = self.GetWidget()
            if w:
                gv = ida_graph.get_graph_viewer(w)
                mg = ida_graph.get_viewer_graph(gv)
                if mg is not None:
                    mg.del_custom_layout()
            self.Refresh()
            #self.exec_ui_action("GraphLayout")
            if self.config.debug: self.dbg_print("Refreshed!")
            t2 = time.time()
            if self.config.debug: self.dbg_print("refresh() time: %d" % (t2-t1))
        except Exception as e:
            exc_type, exc_obj, tb = sys.exc_info()
            lineno = tb.tb_lineno
            ida_kernwin.msg("Got a unexpected error (%s: %s) (%d)%s" % (str(type(e)), e, lineno, os.linesep))
            if self.config.debug:
                self.dbg_print(traceback.format_exc())
            else:
                traceback.print_exc()
    
    # This method is mandary.
    def OnGetText(self, node_id):
        # If it returns self[node_id], IDA can color background nodes
        # automatically. However, you can not use SetNodeInfo for the
        # background of the nodes.
        # Here, it returns just a text instead of a tuple.
        # Then you can use SetNodeInfo afterward.
        try:
            return self[node_id][0]
            #return self[node_id]
        except Exception as e:
            exc_type, exc_obj, tb = sys.exc_info()
            lineno = tb.tb_lineno
            ida_kernwin.msg("Got a unexpected error (%s: %s) (%d)%s" % (str(type(e)), e, lineno, os.linesep))
            if self.config.debug:
                self.dbg_print(traceback.format_exc())
            else:
                traceback.print_exc()
        return ""
    
    def append_result_every_given_items(self, ar_input, ar_result, divided_by=3, separator=", "):
        tmp = []
        for i, name in enumerate(ar_input, 1):
            tmp.append(name)
            if i % divided_by == 0:
                ar_result.append(", ".join(tmp) + ", ")
                tmp = []
        if len(tmp) > 0:
            ar_result.append(", ".join(tmp))
        
    def get_focus(self, w=None):
        if w is None:
            w = self.GetWidget()
        if w is None:
            return False
        if self.config.debug: self.dbg_print("take focus")
        return ida_kernwin.activate_widget(w, True)

    def generate_hint(self, node_id):
        if node_id in self.node_ids:
            ar_result = []
            ea = self.node_ids[node_id]

            if ea in self.func_relations:
                func_type = self.func_relations[ea]['func_type']
                name = self.get_callee_name(ea, func_type) + " (" + hex(ea).rstrip("L") + ")"
                ar_result.append(name)
            else:
                name = ida_name.get_name(ea)
                if name:
                    name += " (" + hex(ea).rstrip("L") + ")"
                else:
                    name = hex(ea).rstrip("L")
                ar_result.append(name)

            f = ida_funcs.get_func(ea)
            if f and f.start_ea == ea:
                fcmt = ida_funcs.get_func_cmt(f, 1)
                if fcmt:
                    ar_result.append("")
                    ar_result.append("[Function Comment]")
                    ar_result.append(fcmt)

            if ea in self.func_relations:
                ar_apis = set([])
                ar_st_libs = set([])
                ar_mem_calls = set([])
                ar_gen_calls = set([])
                for caller, (func_ea, func_type, op, func_name) in [(x, self.func_relations[ea]['children'][x]) for x in self.func_relations[ea]['children']]:
                    if func_name:
                        name = func_name + " (" + hex(caller).rstrip("L") + ")"
                    else:
                        name = self.get_callee_name(func_ea, func_type) + " (" + hex(caller).rstrip("L") + ")"
                    if func_type == FT_API:
                        ar_apis.add(name)
                    elif func_type == FT_LIB:
                        ar_st_libs.add(name)
                    elif func_type == FT_MEM and func_ea == ida_idaapi.BADADDR:
                        name = "%x: %s" % (caller, self.get_space_removed_disasm(caller))
                        ar_mem_calls.add(name)
                    else:
                        ar_gen_calls.add(name)

                if len(ar_apis) > 0:
                    ar_result.append("")
                    ar_result.append("[APIs]")
                    self.append_result_every_given_items(ar_apis, ar_result)
                        
                if len(ar_st_libs) > 0:
                    ar_result.append("")
                    ar_result.append("[Static Linked Libraries]")
                    self.append_result_every_given_items(ar_st_libs, ar_result)
            
                if len(ar_mem_calls) > 0:
                    ar_result.append("")
                    ar_result.append("[Unresolved Indirect Calls]")
                    self.append_result_every_given_items(ar_mem_calls, ar_result)
            
                if len(ar_gen_calls) > 0:
                    ar_result.append("")
                    ar_result.append("[General Internal Calls]")
                    self.append_result_every_given_items(ar_gen_calls, ar_result)
            
                if len(self.func_relations[ea]["strings"]) > 0:
                    ar_result.append("")
                    ar_result.append("[Strings]")
                    for ref_ea in self.func_relations[ea]["strings"]:
                        str_ea = self.func_relations[ea]["strings"][ref_ea][0]
                        str_var_name = ida_name.get_name(str_ea)
                        str_cont = self.func_relations[ea]["strings"][ref_ea][3].replace('\r', '\\r').replace('\n', '\\n')
                        ar_result.append("%x -> %s (%x): %s" % (ref_ea, str_var_name, str_ea, str_cont))
            
                if len(self.func_relations[ea]["gvars"]) > 0:
                    ar_result.append("")
                    ar_result.append("[Global/Static Variables]")
                    for src_ea in self.func_relations[ea]["gvars"]:
                        dst_ea = self.func_relations[ea]["gvars"][src_ea][0]
                        dst_var_name = ida_name.get_name(dst_ea)
                        dst_val = self.func_relations[ea]["gvars"][src_ea][3]
                        ar_result.append("%x -> %s (%x): %s" % (src_ea, dst_var_name, dst_ea, dst_val))
            
                if len(self.func_relations[ea]["struct_offsets"]) > 0:
                    ar_result.append("")
                    ar_result.append("[Struct Members]")
                    for src_ea in self.func_relations[ea]["struct_offsets"]:
                        dst_ea = self.func_relations[ea]["struct_offsets"][src_ea][0]
                        dst_var_name = ida_name.get_name(dst_ea)
                        dst_val = self.func_relations[ea]["struct_offsets"][src_ea][3]
                        opn = self.func_relations[ea]["struct_offsets"][src_ea][2]
                        opstr = idc.print_operand(src_ea, opn)
                        #ar_result.append("%x -> %s (%x): %s" % (src_ea, dst_var_name, dst_ea, dst_val))
                        ar_result.append("%x: %s" % (src_ea, opstr))
            
                if len(self.func_relations[ea]["rcmt"]) > 0:
                    ar_result.append("")
                    ar_result.append("[Repeatable Comments]")
                    for cmt_ea in self.func_relations[ea]["rcmt"]:
                        ar_result.append("%x: %s" % (cmt_ea, self.func_relations[ea]["rcmt"][cmt_ea].replace('\r', '\\r').replace('\n', '\\n')))
                if len(self.func_relations[ea]["cmt"]) > 0:
                    ar_result.append("")
                    ar_result.append("[Comments (for the output of several tools)]")
                    for cmt_ea in self.func_relations[ea]["cmt"]:
                        ar_result.append("%x: %s" % (cmt_ea, self.func_relations[ea]["cmt"][cmt_ea].replace('\r', '\\r').replace('\n', '\\n')))
            else:
                # for a node refering to a string or a global/static variable
                if f and f.start_ea in self.func_relations:
                    if ea in self.func_relations[f.start_ea]["strings"]:
                        ar_result.append("")
                        ar_result.append("[Strings]")
                        str_ea = self.func_relations[f.start_ea]["strings"][ea][0]
                        str_var_name = ida_name.get_name(str_ea)
                        str_cont = self.func_relations[f.start_ea]["strings"][ea][3].replace('\r', '\\r').replace('\n', '\\n')
                        ar_result.append("%x -> %s (%x): %s" % (ea, str_var_name, str_ea, str_cont))
                        
                    if ea in self.func_relations[f.start_ea]["gvars"]:
                        ar_result.append("")
                        ar_result.append("[Global/Static Variables]")
                        str_ea = self.func_relations[f.start_ea]["gvars"][ea][0]
                        str_var_name = ida_name.get_name(str_ea)
                        str_cont = self.func_relations[f.start_ea]["gvars"][ea][3]
                        ar_result.append("%x -> %s (%x): %s" % (ea, str_var_name, str_ea, str_cont))
                        
                    if ea in self.func_relations[f.start_ea]["struct_offsets"]:
                        ar_result.append("")
                        ar_result.append("[Struct Members]")
                        str_ea = self.func_relations[f.start_ea]["struct_offsets"][ea][0]
                        str_var_name = ida_name.get_name(str_ea)
                        str_cont = self.func_relations[f.start_ea]["struct_offsets"][ea][3]
                        opn = self.func_relations[f.start_ea]["struct_offsets"][ea][2]
                        opstr = idc.print_operand(ea, opn)
                        #ar_result.append("%x -> %s (%x): %s" % (ea, str_var_name, str_ea, str_cont))
                        ar_result.append("%x: %s" % (ea, opstr))
                        
                # for a node of a string content or a global/static variable
                else:
                    dref_to = ida_xref.get_first_dref_to(ea)
                    while dref_to != ida_idaapi.BADADDR:
                        tmp_f = ida_funcs.get_func(dref_to)
                        if tmp_f and tmp_f.start_ea in self.func_relations and (dref_to in self.func_relations[tmp_f.start_ea]["strings"] or dref_to in self.func_relations[tmp_f.start_ea]["gvars"]):
                            break
                        dref_to = ida_xref.get_first_dref_to(dref_to)
                        
                    if dref_to != ida_idaapi.BADADDR and dref_to in self.func_relations[tmp_f.start_ea]["strings"]:
                        ar_result.append("")
                        ar_result.append("[Strings]")
                        str_ea = self.func_relations[tmp_f.start_ea]["strings"][dref_to][0]
                        str_var_name = ida_name.get_name(str_ea)
                        str_cont = self.func_relations[tmp_f.start_ea]["strings"][dref_to][3].replace('\r', '\\r').replace('\n', '\\n')
                        ar_result.append("%x -> %s (%x): %s" % (dref_to, str_var_name, str_ea, str_cont))
                        for i, x in enumerate(idautils.DataRefsTo(str_ea)):
                            pass
                        ar_result.append("and %d more references" % (i))
                    if dref_to != ida_idaapi.BADADDR and dref_to in self.func_relations[tmp_f.start_ea]["gvars"]:
                        ar_result.append("")
                        ar_result.append("[Global/Static Variables]")
                        str_ea = self.func_relations[tmp_f.start_ea]["gvars"][dref_to][0]
                        str_var_name = ida_name.get_name(str_ea)
                        str_cont = self.func_relations[tmp_f.start_ea]["gvars"][dref_to][3]
                        ar_result.append("%x -> %s (%x): %s" % (dref_to, str_var_name, str_ea, str_cont))
                        for i, x in enumerate(idautils.DataRefsTo(str_ea)):
                            pass
                        ar_result.append("and %d more references" % (i))
                        
                rcmt = ida_bytes.get_cmt(ea, 1)
                if rcmt:
                    ar_result.append("")
                    ar_result.append("[repeatable Comments]")
                    ar_result.append(rcmt)
                cmt = ida_bytes.get_cmt(ea, 0)
                if cmt:
                    ar_result.append("")
                    ar_result.append("[Comments]")
                    ar_result.append(cmt)
            
            x = os.linesep.join(ar_result)
        # for exceeded nodes
        elif node_id > 0 and node_id in self.exceeded_node_ids:
            x = "<more nodes>"
            next_to_ea = self.exceeded_node_ids[node_id]
            
            ids = list(self.find_src_nodes_from_edges(node_id))
            direction = "parents"
            src = self.find_src_node_from_edges(node_id, text=self.exceeded_node_symbol)
            if src < 0:
                direction = "children"
                ids = list(self.find_dst_nodes_from_edges(node_id))

            f = ida_funcs.get_func(next_to_ea)
            if f:
                func_ea = f.start_ea
                num = len(self.func_relations[func_ea][direction])
                if self.config.skip_caller:
                    num = 0
                    callees = set([])
                    for caller in self.func_relations[func_ea][direction]:
                        callee, _, _, _ = self.func_relations[func_ea][direction][caller]
                        if callee != ida_idaapi.BADADDR:
                            callees.add(callee)
                        # for unresolved indirect calls
                        # count up num for each indirect call instead of adding the callee address because they are represented by BADADDR
                        else:
                            num += 1
                    num += len(callees)
                if len(ids) > 1:
                    num -= len(ids) - 1 # minus one is for an exceeded node

                if num >= 0:
                    x = "<%d more function call nodes>" % num
        else:
            x = ""
        return x
    
    def OnHint(self, node_id):
        """
        Triggered when the graph viewer wants to retrieve hint text associated with a given node

        @return: None if no hint is avail or a string designating the hint
        """
        try:
            x = self.generate_hint(node_id)
        except Exception as e:
            exc_type, exc_obj, tb = sys.exc_info()
            lineno = tb.tb_lineno
            ida_kernwin.msg("Got a unexpected error (%s: %s) (%d)%s" % (str(type(e)), e, lineno, os.linesep))
            if self.config.debug:
                self.dbg_print(traceback.format_exc())
            else:
                traceback.print_exc()
            return ""
        return x
        
    def generate_edge_hint(self, src, dst):
        """
        Triggered when the graph viewer wants to retrieve hint text associated with a edge

        @return: None if no hint is avail or a string designating the hint
        """
        if src in self.node_ids:
            src_str = self._nodes[src][0]
            src_ea = self.node_ids[src]
            f = ida_funcs.get_func(src_ea)
            if f and src_ea not in self.func_relations:
                func_name = ida_funcs.get_func_name(src_ea)
                src_str += " (in %s)" % func_name
        elif src > 0:
            src_str = "<more nodes>"
        else:
            src_str = ""

        if dst in self.node_ids:
            dst_str = self._nodes[dst][0]
            dst_ea = self.node_ids[dst]
            f = ida_funcs.get_func(dst_ea)
            if f and dst_ea not in self.func_relations:
                func_name = ida_funcs.get_func_name(dst_ea)
                dst_str += " (in %s)" % func_name
        elif dst > 0:
            dst_str = "<more nodes>"
        else:
            dst_str = ""
        if not src_str and not dst_str:
            return ""
        return "%s -> %s" % (src_str, dst_str)
    
    def OnEdgeHint(self, src, dst):
        """
        Triggered when the graph viewer wants to retrieve hint text associated with a edge

        @return: None if no hint is avail or a string designating the hint
        """
        try:
            x = self.generate_edge_hint(src, dst)
        except Exception as e:
            exc_type, exc_obj, tb = sys.exc_info()
            lineno = tb.tb_lineno
            ida_kernwin.msg("Got a unexpected error (%s: %s) (%d)%s" % (str(type(e)), e, lineno, os.linesep))
            if self.config.debug:
                self.dbg_print(traceback.format_exc())
            else:
                traceback.print_exc()
            return ""
        return x
        
    def popup_dispatcher(self, form, popup_handle):
        # get the selected node
        r = self.get_selected_node()
        if not r or len(r) != 1:
            return
        
        # Print hint
        actname = "hint_printer:%s" % self.title
        desc = ida_kernwin.action_desc_t(actname, "Print hint", self.hint_printer(self))
        ida_kernwin.attach_dynamic_action_to_popup(form, popup_handle, desc)

        # Change Primary Node
        if r and len(r) == 1:
            nid = r[0]
            if nid in self.node_ids and self.node_ids[nid] in self.func_relations:
                actname = "change_primary_node:%s" % self.title
                desc = ida_kernwin.action_desc_t(actname, "Change the primary node here", self.change_primary_node(self))
                ida_kernwin.attach_dynamic_action_to_popup(form, popup_handle, desc)
        
        # Go to a next to node
        if r and len(r) == 1:
            for d in ["parents", "children"]:
                actname = "go_to_%s:%s" % (d, self.title)
                if d == "parents":
                    desc_direction = "parent"
                else:
                    desc_direction = "child"
                desc = ida_kernwin.action_desc_t(actname, "Go to a %s node" % desc_direction, self.go_to_node(self, d))
                ida_kernwin.attach_dynamic_action_to_popup(form, popup_handle, desc)
        
        # Add a "cref from" to this node
        if r and len(r) == 1:
            nid = r[0]
            if nid in self.node_ids:
                ea = self.node_ids[nid]
                f = ida_funcs.get_func(ea)
                if f and f.start_ea in self.func_relations:
                    if ea in self.func_relations[f.start_ea]['children']:
                        callee, func_type, op, func_name =  self.func_relations[f.start_ea]['children'][ea]
                        if callee == ida_idaapi.BADADDR and not func_name:
                            actname = "add_cref_from:%s" % self.title
                            desc = ida_kernwin.action_desc_t(actname, "Add a \"cref from\" to this node", self.add_cref_from(self))
                            ida_kernwin.attach_dynamic_action_to_popup(form, popup_handle, desc)
        
        # Delete a "cref from" from this node
        if r and len(r) == 1:
            nid = r[0]
            if nid in self.node_ids:
                ea = self.node_ids[nid]
                f = ida_funcs.get_func(ea)
                if f and f.start_ea in self.func_relations:
                    if ea in self.func_relations[f.start_ea]['children']:
                        callee, func_type, op, func_name =  self.func_relations[f.start_ea]['children'][ea]
                        if callee != ida_idaapi.BADADDR and not func_name:
                            actname = "del_cref_from:%s" % self.title
                            desc = ida_kernwin.action_desc_t(actname, "Delete a \"cref from\" to this node", self.del_cref_from(self))
                            ida_kernwin.attach_dynamic_action_to_popup(form, popup_handle, desc)
        
        # Expand/Collapse a node
        actname = "expand_collapse:%s" % self.title
        if r and len(r) == 1:
            msg = ""
            nid = r[0]
            if nid in self.node_ids:
                msg = "Collapse under/over this node"
            elif nid in self.exceeded_node_ids:
                msg = "Expand this"
            if msg:
                desc = ida_kernwin.action_desc_t(actname, msg, self.expand_collapse_node(self))
                ida_kernwin.attach_dynamic_action_to_popup(form, popup_handle, desc)
        
        # Path finder
        pf_args = ((self.SKIP_CHILDREN,  "_skip_children" , "to"     , ""),
                   (self.SKIP_PARENTS,   "_skip_parents"  , "from"   , ""),
                   (self.DO_NOT_SKIP,    "_do_not_skip"   , "from/to", ""),
                   (self.DO_NOT_SKIP,    "_start_end"     , "from"   , " to ... (extremely slow)"),
                   (self.DO_NOT_SKIP,    "_end_start"     , "to"     , " from ... (extremely slow)"),
                   (self.DO_NOT_SKIP,    "_start_end_skip", "from"   , " to ... (tracing til libs and APIs) (very slow)"),
                   (self.DO_NOT_SKIP,    "_end_start_skip", "to"     , " from ... (tracing til libs and APIs) (very slow)"),
                   )
        for skip, act_postfix, direction, direction2 in pf_args:
            actname = "path_finder%s:%s" % (act_postfix, self.title)
            if r and len(r) == 1:
                nid = r[0]
                if nid in self.node_ids:
                    desc = ida_kernwin.action_desc_t(actname, "Find the path(s) %s this node%s" % (direction, direction2), self.path_finder(self, skip, actname))
                    ida_kernwin.attach_dynamic_action_to_popup(form, popup_handle, desc)
                    
    def OnPopup(self, form, popup_handle):
        try:
            self.popup_dispatcher(form, popup_handle)
        except Exception as e:
            exc_type, exc_obj, tb = sys.exc_info()
            lineno = tb.tb_lineno
            ida_kernwin.msg("Got a unexpected error (%s: %s) (%d)%s" % (str(type(e)), e, lineno, os.linesep))
            if self.config.debug:
                self.dbg_print(traceback.format_exc())
            else:
                traceback.print_exc()
            
    def _get_selected_node(self, w=None):
        if w is None:
            w = self.GetWidget()
        if w is None:
            return None
        sel = ida_graph.screen_graph_selection_t()
        gv = ida_graph.get_graph_viewer(w)
        ida_graph.viewer_get_selection(gv, sel)
        if sel:
            for s in sel:
                if s.is_node:
                    if self.config.debug: self.dbg_print("Selected node %d" % s.node)
                    return (s.node,)
                else:
                    if self.config.debug: self.dbg_print("Selected edge %d -> %d" % (s.elp.e.src, s.elp.e.dst))
                    return (s.elp.e.src, s.elp.e.dst)
        return None
    
    def get_selected_node(self, w=None):
        r = None
        try:
            r = self._get_selected_node(w)
        except Exception as e:
            exc_type, exc_obj, tb = sys.exc_info()
            lineno = tb.tb_lineno
            ida_kernwin.msg("Got a unexpected error (%s: %s) (%d)%s" % (str(type(e)), e, lineno, os.linesep))
            if self.config.debug:
                self.dbg_print(traceback.format_exc())
            else:
                traceback.print_exc()
        return r
        
    def OnRefresh(self):
        try:
            if self.config.debug: self.dbg_print("OnRefresh() started.")
            self.clear_all_node_infos()
            self.Clear()
            self.clear_internal_caches(all_clear=False)
            self.color_settings()
            self.draw_call_tree()
            self.color_all_nodes()
            if self.config.debug: self.dbg_print("OnRefresh() finished.")
        except Exception as e:
            exc_type, exc_obj, tb = sys.exc_info()
            lineno = tb.tb_lineno
            ida_kernwin.msg("Got a unexpected error (%s: %s) (%d)%s" % (str(type(e)), e, lineno, os.linesep))
            if self.config.debug:
                self.dbg_print(traceback.format_exc())
            else:
                traceback.print_exc()
            return False
        return True

    # action before quitting
    def _close(self):
        to_be_removed = []
        # for main CTO
        if self.parent is None:
            len_sb = len(self.sub_graphs)
            for i in reversed(range(len_sb)):
                if self.config.debug: self.dbg_print("Quitting %s%s" % (self.sub_graphs[i].title, os.linesep))

                # unhook subgraph's hooks
                # note that it seems that IDA does not wait to finish Close() method of the call graph.
                # that's why it unhooks them manually here.
                r = self.sub_graphs[i].my_ui_hooks.unhook()
                if self.config.debug: self.dbg_print("unhooked my_ui_hooks for %s. result: %s%s" % (self.sub_graphs[i].title, str(r), os.linesep))
                r = self.sub_graphs[i].my_view_hooks.unhook()
                if self.config.debug: self.dbg_print("unhooked my_view_hooks for %s. result: %s%s" % (self.sub_graphs[i].title, str(r), os.linesep))

                # close a debug file if avaiable
                if self.sub_graphs[i].f:
                    fn = self.sub_graphs[i].f.name
                    self.sub_graphs[i].f.close()
                    self.sub_graphs[i].f = None
                    if self.config.debug: self.dbg_print("closed debug file (%s) for %s. %s" % (fn, self.sub_graphs[i].title, os.linesep))

                # close the subgraph window
                self.sub_graphs[i].Close()
                if self.config.debug: self.dbg_print("Close() method executed for %s%s" % (self.sub_graphs[i].title, os.linesep))
                
                if i < len(self.sub_graphs):
                    to_be_removed.append(i)
            
        # for a subgraph
        else:
            for i, sg in enumerate(self.parent.sub_graphs):
                if sg.title == self.title:
                    to_be_removed.append(i)
                    break
                
        # unhook ui hooks and view hooks
        # we do not need to care about what unhook methods are called twice. it might occur when this method (Close()) is called from a subgraph.
        if self.config.debug: self.dbg_print("Unhooking ui and view hooks for %s%s" % (self.title, os.linesep))
        self.my_ui_hooks.unhook()
        self.my_view_hooks.unhook()
        if self.config.debug: self.dbg_print("Unhooked ui and view hooks for %s%s" % (self.title, os.linesep))
        
        # remove subgraphs
        for i in sorted(to_be_removed, reverse=True):
            if self.config.debug: self.dbg_print("removing %s from the subgraph list%s" % (self.title, os.linesep))
            # for main CTO
            if self.parent is None:
                if  len(self.sub_graphs) > i:
                    self.sub_graphs.pop(i)
            # for subgraphs
            else:
                if len(self.parent.sub_graphs) > i:
                    self.parent.sub_graphs.pop(i)
        if self.config.debug: self.dbg_print("Quited %s%s" % (self.title, os.linesep))

        self.close_data()
        if hasattr(self, "sd"):
            self.sd.close()
        
        # close tempfile for debug log
        if self.f:
            self.f.close()
            self.f = None
        
    def close(self):
        try:
            self._close()
        except Exception as e:
            exc_type, exc_obj, tb = sys.exc_info()
            lineno = tb.tb_lineno
            ida_kernwin.msg("Got a unexpected error (%s: %s) (%d)%s" % (str(type(e)), e, lineno, os.linesep))
            if self.config.debug:
                self.dbg_print(traceback.format_exc())
            else:
                traceback.print_exc()
            
    def OnClose(self):
        if self.config.debug: self.dbg_print("Quitting %s%s" % (self.title, os.linesep))
        #r = ida_kernwin.execute_sync(self.close, ida_kernwin.MFF_NOWAIT)
        self.close()
    
    def _refresh_with_center_node(self, ea=ida_idaapi.BADADDR):
        ea = ida_kernwin.get_screen_ea()
        orig_ea = ea
        if ea not in self.nodes:
            f = ida_funcs.get_func(ea)
            if f:
                ea = f.start_ea
            else:
                ea = self.start_ea
        else:
            ea = self.start_ea
        self._refresh(ea)
        if orig_ea in self.nodes:
            ea = orig_ea
        if ea in self.nodes:
            nid = self.nodes[ea]
            if self.config.center_node and not self.is_node_in_canvas(nid):
                self.do_center_node(nid)
                self.select(nid)
            elif nid in self.node_ids:
                self.select(nid)

    def refresh_with_center_node(self, ea=ida_idaapi.BADADDR):
        try:
            self._refresh_with_center_node(ea)
        except Exception as e:
            exc_type, exc_obj, tb = sys.exc_info()
            lineno = tb.tb_lineno
            ida_kernwin.msg("Got a unexpected error (%s: %s) (%d)%s" % (str(type(e)), e, lineno, os.linesep))
            if self.config.debug:
                self.dbg_print(traceback.format_exc())
            else:
                traceback.print_exc()
        
    def force_reload(self):
        ea = ida_kernwin.get_screen_ea()
        f = ida_funcs.get_func(ea)
        if f:
            ea = f.start_ea
        if ea not in self.func_relations:
            ida_kernwin.msg("Must be in a function" + os.linesep)
            return False

        ## clear the navigation history to avoid IDA crashes
        #if ea != self.start_ea:
        #    self.exec_ui_action("EmptyStack")
            
        # replace the primary node ea to screen ea.
        self.start_ea = ea
        self.partial_cache_update(ea)
        self.clear_internal_caches(all_clear=True)
        self.refresh_with_center_node()
        return True

    # ida_graph.viewer_center_on will crash if the give node id is invalid.
    # so we need to protect.
    def do_center_node(self, nid, w=None):
        r = False
        try:
            if w is None:
                w = self.GetWidget()
            if w is None:
                return False
            if nid < len(self._nodes):
                gv = ida_graph.get_graph_viewer(w)
                ida_graph.viewer_center_on(gv, nid)
                r = True
        except Exception as e:
            exc_type, exc_obj, tb = sys.exc_info()
            lineno = tb.tb_lineno
            ida_kernwin.msg("Got a unexpected error (%s: %s) (%d)%s" % (str(type(e)), e, lineno, os.linesep))
            if self.config.debug:
                self.dbg_print(traceback.format_exc())
            else:
                traceback.print_exc()
        return r
        
    def select(self, nid):
        try:
            if nid < len(self._nodes):
                self.Select(nid)
        except Exception as e:
            exc_type, exc_obj, tb = sys.exc_info()
            lineno = tb.tb_lineno
            ida_kernwin.msg("Got a unexpected error (%s: %s) (%d)%s" % (str(type(e)), e, lineno, os.linesep))
            if self.config.debug:
                self.dbg_print(traceback.format_exc())
            else:
                traceback.print_exc()
        
    # this action is impremented in viewer hooks. see my_view_hooks_t
    '''
    def OnClick(self, node_id):
        """
        Triggered when a node is clicked
        @return: False to ignore the click and True otherwise
        """
        self.dbg_print("clicked on node:", node_id, node_text)
        return True
    '''
        
    def trace_additional_points(self, ea, target_ea=ida_idaapi.BADADDR, direction="parents", result=None, force_trace=False, nrecursive=0):
        if result is None:
            result = [ea]

        if self.config.debug:
            self.dbg_print("ea: %x, related_nodes: %s" % (ea, str([hex(x).rstrip("L") for x in self.related_nodes])))
        if target_ea != ida_idaapi.BADADDR and target_ea in self.related_nodes[ea]:
            yield tuple(result)
        elif nrecursive >= self.max_recursive and target_ea == ida_idaapi.BADADDR and force_trace == False:
            if len(self.trace_points_relations[ea][direction]) > 0:
                result.append(ida_idaapi.BADADDR)
                yield tuple(result)
                result.pop(-1)
            else:
                yield tuple(result)
        else:
            # ea reached a limitation
            if ea == ida_idaapi.BADADDR:
                if target_ea == ida_idaapi.BADADDR:
                    yield tuple(result)
            # ea is a leaf node
            elif len(self.trace_points_relations[ea][direction]) == 0:
                if target_ea == ida_idaapi.BADADDR:
                    yield tuple(result)
            # ea is in the middle of the function tree
            else:
                for next_ea in self.trace_points_relations[ea][direction]:
                    # trace a upper or lower function
                    if next_ea not in result:
                        result.append(next_ea)
                        for r in self.trace_additional_points(next_ea, target_ea, direction, result, force_trace, nrecursive+1):
                            yield r
                        result.pop(-1)
                    else:
                        # detecting a recursive call
                        if target_ea == ida_idaapi.BADADDR:
                            result.append(next_ea)
                            yield tuple(result)
                            result.pop(-1)
    
    def filter_nodes(self, node_id):
        refresh_flag = False
        node_ea = self.node_ids[node_id]
        if self.config.debug:
            self.dbg_print("filtering out %x (%d)" % (node_ea, node_id))
        start_ea = ida_idaapi.BADADDR
        if node_ea in self.additional_trace_points:
            start_ea = node_ea
        else:
            if self.config.debug:
                self.dbg_print("keys and quantities of related_nodes:", [(hex(x).rstrip("L"), len(self.related_nodes[x])) for x in self.related_nodes])
            for ea in self.related_nodes:
                if self.config.debug:
                    self.dbg_print("node_ea:", hex(node_ea).rstrip("L"), ", ea of related_nodes'key:", hex(ea).rstrip("L"), ", related_nodes:", [hex(x).rstrip("L") for x in self.related_nodes[ea]])
                if node_ea in self.related_nodes[ea]:
                    start_ea = ea
                    break
        if start_ea == ida_idaapi.BADADDR:
            ida_kernwin.msg("we cannot filter the node" + os.linesep)
            return refresh_flag

        # get the direction to the primary ea        
        direction = None
        # we skip this process if start_ea is the primary node. we need node-to-node tracing.
        if start_ea != self.start_ea:
            r = list(self.trace_additional_points(start_ea, target_ea=self.start_ea, direction="parents"))
            if self.config.debug: self.dbg_print([hex(x).rstrip("L") for y in r for x in y])
            if len(r) > 0:
                direction = "parents"
                if self.config.debug: self.dbg_print("parents1")
            else:
                r = list(self.trace_additional_points(start_ea, target_ea=self.start_ea, direction="children"))
                if self.config.debug: self.dbg_print([hex(x).rstrip("L") for y in r for x in y])
                if len(r) > 0:
                    direction = "children"
                    if self.config.debug: self.dbg_print("children1")
        if direction is None:
            r = list(self.find_path(node_id, end_nid=self.nodes[self.start_ea], direction='up'))
            if self.config.debug: self.dbg_print(r)
            if len(r) > 0:
                if self.config.debug: self.dbg_print("parents2")
                direction = "parents"
            else:
                r = list(self.find_path(node_id, end_nid=self.nodes[self.start_ea], direction='down'))
                if self.config.debug: self.dbg_print(r)
                if len(r) > 0:
                    if self.config.debug: self.dbg_print("children2")
                    direction = "children"
        
        if start_ea == ida_idaapi.BADADDR:
            ida_kernwin.msg("we cannot filter the node" + os.linesep)
            return refresh_flag

        if self.config.debug: self.dbg_print("start_ea:", hex(start_ea).rstrip("L"))
        
        if direction == "children":
            next_nids = self.find_src_nodes_from_edges(node_id)
        else:
            next_nids = self.find_dst_nodes_from_edges(node_id)
        
        # if the next_nid isn't in nodes, it is already filtered out. we do not need to do anything.
        # remove additional points beyond a filtered node
        tobe_removed = set([])
        for next_nid in next_nids:
            if self.config.debug: self.dbg_print("next_nid:", next_nid, "next_ea:", hex(self.node_ids[next_nid]).rstrip("L"))
            if next_nid in self.node_ids:
                node_id_ea = self.node_ids[node_id]
                next_id_ea = self.node_ids[next_nid]
                call_type = 'caller'
                if node_id_ea in self.func_relations:
                    call_type = 'callee'
                if call_type == 'callee':
                    callee_ea = node_id_ea
                else:
                    callee_ea = next_id_ea
                if self.config.debug: self.dbg_print("filtering out node key: %x, value: %x, call_type: %s" % (node_id_ea, callee_ea, call_type))
                self.filtered_nodes[node_id_ea] = (callee_ea, self._nodes[node_id], call_type)
                if self.config.debug:
                    self.dbg_print(direction, [(hex(x).rstrip("L"), hex(self.filtered_nodes[x][0]).rstrip("L"), self.filtered_nodes[x][1], self.filtered_nodes[x][2]) for x in self.filtered_nodes])
                refresh_flag = True
                if len(self.additional_trace_points) > 0:
                    opposite_direction = "parents"
                    if direction == "parents":
                        opposite_direction = "children"
                    for r in self.trace_additional_points(start_ea, target_ea=ida_idaapi.BADADDR, direction=opposite_direction):
                        if self.config.debug:
                            self.dbg_print("opposite_direction (to start address):", opposite_direction, ", start_ea:", hex(start_ea).rstrip("L"), ", node_ea:", hex(node_ea).rstrip("L"), ", trace result:", [hex(x).rstrip("L") for x in r], ", additional_points:", [hex(x).rstrip("L") for x in self.additional_trace_points])
                        remove_flag = False
                        first_flag = True
                        for i, ea in enumerate(r):
                            if self.config.debug: self.dbg_print("i: %d, node_ea:%x, loop_ea: %x, additonal_trace_points:%s, related_nodes: %s" % (i, node_ea, ea, str([hex(x).rstrip("L") for x in self.additional_trace_points]), str([hex(x).rstrip("L") for x in self.related_nodes[ea]])))
                            # if the path starts in the midle of the additional trace points, ignore the first node.
                            if first_flag and self.start_ea == start_ea:
                                # if flag is turned on, remove the additonal tracing point and further ones.
                                if self.config.debug: self.dbg_print("remove_flag turned on, but it will affect next time")
                                remove_flag = True
                                first_flag = False
                                continue
                            elif node_ea in self.additional_trace_points:
                                # if flag is turned on, remove the additonal tracing point and further ones.
                                if self.config.debug: self.dbg_print("remove_flag turned on")
                                remove_flag = True
                                #continue
                            elif ea in self.related_nodes and node_ea in self.related_nodes[ea]:
                                # we need to trace in ea reladted nodes if they have multiple paths.
                                tmp_direction = 'up'
                                if opposite_direction == 'children':
                                    tmp_direction = 'down'
                                for p in self.find_path(self.nodes[ea], end_nid=-1, direction=tmp_direction, nodes_limitation=set([self.nodes[x] for x in self.related_nodes[ea]])):
                                    if self.config.debug: self.dbg_print("path (id):", [x for x in p])
                                    if self.config.debug:
                                        tmp_ar = []
                                        for x in p:
                                            if x in self.node_ids:
                                                tmp_ar.append(hex(self.node_ids[x]).rstrip("L"))
                                            elif x in self.exceeded_node_ids:
                                                tmp_ar.append(hex(self.exceeded_node_ids[x]).rstrip("L") + " (exceeded)")
                                            else:
                                                tmp_ar.append("error_id (%d)" % x)
                                        self.dbg_print("path (ea):", tmp_ar)
                                    if node_id in p:
                                        last_id = p[-1]
                                        if self.config.debug: self.dbg_print("last_id: %d, %x, i: %d, len(r):%d" % (last_id, self.node_ids[last_id], i, len(r)))
                                        if i+1 < len(r):
                                            next_additional_point_ea = r[i+1]
                                            if self.config.debug: self.dbg_print("next_additonal_point: %x" % (next_additional_point_ea))
                                            if direction == "children":
                                                next_nids = self.find_src_nodes_from_edges(last_id)
                                            else:
                                                next_nids = self.find_dst_nodes_from_edges(last_id)
                                            for tmp_id in next_nids:
                                                if tmp_id in self.node_ids:
                                                    if self.config.debug:
                                                        self.dbg_print("tmp_id: %d, tmp_id_ea: %x" % (tmp_id, self.node_ids[tmp_id]))
                                                        self.dbg_print("next id of the last id: %d, %x, next_additional_point_ea: %x" % (tmp_id, self.node_ids[tmp_id], next_additional_point_ea))
                                                    if next_additional_point_ea == self.node_ids[tmp_id]:
                                                        if self.config.debug: self.dbg_print("add %x to be removed list" % self.node_ids[tmp_id])
                                                        tobe_removed.add(self.node_ids[tmp_id])
                                                        remove_flag = True
                                                else:
                                                    if self.config.debug: self.dbg_print("tmp_id: %d, exceeded_tmp_id_ea: %x" % (tmp_id, self.exceeded_node_ids[tmp_id]))
                                # turned on but skip this area into the remove list
                                if remove_flag:
                                    if self.config.debug: self.dbg_print("remove_flag turned on (2)")
                                    continue
                            if remove_flag:
                                if self.config.debug:
                                    self.dbg_print("to be removed %x from the additonal tracing point list" %ea)
                                if ea != self.start_ea:
                                    tobe_removed.add(ea)
                                """
                                if ea == self.start_ea:
                                    for r in self.trace_additional_points(start_ea, direction=direction):
                                        for tmp_ea in r:
                                            tobe_removed.add(tmp_ea)
                                    break
                                else:
                                    tobe_removed.add(ea)
                                """
        """
        # to be impremented if needed
        # second selection
        if self.config.debug: self.dbg_print("to_be_removed:", str([hex(x) for x in tobe_removed]))
        d = 'up'
        if direction == 'parents':
            d = 'down'
        r = list(self.find_path(self.nodes[self.start_ea], end_nid=node_id, direction=d))
        print("r (id):", r)
        print("r (ea):", [hex(self.node_ids[y]) for x in r for y in x])
        print("tobe_removed:", [hex(x) for x in tobe_removed])
        second_tobe_removed = []
        for i, p in enumerate(r):
            for ea in tobe_removed:
                print(self.nodes[ea])
                if self.nodes[ea] in p:
                    if i == len(second_tobe_removed):
                        second_tobe_removed.append([ea])
                    else:
                        second_tobe_removed[i].append(ea)
        if len(second_tobe_removed) > 0:
            final_tobe_removed = set(second_tobe_removed[0])
            for x in second_tobe_removed:
                final_tobe_removed &= set(x)
                print("a second candidate list", [hex(y) for y in x])
            print("final list:", [hex(x) for x in final_tobe_removed])
        """
        
        d = 'down'
        if direction == 'parents':
            d = 'up'
        second_tobe_removed = []
        for ea in tobe_removed:
            if ea in self.additional_trace_points:
                # check if node id has another path and if it has, skip removing.
                flag = True
                r = list(self.find_path(self.nodes[ea], end_nid=self.nodes[self.start_ea], direction=d))
                if self.config.debug: self.dbg_print(r)
                if len(r) > 1:
                    for i, p in enumerate(r):
                        # if a point is not in the path, it means the point has another path.
                        if self.config.debug: self.dbg_print("node_id:", node_id, "nid of an additional_trace_point:", self.nodes[ea], "a path:", p)
                        if node_id not in p:
                            flag = False
                            break
                        """
                        # to be impremented if needed
                        else:
                            for nid in p:
                                #if nid in self.additional_trace_points:
                                if self.node_ids[nid] in tobe_removed:
                                    if i == len(second_tobe_removed):
                                        second_tobe_removed.append([self.node_ids[nid]])
                                    else:
                                        second_tobe_removed[i].append(self.node_ids[nid])
                            print(second_tobe_removed)
                        """
                                    
                if flag:
                    if self.config.debug: self.dbg_print("removing %x from the additonal_trace_points list" % ea)
                    self.additional_trace_points.pop(ea)
                    if ea in self.trace_points_relations:
                        self.trace_points_relations.pop(ea)
                        for tmp_ea in self.trace_points_relations:
                            if ea in self.trace_points_relations[tmp_ea]['parents']:
                                self.trace_points_relations[tmp_ea]['parents'].remove(ea)
                            if ea in self.trace_points_relations[tmp_ea]['children']:
                                self.trace_points_relations[tmp_ea]['children'].remove(ea)
                            
        return refresh_flag
    
    def _expand_collapse_node(self, node_id):
        """
        Triggerd when a node is double-clicked.
        @return: False to ignore the click and True otherwise
        """
        if self.config.debug:
            self.dbg_print("double-clicked on", self[node_id])
        
        refresh_flag = False
        skip_add_trace_points = False
        saved_ea = ida_idaapi.BADADDR
        # for double clicking on an exceeded node to expand the node.
        if node_id in self.exceeded_node_ids:
            if self.config.debug:
                self.dbg_print(node_id, hex(self.exceeded_node_ids[node_id]).rstrip("L"))
            next_ea = self.exceeded_node_ids[node_id]
            src = self.find_src_node_from_edges(node_id)
            dst = self.find_dst_node_from_edges(node_id)
            # next_id will be its previous caller if a callee is filtered. Otherwise, it will be its previous callee.
            if self.config.debug:
                self.dbg_print(src, dst)
            next_id = src
            if dst >= 0:
                next_id = dst
            saved_ea = self.node_ids[next_id]
            # for a filtered out callee
            if self.config.debug:
                self.dbg_print("next_id:", next_id, ", next_ea:", hex(next_ea).rstrip("L"), ", next_node_ea:", hex(self.node_ids[next_id]).rstrip("L"), ", filtered_nodes:", [hex(x).rstrip("L") for x in self.filtered_nodes], ", additonal trace points:", [hex(x).rstrip("L") for x in self.additional_trace_points], ", exceeded_nodes:", [hex(x).rstrip("L") for x in self.exceeded_nodes])
            if next_ea in self.filtered_nodes:
                callee_or_caller = self.filtered_nodes.pop(next_ea)
                self.dbg_print("filtered_node_value:", hex(callee_or_caller[0]).rstrip("L"), ". it might be a callee address.")
                # the double-clicked node was caller. we need to register a callee ea. replace next_ea with the callee ea.
                if next_ea != callee_or_caller[0]:
                    next_ea = callee_or_caller[0]
                    # we will skip adding additional trace points and updating relationships between those points if the next to ea of the exceeded node is not callee but caller.
                    skip_add_trace_points = True
            
            # get the exact node id of the callee/caller node and push to to the additonal trace points's queue
            if not skip_add_trace_points:
                if next_ea not in self.trace_points_relations:
                    self.trace_points_relations[next_ea] = {"parents":set([]), "children":set([])}
                
                # for parents nodes
                src = self.find_src_node_from_edges(next_id, text=self.exceeded_node_symbol)
                if self.config.debug:
                    self.dbg_print("exceeded_node:", src, ", prev_node:", hex(self.node_ids[next_id]).rstrip("L"), next_id, [hex(x).rstrip("L") for x in self.filtered_nodes])
                if src == node_id:
                    self.additional_trace_points[next_ea] = "parents"
                    # build parents/children or succs/preds relations
                    for start_ea in self.related_nodes:
                        if self.config.debug:
                            self.dbg_print("start_ea: %x, next_ea: %x, related_nodes:%s" % (start_ea, next_ea, str([hex(x).rstrip("L") for x in self.related_nodes[start_ea]])))
                        if next_ea in self.related_nodes[start_ea]:
                            self.trace_points_relations[next_ea]["children"].add(start_ea)
                            if start_ea not in self.trace_points_relations:
                                self.trace_points_relations[start_ea] = {"parents":set([]), "children":set([])}
                            self.trace_points_relations[start_ea]["parents"].add(next_ea)
                
                # for children nodes
                dst = self.find_dst_node_from_edges(next_id, text=self.exceeded_node_symbol)
                if self.config.debug:
                    self.dbg_print("exceeded_node:", dst, ", next_node:", hex(self.node_ids[next_id]).rstrip("L"), next_id, [hex(x).rstrip("L") for x in self.filtered_nodes])
                if dst == node_id:
                    self.additional_trace_points[next_ea] = "children"
                    # build parents/children or succs/preds relations
                    for start_ea in self.related_nodes:
                        if self.config.debug:
                            self.dbg_print("start_ea: %x, next_ea: %x, related_nodes:%s" % (start_ea, next_ea, str([hex(x).rstrip("L") for x in self.related_nodes[start_ea]])))
                        if next_ea in self.related_nodes[start_ea]:
                            self.trace_points_relations[next_ea]["parents"].add(start_ea)
                            if start_ea not in self.trace_points_relations:
                                self.trace_points_relations[start_ea] = {"parents":set([]), "children":set([])}
                            self.trace_points_relations[start_ea]["children"].add(next_ea)
            
            refresh_flag = True
        
        # for double-clicking on a general node for filtering out nodes under/over it.
        elif node_id in self.node_ids:
            if self.config.debug:
                self.dbg_print(node_id, hex(self.node_ids[node_id]).rstrip("L"))
            if node_id == self.nodes[self.start_ea]:
                ida_kernwin.msg("The primary node cannot be filtered out." + os.linesep)
                return False
            
            saved_ea = self.node_ids[node_id]

            # check if the node is leaf or not
            src = self.find_src_node_from_edges(node_id)
            dst = self.find_dst_node_from_edges(node_id)
            if src < 0 or dst < 0:
                next_node = src
                direction = "parents"
                if dst >= 0:
                    next_node = dst
                    direction = "children"
                ida_kernwin.msg("if you want to filter out the leaf node, double-click \"%s\" node, which is the %s node of the node.%s" % (ida_lines.tag_remove(self._nodes[next_node][0]), direction, os.linesep))
                return False

            # filter the node
            refresh_flag = self.filter_nodes(node_id)

            ## clear navigation history before decreasing nodes to avoid IDA crashes
            #if refresh_flag:
            #    self.exec_ui_action("EmptyStack")
        
        if refresh_flag:
            self.refresh()
        if saved_ea != ida_idaapi.BADADDR:
            self.jumpto(saved_ea)
            flag = True
            if saved_ea in self.nodes:
                nid = self.nodes[saved_ea]
            elif self.start_ea in self.nodes:
                nid = self.nodes[self.start_ea]
            else:
                nid = -1
                flag = False
            if flag and nid >= 0:
                if self.config.center_node and not self.is_node_in_canvas(nid):
                    self.do_center_node(nid)
                self.select(nid)
        
        if self.config.debug:
            self.dbg_print("OnDblClick() finished.")
        self.get_focus(self.GetWidget())
        return True
    
    def OnDblClick(self, node_id):
        r = False
        try:
            r = self._expand_collapse_node(node_id)
        except Exception as e:
            exc_type, exc_obj, tb = sys.exc_info()
            lineno = tb.tb_lineno
            ida_kernwin.msg("Got a unexpected error (%s: %s) (%d)%s" % (str(type(e)), e, lineno, os.linesep))
            if self.config.debug:
                self.dbg_print(traceback.format_exc())
            else:
                traceback.print_exc()
        return r
    
    def jumpto(self, ea):
        r = False
        try:
            if ea in self.nodes:
                r = self._jumpto(ea)
        except Exception as e:
            exc_type, exc_obj, tb = sys.exc_info()
            lineno = tb.tb_lineno
            ida_kernwin.msg("Got a unexpected error (%s: %s) (%d)%s" % (str(type(e)), e, lineno, os.linesep))
            if self.config.debug:
                self.dbg_print(traceback.format_exc())
            else:
                traceback.print_exc()
        return r

    def exec_ui_action(self, action, w=None):
        if w is None:
            w = self.GetWidget()
        return self._exec_ui_action(action, w)
            
    def get_node_hint(self):
        r = self.get_selected_node()
        if r:
            if len(r) == 1:
                x = self.OnHint(r[0])
            else:
                x = self.OnEdgeHint(*r)
            if x:
                hint = ida_lines.tag_remove(x) + os.linesep
                return hint
        return ""
    
    def OnViewKeydown(self, key, state):
        # for state
        SHIFT = 1
        ALT = 2
        CTRL = 4
        ESC_KEY = 0x1000000
        ENTER_KEY = 0x1000004
        RETURN_KEY = 0x1000005

        if self.config.debug: self.dbg_print("pressed key: %d, state; %d" % (key, state))
        
        c = chr(key & 0xFF)

        # toggle Centering the clicked node
        if key == ESC_KEY and state == 0:
            self.exec_ida_ui_action("Return")
        elif key in [ENTER_KEY, RETURN_KEY] and state == CTRL:
            self.exec_ida_ui_action("UndoReturn")
        elif c == 'C' and state == 0:
            self.config.center_node = not self.config.center_node
            ida_kernwin.msg("centering graph %sabled%s" % ("en" if self.config.center_node else "dis", os.linesep))
        # toggle Debug message
        elif c == 'D' and state == 0:
            self.config.debug = not self.config.debug
            ida_kernwin.msg("debugging %sabled%s" % ("en" if self.config.debug else "dis", os.linesep))
        # Reload the current view
        elif c == 'R' and state == 0:
            self.refresh()
            ida_kernwin.msg("Refreshed." + os.linesep)
        # Force refresh
        elif c == 'F' and state == 0:
            if self.force_reload():
                ida_kernwin.msg("Force reloaded." + os.linesep)
            else:
                ida_kernwin.msg("Not reloaded." + os.linesep)
        # go to the Start address
        elif c == 'S' and state == 0:
            self.jumpto(self.start_ea)
            nid = self.nodes[self.start_ea]
            if self.config.center_node and not self.is_node_in_canvas(nid):
                self.do_center_node(nid)
            self.select(nid)
        # go to the End address
        elif c == 'E':
            if self.end_ea != ida_idaapi.BADADDR:
                self.jumpto(self.end_ea)
                nid = self.nodes[self.end_ea]
                if self.config.center_node and not self.is_node_in_canvas(nid):
                    self.do_center_node(nid)
                self.select(nid)
        # show referred String in functions
        elif c == 'S' and state == ALT:
            self.config.show_strings_nodes = not self.config.show_strings_nodes
            ## remove past navigation history of this graph.
            #self.exec_ui_action("EmptyStack")
            #self.refresh()
            self.refresh_with_center_node()
            ida_kernwin.msg("Strings %sabled%s" % ("en" if self.config.show_strings_nodes else "dis", os.linesep))
        # show referred global/static Variables in functions
        elif c == 'V' and state == 0:
            self.config.show_gvars_nodes = not self.config.show_gvars_nodes
            ## remove past navigation history of this graph.
            #self.exec_ui_action("EmptyStack")
            #self.refresh()
            self.refresh_with_center_node()
            ida_kernwin.msg("global/static Variables %sabled%s" % ("en" if self.config.show_gvars_nodes else "dis", os.linesep))
        # show structure member access in functions
        elif c == 'T' and state == SHIFT:
            self.config.show_stroff_nodes = not self.config.show_stroff_nodes
            ## remove past navigation history of this graph.
            #self.exec_ui_action("EmptyStack")
            #self.refresh()
            self.refresh_with_center_node()
            ida_kernwin.msg("sTructure members %sabled%s" % ("en" if self.config.show_stroff_nodes else "dis", os.linesep))
        # show referred sTring in functions
        elif c == 'O' and state == 0:
            self.config.show_comment_nodes = not self.config.show_comment_nodes
            ## remove past navigation history of this graph.
            #self.exec_ui_action("EmptyStack")
            #self.refresh()
            self.refresh_with_center_node()
            ida_kernwin.msg("repeatable cOmments %sabled%s" % ("en" if self.config.show_comment_nodes else "dis", os.linesep))
        # show unresolved Indrect calls
        elif c == 'I' and state == 0:
            self.config.show_indirect_calls = not self.config.show_indirect_calls
            #self.refresh()
            self.refresh_with_center_node()
            ## remove past navigation history of this graph.
            #self.exec_ui_action("EmptyStack")
            ida_kernwin.msg("unresolved Indirect Calls %sabled%s" % ("en" if self.config.show_indirect_calls else "dis", os.linesep))
        # disable to display cAller functions
        elif c == 'A' and state == 0:
            self.config.skip_caller = not self.config.skip_caller
            ## remove past navigation history of this graph.
            #self.exec_ui_action("EmptyStack")
            self.refresh_with_center_node()
            ida_kernwin.msg("skip cAller %sabled%s" % ("en" if self.config.skip_caller else "dis", os.linesep))
        # show Parent's children node
        elif c == 'P' and state == 0:
            self.config.display_children_in_parent_funcs = not self.config.display_children_in_parent_funcs
            ## remove past navigation history of this graph.
            #self.exec_ui_action("EmptyStack")
            self.refresh_with_center_node()
            ida_kernwin.msg("display child nodes in Parent functions %sabled%s" % ("en" if self.config.display_children_in_parent_funcs else "dis", os.linesep))
        # Update func relations
        elif c == 'U' and state == 0:
            ida_kernwin.show_wait_box("Wait for updating the cache")
            #self.cache_update()
            self.update_data()
            ## remove past navigation history of this graph.
            #self.exec_ui_action("EmptyStack")
            #self.refresh()
            self.refresh_all()
            ida_kernwin.msg("the caches of the function relationships and the referred string were Updated." + os.linesep)
            ida_kernwin.hide_wait_box()
        # Update func relations partially
        elif c == 'U' and state == SHIFT:
            ea = ida_kernwin.get_screen_ea()
            self.partial_cache_update(ea)
            #self.exec_ui_action("EmptyStack")
            self.refresh_all(ea)
            ida_kernwin.msg("the caches of the function relationships and the referred string were Updated partially." + os.linesep)
        # Update func relations partially
        elif c == 'U' and state == CTRL:
            self.cache_cmt_update()
            self.refresh_all()
            ida_kernwin.msg("the caches related to comments were Updated." + os.linesep)
        # Help
        elif c == 'H' and state == 0:
            self.print_help()
        # darK mode
        elif c == 'K' and state == 0:
            self.config.dark_mode = not self.config.dark_mode
            self.change_widget_icon(bg_change=self.config.dark_mode)
            self.color_settings()
            self.refresh()
            ida_kernwin.msg("darK mode %sabled%s" % ("en" if self.config.dark_mode else "dis", os.linesep))
        # print several important caches for debugging
        elif c == '_':
            self.print_caches()
        # Print node hint
        elif c == 'P' and state == SHIFT:
            hint = self.get_node_hint()
            if hint is not None:
                ida_kernwin.msg(hint)
            else:
                ida_kernwin.msg("Select a node first.")
        # go to an address or an address of a function name
        elif c == 'G' and state == 0:
            self.exec_ida_ui_action("JumpAsk")
        # rename a function
        elif c == 'N' and state == 0:
            flag = self.check_and_rename_var()
            if not flag:
                self.get_focus(self.GetWidget())
                return False
        # edit function
        elif c == 'P' and state == ALT:
            self.check_and_rename_func_info()
        # repeatable comment
        elif c == ';':
            self.check_and_add_rcmt()
        # comment
        elif c == ':':
            self.check_and_add_cmt()
        # apply structure
        elif c == 'T' and state == 0:
            self.check_and_apply_strunct()
        # show xrefs to
        elif c == 'X' and state == 0:
            self.check_xrefs()
            #self.exec_ida_ui_action("JumpOpXref")
        # add cref
        elif c == 'A' and state == CTRL:
            ea = ida_kernwin.get_screen_ea()
            #if self.add_cref(ea):
            if self.add_cref(ea, CallTreeOverviewer.func_chooser_t):
                self.partial_cache_update(ea)
                #self.exec_ui_action("EmptyStack")
                #self.refresh()
                self.refresh_with_center_node()
                ida_kernwin.msg("added the cref to the node." + os.linesep)
        # del cref
        elif c == 'D' and state == CTRL:
            ea = ida_kernwin.get_screen_ea()
            #if self.del_cref(ea):
            if self.del_cref(ea, CallTreeOverviewer.cref_chooser_t):
                self.partial_cache_update(ea)
                #self.exec_ui_action("EmptyStack")
                self.refresh()
                ida_kernwin.msg("deleted the cref from the node." + os.linesep)
        elif c == '!' and state == SHIFT:
            ida_kernwin.msg("the maximum depth is one." + os.linesep)
            #if self.max_depth != 1:
            #    # remove past navigation history of this graph.
            #    self.exec_ui_action("EmptyStack")
            self.max_depth = 1
            self.refresh()
        # decrease the maximum depth to dig deeper
        elif c == '-':
            if self.max_depth > 1:
                self.max_depth -= 1
                ida_kernwin.msg("the maximum depth is now %d.%s" % (self.max_depth, os.linesep))
                ## remove past navigation history of this graph.
                #self.exec_ui_action("EmptyStack")
                self.refresh_with_center_node()
            else:
                ida_kernwin.msg("the maximum depth is already one." + os.linesep)
        # increase the maximum depth to dig deeper
        elif c == '+':
            if self.max_depth < self.limit_depth and len(self._nodes) < self.max_nodes:
                self.max_depth += 1
                ida_kernwin.msg("the maximum depth is now %d.%s" % (self.max_depth, os.linesep))
                #self.refresh()
                self.refresh_with_center_node()
            else:
                ida_kernwin.msg("the maximum depth (%d) or the number of the nodes (%d) is too big. Expand a node you want manually.%s" % (self.max_depth, len(self.nodes), os.linesep))
        # show portable config information
        elif c == '*':
            self.print_config_info()
        # jump to a node with chooser
        elif c == 'J' and state == 0:
            ch = self.node_chooser_t("Which node do you want to move to?", self)
            r = ch.Show(modal=True)
            if r >= 0 and len(self._nodes) > r:
                self.select(r)
                ida_kernwin.msg("%d was selected.%s" % (r, os.linesep))
        # detect xor loops
        elif c == 'X' and state == CTRL:
            for func_ea, ea, annotation_type in xor_loop_detector.find_xor_loop():
                ida_kernwin.msg("%x: %s, %x: %s%s" % (func_ea, annotation_type, ea, idc.generate_disasm_line(ea, 0), os.linesep))
            self.cache_cmt_update()
            self.refresh_all()
        
        self.get_focus(self.GetWidget())
        return True

    def print_help(self):
        ida_kernwin.msg("""
[How to use]
- If you see \"""" + self.exceeded_node_symbol + """\", you can double-click such nodes to expand them to look into.
- If you want to filter out nodes under/over a node, double-click the node after single-clicking
  the node once. If you double-click the node directly, it might fail because of centring the node.
  In that case, sigle-click it first, or disable centering a node feature by pressing the "C" key.
  See the [Shortcuts] section below. Note that the node itself is not removed by design.
- If you use the shortcuts below, click the background of the call graph window before pushing a
  shortcut key.
- If the mouse cursor on a node or an edge (a blue arrow), you can see a hint such as the function
  name, refered strings of the node and the other side node of the edge.
- If you see a complete messy nodes layout, right-click on the call graph window and choose
  "Layout graph". It rarely happens, but I do not know how to deal with automatically if it does.
  In that case, please deal with it manually like this way.
- If you right-click a node, you will see some context menu items such as "Find path(s) to this
  node" and "Print hint" (summarised information of a node).

[Shortcuts]
H: Help that you are looking.
R: Refresh the call graph manually.
F: Force relaod for the call graph. This is useful to look into a function out of the
   current call tree without executing this script again.
S: go back to the Start node.
E: go to the End node (if you specify for finding a path between two points).
J: Jump to a displayed node with a chooser.
A: enable/disable displaying cAllers (default is caller/callee mode)
K: enable/disable darK mode.
-: decrease the number of depth for digging into at once.
+: increase the number of depth for digging into at once.
!: set one to the number of depth.
U: Update function relationships which is a cache and used for building the call tree.
Shift+U: Update function relationships partially. It updates only the node on the caches
   and its parent function and its parent functions's children if the node is a caller.
Ctrl+U: Update all comment caches. This is useful for collecting some tools'a results such as
   ironstrings and findcrypt.py.
N: reName a function (this option redirects to IDA View-A so that you can use it transparently).
G: Go to a place (this option redirects to IDA View-A so that you can use it transparently).
X: display Xrefs (this option redirects to IDA View-A so that you can use it transparently).
T: apply a sTructure member to an operand (this option redirects to IDA View-A so that
   you can use it transparently).
;: make repeatable comment (this option redirects to IDA View-A so that you can use it
   transparently).
:: make comment (this option redirects to IDA View-A so that you can use it transparently).
Alt+P: edit function (this option redirects to IDA View-A so that you can use it transparently).
Ctrl+A: add "cref from" on an indirect call/jump node.
Ctrl+D: del "cref from" on an indirect call/jump node.
Shift+P: Print the hint of a selected node.
P: enable/disable to show children nodes in Parents.
I: enable/disable to show unresolved Indirect calls as nodes.
O: enable/disable to show repeatable cOmments as nodes.
V: enable/disable to show gobal/static Variables as nodes.
Alt+S: enable/disable to show referenced Strings in functions as nodes.
Shift+T: enable/disable to show sTructure members as nodes.
Ctrl+X: detect Xor instructions in a loop.
C: enable/disable Centering the node you are looking at on the call graph window.
D: enable/disable Debug mode
_: print several important internal caches for debugging.
*: print config values
""")
    
    # for node backgrounds and node frame colors
    def color_settings(self):
        self.start_color = 0x00ff00
        self.end_color = 0xbfdcf7
        self.ep_color = 0xbfdcf7
        self.lib_color = 0xffffc0
        self.api_color = 0xffffc0
        self.default_color = 0xffffff
        self.transparent_color = 0xffffff
        self.selected_frame_color = 0x0000ff
        self.selected_bg_color = 0x00d8ff
        self.strings_color = 0xc4c4c4
        self.gvars_color = 0xc4c4c4
        self.stroff_color = 0xc4c4c4
        self.comments_color = 0x999999
        if self.config.dark_mode:
            self.start_color = 0x005a00
            self.end_color = 0x2f3c7a
            self.ep_color = 0x2f3c7a
            self.lib_color = 0x703726
            self.api_color = 0x703726
            self.selected_frame_color = 0x0000aa
            self.selected_bg_color = 0x004077
            self.strings_color = self.strings_color ^ 0xffffff
            self.stroff_color = self.stroff_color ^ 0xffffff
            self.comments_color = self.comments_color ^ 0xffffff
    
    def clear_all_node_infos(self):
        try:
            if len(self._nodes) > 0:
                self.DelNodesInfos(*[x for x in range(len(self._nodes))])
        except Exception as e:
            exc_type, exc_obj, tb = sys.exc_info()
            lineno = tb.tb_lineno
            ida_kernwin.msg("Got a unexpected error (%s: %s) (%d)%s" % (str(type(e)), e, lineno, os.linesep))
            if self.config.debug:
                self.dbg_print(traceback.format_exc())
            else:
                traceback.print_exc()
    
    def color_node(self, nid):
        try:
            if len(self._nodes) <= nid:
                return None
            
            self.DelNodesInfos(nid)
            color = self._nodes[nid][1]
            if self.default_color == self.transparent_color and color != self.default_color:
                ni = ida_graph.node_info_t()
                ni.bg_color = color
                self.SetNodeInfo(nid, ni, ida_graph.NIF_BG_COLOR|ida_graph.NIF_FRAME_COLOR)
        except Exception as e:
            exc_type, exc_obj, tb = sys.exc_info()
            lineno = tb.tb_lineno
            ida_kernwin.msg("Got a unexpected error (%s: %s) (%d)%s" % (str(type(e)), e, lineno, os.linesep))
            if self.config.debug:
                self.dbg_print(traceback.format_exc())
            else:
                traceback.print_exc()
    
    def color_all_nodes(self):
        for nid in range(len(self._nodes)):
            self.color_node(nid)
    
    def get_color(self, func_type, target_ea=ida_idaapi.BADADDR):
        color = self.default_color
        if target_ea != ida_idaapi.BADADDR and target_ea == self.end_ea:
            color = self.end_color
        elif func_type == FT_API:
            color = self.api_color
        elif func_type == FT_LIB:
            color = self.lib_color
        return color
    
    # for string color for callee nodes
    def color_callee_str(self, callee_str, func_type):
        color_tag = ida_lines.SCOLOR_CNAME
        if func_type == FT_LIB and self.config.dark_mode:
            color_tag = ida_lines.SCOLOR_LIBNAME
        elif func_type == FT_API:
            color_tag = ida_lines.SCOLOR_IMPNAME
        return ida_lines.COLSTR(callee_str, color_tag)
    
    """
    def get_func_name(self, ea):
        name = idc.get_func_name(ea)
        mask = idc.get_inf_attr(idc.INF_SHORT_DN)
        demangled = idc.demangle_name(name, mask)
        if demangled:
            return demangled
        else:
            return name
    """

    """
    def get_space_removed_disasm(self, ea):
        mnem = idc.print_insn_mnem(ea)
        #mnem = ida_lines.COLSTR(mnem, ida_lines.SCOLOR_INSN)
        op = idc.print_operand(ea, 0)
        if op:
            op2 = idc.print_operand(ea, 1)
            if op2:
                op += ", " + op2
        return "%s %s" % (mnem, op)
    """

    # we can get a disassembly line with string color tag as we use the API in ida_lines.
    def get_space_removed_disasm(self, ea, remove_comment=True):
        #disasm = idc.generate_disasm_line(ea, 0)
        disasm = self.rm_space_rule.sub(" ", ida_lines.generate_disasm_line(ea, 0))
        rcmt = ida_bytes.get_cmt(ea, 1)
        if self.remove_comment and remove_comment and not(self.config.show_comment_nodes and rcmt):
            disasm = disasm.split(";", 1)[0]
        if self.config.show_comment_nodes and rcmt:
            disasm = disasm[:self.maximum_comment_length+ida_lines.tag_strlen(disasm)]
        return disasm
    
    def get_callee_name(self, ea, func_type):
        func_name = ida_funcs.get_func_name(ea)
        func_flags = idc.get_func_attr(ea, idc.FUNCATTR_FLAGS)
        f = ida_funcs.get_func(ea)
        if func_type == FT_API:
            func_name = ida_name.get_name(ea)
        elif func_type == FT_GEN and func_flags & ida_funcs.FUNC_THUNK:
            func_name = self.get_space_removed_disasm(ea)
        if not func_name:
            func_name = hex(ea).rstrip("L")
        # for a func chunk but it's located in a diffrent segment or something like that.
        # it happens in a certain type of packer.
        elif f and ea != f.start_ea:
            func_name = ida_name.get_name(ea)
        return self.color_callee_str(func_name, func_type)
    
    def get_widget_offset(self, w=None):
        if w is None:
            w = self.GetWidget()
        if w is None:
            return None, None
        w_gli = ida_moves.graph_location_info_t()
        if ida_graph.viewer_get_gli(w_gli, w, 0):
            return w_gli.orgx, w_gli.orgy
        return None, None
    
    def get_widget_size(self, w=None):
        if w is None:
            w = self.GetWidget()
        if w is None:
            return None, None
        x, y = self.get_widget_offset(w)
        w_gli = ida_moves.graph_location_info_t()
        if x and y and ida_graph.viewer_get_gli(w_gli, w, ida_graph.GLICTL_CENTER):
            return (w_gli.orgx-x)*2, (w_gli.orgy-y)*2
        return None, None
    
    def get_node_offset_of_displayed_canvas(self, nid, w=None):
        if w is None:
            w = self.GetWidget()
        if w is None:
            return (None, None), (None, None)
        gv = ida_graph.get_graph_viewer(w)
        mg = ida_graph.get_viewer_graph(gv)
        if mg is None:
            return (None, None), (None, None)
        return ((mg.nrect(nid).topleft().x,     mg.nrect(nid).topleft().y),
                (mg.nrect(nid).bottomright().x, mg.nrect(nid).bottomright().y))
    
    def get_node_offset(self, nid, w=None):
        if w is None:
            w = self.GetWidget()
        if w is None:
            return (None, None), (None, None)
        co_x, co_y = self.get_widget_offset(w)
        if co_x is not None and co_y is not None:
            (top_left_x, top_lef_y), (bottom_right_x, bottom_right_y) = self.get_node_offset_of_displayed_canvas(nid, w)
            if top_left_x is not None and top_lef_y is not None and bottom_right_x is not None and bottom_right_y is not None:
                return (top_left_x-co_x, top_lef_y-co_y), (bottom_right_x-co_x, bottom_right_y-co_y)
        return (None, None), (None, None)
        
    def _is_node_in_canvas(self, nid, w=None, margin=None):
        if nid >= len(self._nodes):
            return False
        
        if w is None:
            w = self.GetWidget()
        if w is None:
            return False
        
        if margin is None:
            margin = self.canvas_margin
        
        (node_tl_x, node_tl_y), (node_br_x, node_br_y) = self.get_node_offset(nid, w)
        canvas_br_x, canvas_br_y = self.get_widget_size(w)
        
        if node_br_x and node_br_y and canvas_br_x and canvas_br_y:
            if node_tl_x < (canvas_br_x*margin) or node_tl_y < (canvas_br_y*margin):
                self.dbg_print("left or top of the node frame is out of the canvas or around the canvas edge")
                return False
            if node_br_x > canvas_br_x - (canvas_br_x*margin) or node_br_y > canvas_br_y - (canvas_br_y*margin):
                self.dbg_print("right or bottom of the node frame is out of the canvas or around the canvas edge")
                return False
        return True
    
    def is_node_in_canvas(self, nid, w=None, margin=None):
        r = False
        try:
            r = self._is_node_in_canvas(nid, w, margin)
        except Exception as e:
            exc_type, exc_obj, tb = sys.exc_info()
            lineno = tb.tb_lineno
            ida_kernwin.msg("Got a unexpected error (%s: %s) (%d)%s" % (str(type(e)), e, lineno, os.linesep))
            if self.config.debug:
                self.dbg_print(traceback.format_exc())
            else:
                traceback.print_exc()
        return r

    def get_node_default_bgcolor(self, w=None, nid=0, adjustment=3):
        bg_color = -1
        if w is None:
            w = self.GetWidget()
        if w is None:
            return bg_color

        self.DelNodesInfos(nid)
        (node_tl_x, node_tl_y), (node_br_x, node_br_y) = self.get_node_offset(nid, w)
        bg_color = self.get_bgcolor(node_tl_x+adjustment, node_tl_y+adjustment, w)
        self.color_node(nid)
        return bg_color

    @staticmethod
    def _is_dark_mode(bgcolor, threshold=128):
        if bgcolor >= 0:
            alpha = bgcolor >> 24
            bgcolor &= 0xffffff
            green = bgcolor >> 16
            blue = (bgcolor >> 8) & 0xff
            red = bgcolor & 0xff
            if green < threshold and blue < threshold and red < threshold:
                return True
        return False
    
    def is_dark_mode(self, w=None):
        if w is None:
            w = self.GetWidget()
        if w is None:
            return False
        
        x, y = self.get_widget_offset(w)
        if x is None:
            x = 0
        if y is None:
            y = 0
        
        #bgcolor = self.get_bgcolor(x, y, w)
        bgcolor = self.get_node_default_bgcolor(w)
        return self._is_dark_mode(bgcolor)
        
    @staticmethod
    def get_main_window():
        try:
            from PyQt5 import QtWidgets
        except ImportError:
            return None
        
        widget = QtWidgets.QApplication.activeWindow()
        QtWidgets.QApplication.focusWidget()
        for widget in [QtWidgets.QApplication.activeWindow(), QtWidgets.QApplication.focusWidget()] + QtWidgets.QApplication.topLevelWidgets():
            while widget:
                if isinstance(widget, QtWidgets.QMainWindow):
                    break
                widget = widget.parent()
            if isinstance(widget, QtWidgets.QMainWindow):
                return widget
        return None
    
    @staticmethod
    def is_dark_mode_with_main():
        try:
            from PyQt5 import QtWidgets
        except ImportError:
            return False

        widget = CallTreeOverviewer.get_main_window()
        if not isinstance(widget, QtWidgets.QMainWindow):
            return False
        bgcolor = CallTreeOverviewer.get_bgcolor(x=0, y=0, w=widget)
        if bgcolor < 0:
            return False
        return CallTreeOverviewer._is_dark_mode(bgcolor)
        
    @staticmethod
    def get_bgcolor(x=0, y=0, w=None):
        bgcolor = -1
        if w is None:
            return bgcolor
        
        try:
            import sip
            from PyQt5 import QtCore
            from PyQt5 import QtWidgets
            from PyQt5 import QtGui
        except ImportError:
            return bgcolor
        
        if str(w).startswith("<Swig Object of type 'TWidget *' at") and str(type(w)) in ["<class 'SwigPyObject'>", "<type 'SwigPyObject'>"]: # type: for py2, class: for py3                                                                                        
            widget = sip.wrapinstance(int(w), QtWidgets.QWidget)
        else:
            widget = w
            
        pixmap = widget.grab(QtCore.QRect(x, y, x+1, y+1))
        image = QtGui.QImage(pixmap.toImage())
        bgcolor = image.pixel(0, 0)
        
        return bgcolor
        
    # this is avaiable after drawing. Do not use it before or during drawing process.
    def _find_src_nodes_from_edges(self, nid, text=""):
        gv = ida_graph.get_graph_viewer(self.GetWidget())
        mg = ida_graph.get_viewer_graph(gv)
        npred = self.mg.npred(nid)
        for i in range(npred):
            pred_id = self.mg.pred(nid, i)
            if text:
                if text == self._nodes[pred_id][0]:
                    yield pred_id
            else:
                yield pred_id
    
    # this is avaiable after drawing. Do not use it before or during drawing process.
    def _find_dst_nodes_from_edges(self, nid, text=""):
        gv = ida_graph.get_graph_viewer(self.GetWidget())
        mg = ida_graph.get_viewer_graph(gv)
        nsucc = self.mg.nsucc(nid)
        for i in range(nsucc):
            succ_id = self.mg.succ(nid, i)
            if text:
                if text == self._nodes[succ_id][0]:
                    yield succ_id
            else:
                yield succ_id
        
    def find_src_nodes_from_edges(self, nid, text=""):
        if nid in self.node_id_relationships:
            for pred_id in self.node_id_relationships[nid]["preds"]:
                if text:
                    if text == self._nodes[pred_id][0]:
                        yield pred_id
                else:
                    yield pred_id
    
    def find_dst_nodes_from_edges(self, nid, text=""):
        if nid in self.node_id_relationships:
            for succ_id in self.node_id_relationships[nid]["succs"]:
                if text:
                    if text == self._nodes[succ_id][0]:
                        yield succ_id
                else:
                    yield succ_id
    
    def find_dst_node_from_edges(self, nid, text=""):
        for x in self.find_dst_nodes_from_edges(nid, text=text):
            # return only the first node
            return x
        return -1
    
    def find_src_node_from_edges(self, nid, text=""):
        for x in self.find_src_nodes_from_edges(nid, text=text):
            # return only the first node
            return x
        return -1
    
    def trace_edges(self, tracer, start_nid, end_nid=-1, nodes_limitation=None, stop_primary_node=False, result=None, i=0):
        if result is None:
            result = []
        if nodes_limitation is None:
            nodes_limitation = []
        
        if i > self.max_recursive:
            if end_nid == -1:
                yield tuple(result)
                return
        # detect a recursive call
        elif start_nid in result:
            if end_nid == -1:
                yield tuple(result)
                return
        elif start_nid not in nodes_limitation:
            if end_nid == -1:
                yield tuple(result)
                return
        
        result.append(start_nid)
        if stop_primary_node and start_nid == self.nodes[self.start_ea]:
            yield tuple(result)
            return
        else:
            flag = False
            for dst in tracer(start_nid):
                flag = True
                if dst >= 0:
                    result.append(dst)
                    if dst == end_nid:
                        yield tuple(result)
                        result.pop(-1)
                        break
                    elif stop_primary_node and dst == self.nodes[self.start_ea]:
                        yield tuple(result)
                        result.pop(-1)
                        break
                    # recursive call
                    elif dst in result:
                        if end_nid == -1:
                            yield tuple(result)
                            result.pop(-1)
                        break
                    else:
                        for r in self.trace_edges(tracer, dst, end_nid, nodes_limitation, stop_primary_node, result=result, i=i+1):
                            yield tuple(r)
                            r.pop(-1)
            if flag == False and end_nid == -1:
                yield tuple(result)
        result.pop(-1)
    
    def find_path(self, start_nid, end_nid, direction="up", nodes_limitation=None, stop_primary_node=False):
        if direction == "up":
            tracer = self.find_src_nodes_from_edges
        else:
            tracer = self.find_dst_nodes_from_edges
        
        for p in self.trace_edges(tracer, start_nid, end_nid, nodes_limitation, stop_primary_node):
            yield p
    
    def add_edge(self, src, dst):
        if self.config.debug:
            callee_stk = inspect.stack()[1]
            
            #for python2
            if isinstance(callee_stk, tuple):
                frame, filename, lineno, function, source_code, source_index = callee_stk
            # for python 3
            else:
                filename = callee_stk.filename
                lineno = callee_stk.lineno
                function = callee_stk.function
            
            src_ea = ida_idaapi.BADADDR
            src_list_name = "N/A"
            if src in self.node_ids:
                src_ea = self.node_ids[src]
                src_list_name = "nodes"
            elif src in self.exceeded_node_ids:
                src_ea = self.exceeded_node_ids[src]
                src_list_name = "exceeded_nodes"
            dst_ea = ida_idaapi.BADADDR
            dst_list_name = "N/A"
            if dst in self.node_ids:
                dst_ea = self.node_ids[dst]
                dst_list_name = "nodes"
            elif dst in self.exceeded_node_ids:
                dst_ea = self.exceeded_node_ids[dst]
                dst_list_name = "exceeded_nodes"
            if self.config.debug: self.dbg_print("Adding an edge src: %x (%d, %s), dst: %x (%d, %s) from %s:%d" % (src_ea, src, src_list_name, dst_ea, dst, dst_list_name, function, lineno))
        # for a certain packer
        # even if a node is a part of a function and is a callee node and its ea is not at a function head, this script connects this node to destination.
        # but it's not correct. so we skip to insert an edge for the cases except for a correct callee and caller pair.
        if src in self.node_ids and dst in self.node_ids:
            src_ea = self.node_ids[src]
            dst_ea = self.node_ids[dst]
            v = idc.get_operand_value(src_ea, 0)
            # src is a caller
            if src_ea not in self.func_relations:
                f = ida_funcs.get_func(src_ea)
                if f:
                    if f.start_ea != src_ea and src_ea not in self.func_relations and f.start_ea in self.func_relations:
                        add_flag = False
                        for direction in ["parents", "children"]:
                            for caller in self.func_relations[f.start_ea][direction]:
                                callee, _, _, _ = self.func_relations[f.start_ea][direction][caller]
                                if caller == src_ea and callee == dst_ea:
                                    add_flag = True
                                    break
                                
                        if dst_ea == v:
                            add_flag = True
                        if self.config.show_strings_nodes and dst_ea in self.strings_contents:
                            add_flag = True
                        if self.config.show_gvars_nodes and dst_ea in self.gvars_contents:
                            add_flag = True
                        if self.config.show_stroff_nodes and dst_ea in self.stroff_contents:
                            add_flag = True
                            
                        if not add_flag:
                            if self.config.debug: self.dbg_print("Skipping adding an edge src: %x (%d, %s), dst: %x (%d, %s) from %s:%d because this pair does not have proper callee/caller relationships." % (src_ea, src, src_list_name, dst_ea, dst, dst_list_name, function, lineno))
                            return None
                        
                elif src_ea in self.strings_contents:
                    add_flag = True

                elif src_ea in self.gvars_contents:
                    add_flag = True
                elif src_ea in self.stroff_contents:
                    add_flag = True
                    
                else:
                    if self.config.debug: self.dbg_print("Skipping adding an edge src: %x (%d, %s), dst: %x (%d, %s) from %s:%d because this pair does not have proper callee/caller relationships." % (src_ea, src, src_list_name, dst_ea, dst, dst_list_name, function, lineno))
                    return None
                
        self.AddEdge(src, dst)
        if src in self.node_id_relationships:
            self.node_id_relationships[src]["succs"].add(dst)
        else:
            self.node_id_relationships[src] = {"succs":set([dst]), "preds":set([])}
        if dst in self.node_id_relationships:
            self.node_id_relationships[dst]["preds"].add(src)
        else:
            self.node_id_relationships[dst] = {"succs":set([]), "preds":set([src])}
    
    def _add_exceeded_node(self, ea, text, color):
        if ea in self.eps:
            color = self.ep_color
        nid = self.AddNode((text, color))
        if self.config.debug:
            callee_stk = inspect.stack()[2]
            
            #for python2
            if isinstance(callee_stk, tuple):
                frame, filename, lineno, function, source_code, source_index = callee_stk
            # for python 3
            else:
                filename = callee_stk.filename
                lineno = callee_stk.lineno
                function = callee_stk.function
            
            self.dbg_print("inserted an exceeded node next to ea:%x, nid:%d from %s:%d" % (ea, nid, function, lineno))
        self.exceeded_nodes[ea] = nid
        self.exceeded_node_ids[nid] = ea
        return nid

    def update_related_nodes(self, ea, start_ea):
        if start_ea != ida_idaapi.BADADDR:
            # if a node is in addtional_trace_points, we do not add related node list. we will add it later.
            if start_ea == ea or ea not in self.additional_trace_points:
                for tmp_ea in self.related_nodes:
                    if tmp_ea != start_ea and ea in self.related_nodes[tmp_ea]:
                        self.related_nodes[tmp_ea].remove(ea)
                if start_ea in self.related_nodes:
                    self.related_nodes[start_ea].add(ea)
                else:
                    self.related_nodes[start_ea] = set([ea])

    def update_node_type(self, nid, node_type):
        if nid in self.node_ids and self.start_ea == self.node_ids[nid]:
            node_type = "Primary Node"
        if  nid in self.node_ids and self.node_ids[nid] in self.eps:
            node_type = "Entry Point"
        if nid in self.node_ids and self.node_ids[nid] in self.func_relations:
            func_type = self.func_relations[self.node_ids[nid]]['func_type']
            if func_type == FT_LIB:
                node_type += " (LIB)"
            elif func_type == FT_API:
                node_type += " (API)"
            elif func_type == FT_MEM:
                node_type += " (Indirect Call)"
        self.node_types[nid] = node_type
    
    def _add_node(self, ea, text, color, start_ea=ida_idaapi.BADADDR):
        if ea in self.eps:
            color = self.ep_color
        nid = self.AddNode((text, color))
        if self.config.debug:
            callee_stk = inspect.stack()[2]
            
            #for python2
            if isinstance(callee_stk, tuple):
                frame, filename, lineno, function, source_code, source_index = callee_stk
            # for python 3
            else:
                filename = callee_stk.filename
                lineno = callee_stk.lineno
                function = callee_stk.function
            
            self.dbg_print("inserted a node. ea: %x, nid:%d from %s:%d" % (ea, nid, function, lineno))
        self.nodes[ea] = nid
        self.node_ids[nid] = ea
        self.update_related_nodes(ea, start_ea)
        return nid
        
    def add_node(self, ea, text, color, start_ea=ida_idaapi.BADADDR, node_type="Unknown"):
        if text == self.exceeded_node_symbol:
            nid = self._add_exceeded_node(ea, text, color)
            if node_type == "Unknown":
                node_type = "Exceeded Node"
        else:
            nid = self._add_node(ea, text, color, start_ea=start_ea)
        self.update_node_type(nid, node_type)
        return nid

    def replace_node(self, nid, text, color, start_ea=ida_idaapi.BADADDR, dst_ea=ida_idaapi.BADADDR, node_type="Unknown"):
        if self.config.debug:
            callee_stk = inspect.stack()[1]
            
            #for python2
            if isinstance(callee_stk, tuple):
                frame, filename, lineno, function, source_code, source_index = callee_stk
            # for python 3
            else:
                filename = callee_stk.filename
                lineno = callee_stk.lineno
                function = callee_stk.function
            
            self.dbg_print("the node id (%d) is in exceeded nodes. replace the exceeded node with %x from %s:%d" % (nid, self.exceeded_node_ids[nid], function, lineno))
        self._nodes[nid] = (text, color)
        ea = self.exceeded_node_ids.pop(nid)
        self.exceeded_nodes.pop(ea)
        if nid in self.exceeded_node_ids and dst_ea == ida_idaapi.BADADDR and self.exceeded_node_ids[nid] != dst_ea:
            pass
        else:
            ea = dst_ea
        self.nodes[ea] = nid
        self.node_ids[nid] = ea
        self.update_related_nodes(ea, start_ea)
        self.update_node_type(nid, node_type)
        return ea
    
    def get_node_id(self, ea, start_ea):
        nid = self.nodes[ea]
        self.update_related_nodes(ea, start_ea)
        return nid
    
    def insert_string_node(self, ea):
        if self.config.debug: self.dbg_print("Entering insert_string_node function for %x" % ea)
        # adding strings nodes
        if self.config.show_strings_nodes and ea in self.nodes:
            if ea in self.func_relations and ea not in self.filtered_nodes:
                for ref_str_ea in self.func_relations[ea]["strings"]:
                    if not self.config.skip_caller:
                        if ref_str_ea in self.nodes:
                            rsid = self.nodes[ref_str_ea]
                        else:
                            line = self.get_space_removed_disasm(ref_str_ea)
                            rsid = self.add_node(ref_str_ea, line, self.strings_color, node_type="String Ref")
                        self.add_edge(self.nodes[ea], rsid)
                    else:
                        rsid = self.nodes[ea]
                    str_ea, _, _, str_contents = self.func_relations[ea]["strings"][ref_str_ea]
                    if str_ea in self.nodes:
                        sid = self.nodes[str_ea]
                        self.add_edge(rsid, sid)
                    else:
                        str_contents = str_contents.replace('\r', '\\r').replace('\n', '\\n')
                        self.strings_contents[str_ea] = str_contents
                        taglen = len(str_contents) - ida_lines.tag_strlen(str_contents)
                        if len(str_contents) - taglen > self.maximum_string_length:
                            str_contents = str_contents[:self.maximum_string_length+taglen] + "..."
                        sid = self.add_node(str_ea, str_contents, self.strings_color, node_type="String")
                        self.add_edge(rsid, sid)
        if self.config.debug: self.dbg_print("Finished insert_string_node function for %x" % ea)
    
    def insert_var_node(self, ea, func_type, node_type, node_color, loc_cache, show_flag, skip_content):
        if self.config.debug: self.dbg_print("Entering insert_var_node function for %x" % ea)
        # adding strings nodes
        if show_flag and ea in self.nodes:
            if ea in self.func_relations and ea not in self.filtered_nodes:
                for ref_str_ea in self.func_relations[ea][func_type]:
                    if not self.config.skip_caller:
                        if ref_str_ea in self.nodes:
                            rsid = self.nodes[ref_str_ea]
                        else:
                            line = self.get_space_removed_disasm(ref_str_ea)
                            rsid = self.add_node(ref_str_ea, line, node_color, node_type=node_type + "Ref")
                        self.add_edge(self.nodes[ea], rsid)
                    else:
                        rsid = self.nodes[ea]
                    str_ea, _, _, str_contents = self.func_relations[ea][func_type][ref_str_ea]
                    if not skip_content:
                        if str_ea in self.nodes:
                            sid = self.nodes[str_ea]
                            self.add_edge(rsid, sid)
                        else:
                            var_name = ida_name.get_name(str_ea)
                            if var_name:
                                str_contents = var_name + " (" + str_contents + ")"
                            loc_cache[str_ea] = str_contents
                            sid = self.add_node(str_ea, str_contents, node_color, node_type=node_type)
                            self.add_edge(rsid, sid)
        if self.config.debug: self.dbg_print("Finished insert_var_node function for %x" % ea)
    
    def insert_comment_node(self, ea):
        if self.config.debug: self.dbg_print("Entering insert_comment_node function for %x" % ea)
        # adding comment nodes
        if self.config.show_comment_nodes and ea in self.nodes and ea not in self.filtered_nodes:
            #cmts = get_func_relation.get_cmts_in_func(ea, self.filter_out_cmt_regex, self.filter_cmt_regex)
            for node_type in self.func_relations[ea]:
                if node_type in ["cmt", "rcmt"]:
                    cmt_type = node_type
                    for cmt_ea in self.func_relations[ea][cmt_type]:
                        if cmt_ea not in self.nodes:
                            line = self.get_space_removed_disasm(cmt_ea, False)
                            taglen = ida_lines.tag_strlen(line)
                            if (len(line) - taglen) > self.maximum_comment_length+3:
                                line = line[:self.maximum_comment_length+taglen] + "..."
                            if cmt_type == 'cmt':
                                node_type = 'Comment'
                            else:
                                node_type = 'Repeatable Comment'
                            nid = self.add_node(cmt_ea, line, self.comments_color, node_type=node_type)
                            self.add_edge(self.nodes[ea], nid)
        if self.config.debug: self.dbg_print("Finished insert_comment_node function for %x" % ea)

    def trace_paths_with_cache(self, start_ea, end_ea, max_recursive=g_max_recursive, direction="parents", use_cache=True):
        cache = self.paths_cache
        if self.parent:
            cache = self.parent.paths_cache
        fn_keys = tuple(self.filtered_nodes.keys())
        
        if use_cache and (start_ea, end_ea, direction, max_recursive, fn_keys, self.skip_api, self.skip_lib) in cache:
            self.dbg_print("hit a cache", hex(start_ea).rstrip("L"), hex(end_ea).rstrip("L"), direction, max_recursive, [hex(x).rstrip("L") for x in fn_keys], self.skip_api, self.skip_lib)
            for p in cache[(start_ea, end_ea, direction, max_recursive, fn_keys, self.skip_api, self.skip_lib)]:
                yield p
        else:
            found_flag = False
            self.dbg_print("trace the call tree", hex(start_ea).rstrip("L"), hex(end_ea).rstrip("L"), direction, max_recursive, [hex(x).rstrip("L") for x in fn_keys], self.skip_api, self.skip_lib)
            for r in get_func_relation.trace_func_calls(self.func_relations, ea=start_ea, target_ea=end_ea, direction=direction, max_recursive=max_recursive, filtered_nodes=self.filtered_nodes, skip_api=self.skip_api, skip_lib=self.skip_lib, debug=self.config.debug, dbg_print_func=self.dbg_print):
                yield r
                found_flag = True
                if (start_ea, end_ea, direction, max_recursive, fn_keys, self.skip_api, self.skip_lib) in cache:
                    cache[(start_ea, end_ea, direction, max_recursive, fn_keys, self.skip_api, self.skip_lib)].add(tuple(r))
                else:
                    cache[(start_ea, end_ea, direction, max_recursive, fn_keys, self.skip_api, self.skip_lib)] = set([tuple(r)])
            if found_flag == False:
                cache[(start_ea, end_ea, direction, max_recursive, fn_keys, self.skip_api, self.skip_lib)] = set([])
    
    def draw_parents_call_tree(self, start_ea, end_ea, max_recursive=g_max_recursive):
        if start_ea not in self.related_nodes:
            self.related_nodes[start_ea] = set([start_ea])
        prev_id = None

        if self.config.debug:
            self.dbg_print("##################### Start processing %x #################" % start_ea)
        # push a starter node
        if start_ea not in self.nodes:
            func_name = self.get_callee_name(start_ea, self.func_relations[start_ea]["func_type"])
            nid = self.add_node(start_ea, func_name, self.start_color, start_ea, node_type="Callee")
        else:
            nid = self.get_node_id(start_ea, start_ea)
        
        if self.config.debug: self.dbg_print("before tracing... start_ea: %x, end_ea: %x, max_recursive: %d" % (start_ea, end_ea, max_recursive))
        for r in self.trace_paths_with_cache(start_ea, end_ea, max_recursive=max_recursive, direction="parents"):
            if self.config.debug:
                self.dbg_print("$$$$$$$$$$$$$$$$ found a path:", [(hex(y).rstrip("L"), hex(x).rstrip("L"), z) for x,y,z in reversed(r)], "to ", hex(start_ea).rstrip("L"))
            # for path_finder popup menu
            if max_recursive < 0 and self.max_nodes < (len(self._nodes) + len(r)):
                ida_kernwin.msg("The number of max_nodes is exceeded (%d < %d). Note that this graph result is incompleted.%s" % (self.max_nodes, len(self._nodes)+len(r), os.linesep))
                ida_kernwin.msg("Change the graph type and dig into it manually.%s" % (os.linesep))
                break
            
            # to skip nodes in a result if a filtered node is included, check a result first.
            idx = -1
            last_hit = -1
            tmp_flag = False
            for idx, (caller, callee, callee_func_type) in reversed(list(enumerate(r))):
                if caller in self.filtered_nodes or callee in self.filtered_nodes or (((self.skip_api and callee_func_type in [FT_API]) or (self.skip_lib and callee_func_type in [FT_LIB])) and callee not in self.additional_trace_points):
                    tmp_flag = True
                    self.dbg_print("idx: %d, callee: %x or caller: %x was hit" % (idx, callee, caller), [hex(x).rstrip("L") for x in self.filtered_nodes])
                    # do not break. get the last filtered item.
                    #break
                    last_hit = idx
            if not tmp_flag:
                last_hit = -1
            
            # for other nodes except for start nodes
            prev_id = None
            next_callee = None
            next_caller = None
            prev_caller = None
            # note that r and its item indice are reversed for parent nodes.
            for i, (caller, callee, callee_func_type) in reversed(list(enumerate(r))):
                # This is right because the order is reversed. Do not doubt.
                if (len(r) - 1) > i:
                    prev_caller = r[i+1][0]
                    if self.config.skip_caller:
                        prev_caller = r[i+1][1]
                else:
                    prev_caller = None
                # This is right because the order is reversed. Do not doubt.
                if i > 0:
                    next_callee = r[i-1][1]
                    next_caller = r[i-1][0]
                # i == 0 for start_ea ( the bottom item of the parents node)
                else:
                    next_callee = start_ea
                    next_caller = None
                    
                # skip nodes until a filtered node appears
                if last_hit >= 0 and last_hit < i:
                    if self.config.debug:
                        self.dbg_print("Skipping inserting process (%d/%d) last_hit: %d, i: %d" % (len(r)-i, len(r), last_hit, i))
                    continue
                elif last_hit >= 0 and last_hit >= i:
                    # if the path is already existent, just stop it.
                    if not self.config.skip_caller and caller in self.filtered_nodes and caller in self.nodes:
                        if self.config.debug:
                            self.dbg_print("Skipping inserting process (%d/%d) last_hit: %d, i: %d" % (len(r)-i, len(r), last_hit, i))
                        continue
                    elif self.config.skip_caller and callee in self.filtered_nodes and callee in self.nodes:
                        if self.config.debug:
                            self.dbg_print("Skipping inserting process (%d/%d) last_hit: %d, i: %d" % (len(r)-i, len(r), last_hit, i))
                        continue

                if self.config.debug:
                    self.dbg_print("processing %d/%d for a set of callee/caller (%x/%x) with reverse order" % (len(r)-i, len(r), callee, caller))
                do_it_flag = False
                ##########################
                #
                # for callee functions
                #
                if self.config.debug:
                    self.dbg_print("++++++++++ processing the callee (%x)" % callee)
                if callee != ida_idaapi.BADADDR and ((not self.config.skip_caller and caller not in self.filtered_nodes) or self.config.skip_caller):
                    if self.config.debug:
                        self.dbg_print("callee is not BADADDR (%x)" % callee)
                    # for existing nodes (general nodes)
                    if callee in self.nodes:
                        if self.config.debug:
                            self.dbg_print("callee (%x) is in nodes" % callee)
                        if not self.config.skip_caller:
                            callee_id = self.get_node_id(callee, start_ea)
                        else:
                            callee_id = self.get_node_id(callee, start_ea)
                            if prev_id is not None and prev_caller != callee:
                                self.add_edge(prev_id, callee_id)
                            else:
                                pass
                        do_it_flag = True
                        
                    #####################
                    # in skip caller mode
                    # for new nodes to be added
                    elif self.config.skip_caller:
                        line = self.get_callee_name(callee, callee_func_type)
                        color = self.get_color(callee_func_type, callee)
                        if self.config.debug:
                            self.dbg_print("skip caller mode")
                            
                        if callee in self.nodes:
                            if self.config.debug:
                                self.dbg_print("callee (%x) is in nodes" % callee)
                            callee_id = self.get_node_id(callee, start_ea)
                        elif next_callee in self.exceeded_nodes:
                            if self.config.debug:
                                self.dbg_print("next_callee (%x) is in exceeded nodes" % next_callee)
                            callee_id = self.exceeded_nodes[next_callee]
                            nea = self.replace_node(callee_id, line, color, start_ea, callee, node_type="Callee")
                        
                        # find a set of callee and caller but they are not connected yet because the max path limitation is exceeded.
                        elif callee in self.func_relations and len(self.func_relations[callee]['children']) >= 1:
                            if self.config.debug:
                                self.dbg_print("callee (%x) is in func_relations" % callee)
                            src = -1
                            for tmp_ea in [self.func_relations[callee]['children'][x][0] for x in self.func_relations[callee]['children']]:
                                if self.config.debug:
                                    self.dbg_print("tmp_ea:", hex(tmp_ea).rstrip("L"), ", next_callee:", hex(next_callee).rstrip("L"), ", callee:", hex(callee).rstrip("L"))
                                if tmp_ea != next_callee:
                                    if self.config.debug:
                                        self.dbg_print("tmp_ea (%x) != next_callee (%x)" % (tmp_ea, next_callee))
                                    if tmp_ea in self.func_relations and callee in [self.func_relations[tmp_ea]['parents'][x][0] for x in self.func_relations[tmp_ea]['parents']]:
                                        if self.config.debug:
                                            self.dbg_print("next_callee (%x) is a child of tmp_ea (%x)" % (next_callee, tmp_ea))
                                        if tmp_ea in self.nodes:
                                            if tmp_ea not in self.filtered_nodes:
                                                tmp_src = self.find_src_node_from_edges(self.nodes[tmp_ea], self.exceeded_node_symbol)
                                                if tmp_src >= 0:
                                                    if len(self.func_relations[tmp_ea]["parents"]) > 1:
                                                        src = -1
                                                        if self.config.debug:
                                                            self.dbg_print("tmp_ea's (%x) source is an exceeded node (%d), but there are two or more nodes. I do not process it." % (tmp_ea, tmp_src))
                                                        break
                                                    else:
                                                        if self.config.debug:
                                                            self.dbg_print("tmp_ea's (%x) source is an exceeded node (%d)" % (tmp_ea, tmp_src))
                                                        src = tmp_src
                                                        break
                                                else:
                                                    self.dbg_print("There are no exceeded nodes of tmp_ea's (%x) (%d)" % (tmp_ea, tmp_src))
                                            else:
                                                if self.config.debug:
                                                    self.dbg_print("tmp_ea (%x) is in filtered nodes list. skip it." % tmp_ea)
                                        elif tmp_ea in self.exceeded_nodes:
                                            if self.config.debug:
                                                self.dbg_print("tmp_ea (%x) is in the exceeded node list." % tmp_ea)
                                            src = self.exceeded_nodes[tmp_ea]
                                            break
                                        else:
                                            if self.config.debug:
                                                self.dbg_print("tmp_ea (%x) is not displayed on this graph yet." % tmp_ea)
                            if src >= 0:
                                if self.config.debug:
                                    self.dbg_print("found the parent exceeded_node (%x, %d) of tmp_ea (%x)" % (self.exceeded_node_ids[src], src, tmp_ea))
                                callee_id = src
                                if tmp_ea in self.exceeded_nodes:
                                    if self.config.debug:
                                        self.dbg_print("tmp_ea (%x) is in exceeded_nodes" % tmp_ea)
                                    src_ea = self.replace_node(callee_id, line, color, start_ea, callee, node_type="Callee")
                                else:
                                    if self.config.debug:
                                        self.dbg_print("tmp_ea (%x) is not in exceeded_nodes" % tmp_ea)
                                    pass
                            else:
                                if self.config.debug:
                                    self.dbg_print("Adding a node for callee (%x)" % callee)
                                callee_id = self.add_node(callee, line, color, start_ea, node_type="Callee")
                        # create a new node for a new function
                        else:
                            if self.config.debug:
                                self.dbg_print("callee (%x) is not displayed yet. Adding the node." % callee)
                            callee_id = self.add_node(callee, line, color, start_ea, node_type="Callee")
                        
                    #####################
                    #
                    # callee-caller mode
                    #
                    else:
                        if self.config.debug:
                            self.dbg_print("Adding a callee node (%x)." % callee)
                        func_name = self.get_callee_name(callee, callee_func_type)
                        color = self.get_color(callee_func_type, callee)
                        callee_id = self.add_node(callee, func_name, color, start_ea, node_type="Callee")

                    if prev_id is not None:
                        self.add_edge(prev_id, callee_id)
                        if self.config.debug:
                            self.dbg_print("Adding an edge for %d and %d" % (prev_id, callee_id))
                    prev_id = callee_id
                    do_it_flag = True
                
                # caller is BADADDR means this path is exceeded of the path length limitation.
                elif self.config.skip_caller:
                    if self.config.debug:
                        self.dbg_print("caller is BADADDR and skip_caller is ON.")
                    if caller == ida_idaapi.BADADDR:
                        insert_flag = False
                        if self.config.debug:
                            self.dbg_print("next_callee (%x)" % next_callee)
                        if next_callee in self.nodes:
                            if self.config.debug:
                                self.dbg_print("next_callee (%x) is in nodes " % next_callee)
                            next_callee_id = self.get_node_id(next_callee, start_ea)
                            src = self.find_src_node_from_edges(next_callee_id)
                            if src < 0:
                                if self.config.debug:
                                    self.dbg_print("not found any source nodes of next_callee (%x)" % (next_callee))
                                insert_flag = True
                            else:
                                src_node_ea = ida_idaapi.BADADDR
                                if src in self.node_ids:
                                    src_node_ea = src
                                if self.config.debug:
                                    self.dbg_print("found a source node (%x, %d) of next_callee (%x)" % (src_node_ea, src, next_callee))
                                callee_id = src
                        else:
                            if self.config.debug:
                                self.dbg_print("there are no source nodes of next_callee (%x) in nodes" % next_callee)
                            insert_flag = True

                        # find a set of callee and caller but they are not connected yet because the max path limitation is exceeded.
                        if insert_flag and next_callee in self.func_relations and len(self.func_relations[next_callee]['parents']) == 1:
                            if self.config.debug:
                                self.dbg_print("next_callee (%x) is in func_relations" % next_callee)
                            src = -1
                            for tmp_ea in [self.func_relations[next_callee]['parents'][x][0] for x in self.func_relations[next_callee]['parents']]:
                                if self.config.debug:
                                    self.dbg_print("tmp_ea:", hex(tmp_ea).rstrip("L"), ", next_callee:", hex(next_callee).rstrip("L"), ", callee:", hex(callee).rstrip("L"))
                                if tmp_ea in self.nodes:
                                    dst_eas = [self.node_ids[x] for x in self.find_dst_nodes_from_edges(self.nodes[tmp_ea]) if x in self.node_ids]
                                    if next_callee not in dst_eas:
                                        if self.config.debug:
                                            self.dbg_print("next_callee (%x) is not in the destinations of tmp_ea (%x)" % (next_callee, tmp_ea))
                                        if tmp_ea in self.func_relations and next_callee in [self.func_relations[tmp_ea]['children'][x][0] for x in self.func_relations[tmp_ea]['children']]:
                                            if self.config.debug:
                                                self.dbg_print("next_callee (%x) is a child of the tmp_ea (%x)" % (next_callee, tmp_ea))
                                            src = self.nodes[tmp_ea]
                                            if next_callee in self.nodes:
                                                if self.config.debug:
                                                    self.dbg_print("next_callee (%x) is already displayed." % next_callee)
                                                callee_id = self.nodes[next_callee]
                                                self.add_edge(src, callee_id)
                                            elif next_callee in self.additional_trace_points:
                                                insert_flag = False
                                                src = -1
                                                if self.config.debug:
                                                    self.dbg_print("next_callee (%x) is one of the additonal trace points" % next_callee)
                                                pass
                                            else:
                                                src = -1
                                                if self.config.debug:
                                                    self.dbg_print("next_callee (%x) is not displayed yet." % next_callee)
                                            break
                                else:
                                    if tmp_ea in self.func_relations and next_callee in [self.func_relations[tmp_ea]['children'][x][0] for x in self.func_relations[tmp_ea]['children']]:
                                        if self.config.debug:
                                            self.dbg_print("next_callee (%x) is a child of tmp_ea (%x)" % (next_callee, tmp_ea))
                                        if next_callee in self.nodes and tmp_ea in self.exceeded_nodes:
                                            src = self.exceeded_nodes[tmp_ea]
                                            if self.config.debug:
                                                self.dbg_print("next_callee (%x) is already displayed." % next_callee)
                                            self.add_edge(src, callee_id)
                                        elif next_callee in self.additional_trace_points:
                                            insert_flag = False
                                            src = -1
                                            if self.config.debug:
                                                self.dbg_print("next_callee (%x) is one of the additonal trace points" % next_callee)
                                        else:
                                            src = -1
                                            if self.config.debug:
                                                self.dbg_print("next_callee (%x) is not displayed yet" % next_callee)
                                        break
                            if src >= 0:
                                callee_id = self.get_node_id(src, start_ea)
                                insert_flag = False
                                if self.config.debug:
                                    self.dbg_print("src (%x) is already displayed" % self.node_ids[src])
                            elif not insert_flag:
                                callee_id = None
                                pass
                            else:
                                insert_flag = True
                        
                        if insert_flag:
                            callee_id = self.add_node(next_callee, self.exceeded_node_symbol, self.default_color, node_type="Exceeded Node")
                    if self.config.debug:
                        self.dbg_print("updating prev_id, was: ", prev_id, "now: ", callee_id)
                    prev_id = callee_id
                    do_it_flag = True
                # callee is BADADDR or caller is in self.filtered_nodes
                else:
                    if self.config.debug:
                        self.dbg_print("callee is BADADDR (means the path is exceeded of limitation) or caller is in filtered_list")
                    if callee in self.nodes:
                        if self.config.debug: self.dbg_print("updating prev_id, was: ", prev_id, "now: ", self.nodes[callee])
                        prev_id = self.nodes[callee]
                    do_it_flag = True

                if do_it_flag:
                    # insert an exceeded node behind the callee if the callee node is in the filter list.
                    if self.config.debug:
                        self.dbg_print("enter checking if callee (%x) is filtered nodes or not" % callee)
                        self.dbg_print("last_hit: %d, i: %d, callee: %x, filtered_nodes: %s" % (last_hit, i, callee, str([hex(x).rstrip("L") for x in self.filtered_nodes])))
                    if last_hit >= i and callee in self.filtered_nodes:
                        if self.config.debug:
                            self.dbg_print("callee (%x) is in filtered_nodes" % callee)
                            self.dbg_print("exceeded_nodes:", [hex(x).rstrip("L") for x in self.exceeded_nodes])
                        callee_id = -1
                        if callee in self.nodes:
                            callee_id = self.get_node_id(callee, start_ea)
                        if callee_id >= 0:
                            src = self.find_src_node_from_edges(callee_id, self.exceeded_node_symbol)
                            if src < 0:
                                if self.config.debug:
                                    self.dbg_print("callee (%x) does not have an exceeded node. Inserting it." % callee)
                                nid = self.add_node(callee, self.exceeded_node_symbol, self.default_color, node_type="Exceeded Node")
                                self.add_edge(nid, callee_id)
                    # skip underneath nodes of library functions for simple layout
                    elif (self.skip_api and callee_func_type in [FT_API]) or (self.skip_lib and callee_func_type in [FT_LIB]):
                        callee_id = -1
                        if callee in self.nodes:
                            callee_id = self.get_node_id(callee, start_ea)
                        elif callee in self.exceeded_nodes:
                            callee_id = self.exceeded_nodes[callee]
                        src = -1
                        if callee_id >= 0:
                            src = self.find_src_node_from_edges(callee_id)
                            # there are no nodes under this callee at the moment.
                        if src < 0 and callee not in self.additional_trace_points:
                            if callee_id >= 0 and prev_caller is not None:
                                # if there are no nodes under the node yet and there
                                # is a next node, insert an exceeded node.
                                callee_id = self.add_node(callee, self.exceeded_node_symbol, self.default_color, node_type="Exceeded Node")
                                self.add_edge(callee_id, prev_id)
                            else:
                                # no more nodes on the result. Nothing to do. Continue processing...
                                pass
                        # there is a node at least.
                        else:
                            if self._nodes[src][0] == self.exceeded_node_symbol:
                                pass
                            else:
                                # there is a node. In this case, this might be a situation,
                                # that a user expanded an exceeded node.
                                # we do not need to do anything. Contineue processing...
                                pass
                
                #################################
                #
                # for caller functions
                #
                if self.config.debug:
                    self.dbg_print("------------- processing the caller (%x)" % caller)
                if not self.config.skip_caller:
                    # upper or lower function is exceeded of maximum limitation.
                    if caller == ida_idaapi.BADADDR:
                        if next_callee in self.exceeded_nodes:
                            caller_id = self.exceeded_nodes[next_callee]
                        else:
                            if self.config.debug:
                                self.dbg_print("next_callee:", hex(next_callee).rstrip("L"), "filtered_nodes (keys):", str([hex(x).rstrip("L") for x in self.filtered_nodes]), "next_caller:", hex(next_caller).rstrip("L"), "filtered_nodes (values):", str([hex(x).rstrip("L") for x in self.filtered_nodes]))
                            if next_callee in self.nodes:
                                next_callee_id = self.nodes[next_callee]
                                src = self.find_src_node_from_edges(next_callee_id)
                                if src < 0 and next_callee not in self.additional_trace_points:
                                    caller_id = self.add_node(next_callee, self.exceeded_node_symbol, self.default_color, node_type="Exceeded Node")
                                else:
                                    caller_id = self.get_node_id(self.node_ids[src], start_ea)
                            elif next_callee in self.filtered_nodes or next_caller in self.filtered_nodes:
                                caller_id = None
                            else:
                                caller_id = self.add_node(next_callee, self.exceeded_node_symbol, self.default_color, node_type="Exceeded Node")
                    
                    # for existing nodes (general nodes)
                    elif caller in self.nodes:
                        if prev_id is not None and prev_caller != caller:
                            # avoid thunk 
                            if callee == caller and callee in self.func_relations and callee not in self.func_relations[callee]["parents"] and callee in self.func_relations[callee]["children"] and callee != self.func_relations[callee]["children"][callee][0]:
                                caller_id = self.get_node_id(caller, start_ea)
                                pass
                            else:
                                caller_id = self.get_node_id(caller, start_ea)
                                self.add_edge(prev_id, caller_id)
                        else:
                            pass
                    
                    # for new nodes to be added
                    else:
                        line = self.get_space_removed_disasm(caller)
                        color = self.get_color(callee_func_type, caller)
                        if next_callee in self.nodes:
                            next_callee_id = self.nodes[next_callee]
                            src = self.find_src_node_from_edges(next_callee_id, self.exceeded_node_symbol)
                            if src >= 0:
                                src_ea = self.exceeded_node_ids[src]
                                # src_ea is callee. continue the replacing process.
                                if src_ea in self.func_relations:
                                    pass
                                # src_ea is caller. caller's parent is not caller. we do not replace.
                                else:
                                    src = -1
                            if self.config.debug:
                                self.dbg_print("caller: %x, next_callee: %x, next_callee_id: %d, src:%d" % (caller, next_callee, next_callee_id, src,))
                        else:
                            src = -1
                        
                        # if a callee has a "..." node, replace it with actual a function pointer.
                        if src >= 0:
                            caller_id = src
                            src_ea = self.replace_node(src, line, color, start_ea, caller, node_type="Caller")
                            if self.config.debug:
                                self.dbg_print("replace an exceeded_node with the caller (%x %d)" % (src_ea, caller_id))
                        # create a new node for a new function
                        else:
                            caller_id = self.add_node(caller, line, color, start_ea, node_type="Caller")
                        
                        if prev_id is not None and caller not in self.filtered_nodes:
                            self.add_edge(prev_id, caller_id)

                    if self.config.debug:
                        self.dbg_print("updating prev_id. (old) prev_id:", prev_id, "(new) caller_id:", caller_id)
                    prev_id = caller_id
                    
                    # insert an exceeded node if the caller node is in the filter list.
                    if self.config.debug:
                        pid = prev_id
                        if prev_id is None:
                            pid = -1
                        if self.config.debug:
                            self.dbg_print("last_hit: %d, i: %d, caller: %x, filtered_nodes(keys): %s, filtered_nodes (values): %s, prev_id:%d" % (last_hit, i, caller, str([hex(x).rstrip("L") for x in self.filtered_nodes]), str([hex(self.filtered_nodes[x][0]).rstrip("L") for x in self.filtered_nodes]), pid))
                    if last_hit >= i and caller in self.filtered_nodes and prev_id is not None:
                        src = self.find_src_node_from_edges(prev_id, self.exceeded_node_symbol)
                        if src < 0:
                            nid = self.add_node(caller, self.exceeded_node_symbol, self.default_color, node_type="Exceeded Node")
                            self.add_edge(nid, prev_id)
                    
                if callee != ida_idaapi.BADADDR:
                    # adding strings nodes
                    self.insert_string_node(callee)
                    
                    # adding global/static variable nodes
                    self.insert_var_node(callee, func_type="gvars", node_type="Global/Static Vars Ref", node_color=self.gvars_color, loc_cache=self.gvars_contents, show_flag=self.config.show_gvars_nodes, skip_content=False)
                    
                    # adding structure member access nodes
                    self.insert_var_node(callee, func_type="struct_offsets", node_type="Struct Members", node_color=self.stroff_color, loc_cache=self.stroff_contents, show_flag=self.config.show_stroff_nodes, skip_content=True)
                    
                    # adding comments nodes
                    self.insert_comment_node(callee)
            
                    # get children under the callee function.
                    if self.config.display_children_in_parent_funcs:
                        if self.config.debug: self.dbg_print("#################### Start diging into children in  %x #########################" % callee)
                        self.draw_children_call_tree(callee, self.end_ea, max_recursive=1)
                        if self.config.debug: self.dbg_print("#################### Finished diging into children in  %x #########################" % callee)
                    
            # for start node
            if start_ea in self.nodes:
                nid = self.get_node_id(start_ea, start_ea)
                if nid == self.nodes[self.start_ea] and self._nodes[nid][1] != self.start_color:
                    self._nodes[nid] = (self._nodes[nid][0], self.start_color)
            else:
                func_name = self.get_callee_name(start_ea, self.func_relations[start_ea]["func_type"])
                nid = self.add_node(start_ea, func_name, self.start_color, start_ea, node_type="Callee")
            
            if prev_id is not None:
                self.add_edge(prev_id, nid)
        
        # adding strings nodes
        self.insert_string_node(start_ea)
        
        # adding global/static variable nodes
        self.insert_var_node(start_ea, func_type="gvars", node_type="Global/Static Vars Ref", node_color=self.gvars_color, loc_cache=self.gvars_contents, show_flag=self.config.show_gvars_nodes, skip_content=False)
        
        # adding structure member access nodes
        self.insert_var_node(start_ea, func_type="struct_offsets", node_type="Struct Members", node_color=self.stroff_color, loc_cache=self.stroff_contents, show_flag=self.config.show_stroff_nodes, skip_content=True)
        
        # adding comments nodes
        self.insert_comment_node(start_ea)
        
        if self.config.debug: self.dbg_print("#################### Finished processing for parents at %x #########################" % start_ea)
    
    def draw_children_call_tree(self, start_ea, end_ea, max_recursive=g_max_recursive):
        if start_ea not in self.related_nodes:
            self.related_nodes[start_ea] = set([start_ea])
        # push a starter node
        if start_ea not in self.nodes:
            func_name = self.get_callee_name(start_ea, self.func_relations[start_ea]["func_type"])
            nid = self.add_node(start_ea, func_name, self.start_color, start_ea, node_type="Callee")
            if nid == self.nodes[self.start_ea] and self._nodes[nid][1] != self.start_color:
                self._nodes[nid] = (self._nodes[nid][0], self.start_color)
        else:
            nid = self.get_node_id(start_ea, start_ea)
        
        # adding strings nodes
        self.insert_string_node(start_ea)
        
        # adding global/static variable nodes
        self.insert_var_node(start_ea, func_type="gvars", node_type="Global/Static Vars Ref", node_color=self.gvars_color, loc_cache=self.gvars_contents, show_flag=self.config.show_gvars_nodes, skip_content=False)
        
        # adding structure member access nodes
        self.insert_var_node(start_ea, func_type="struct_offsets", node_type="Struct Members", node_color=self.stroff_color, loc_cache=self.stroff_contents, show_flag=self.config.show_stroff_nodes, skip_content=True)
        
        # adding comments nodes
        self.insert_comment_node(start_ea)
        
        if self.config.debug: self.dbg_print("before tracing... start_ea: %x, end_ea: %x, max_recursive: %d" % (start_ea, end_ea, max_recursive))
        for r in self.trace_paths_with_cache(start_ea, end_ea, max_recursive=max_recursive, direction="children"):
            if self.config.debug:
                self.dbg_print("########### found a path", [(hex(x).rstrip("L"), hex(y).rstrip("L"), z) for x,y,z in r])
            
            # for path_finder popup menu
            if max_recursive < 0 and self.max_nodes < (len(self._nodes) + len(r)):
                ida_kernwin.msg("The number of max_nodes is exceeded (%d < %d). Note that this graph result is incompleted.%s" % (self.max_nodes, len(self._nodes)+len(r), os.linesep))
                ida_kernwin.msg("Change the graph type and dig into it manually.%s" % (os.linesep))
                break
                
            # to skip nodes in a result if a filtered node is included, check a result first.
            last_hit = -1
            idx = -1
            tmp_flag = False
            for idx, (caller, callee, callee_func_type) in enumerate(r):
                if caller in self.filtered_nodes or callee in self.filtered_nodes:
                    if self.config.debug:
                        self.dbg_print("found a filteed node:", idx, hex(caller).rstrip("L"), hex(callee).rstrip("L"), [hex(x).rstrip("L") for x in self.filtered_nodes])
                    first_hit = idx
                    tmp_flag = True
                    #### need to break. get the first filtered item.
                    break
            if not tmp_flag:
                first_hit = -1
            
            prev_id = self.nodes[start_ea]
            prev_callee = None
            next_caller = None
            for i, (caller, callee, callee_func_type) in enumerate(r):
                if self.config.debug:
                    self.dbg_print("i:", i, "first_hit:", first_hit, "caller:", hex(caller).rstrip("L"), "callee:", hex(callee).rstrip("L"))
                if i > 0:
                    prev_callee = r[i-1][1]
                # i == 0 for start_ea as the top item of the child nodes
                else:
                    prev_callee = start_ea
                if (len(r) - 1) > i:
                    next_caller = r[i+1][0]
                    if self.config.skip_caller:
                        next_caller = r[i+1][1]
                else:
                    next_caller = None
                
                if self.config.debug:
                    self.dbg_print("processing %d/%d for a set of caller/callee (%x/%x)" % (i+1, len(r), caller, callee))
                
                # insert an exceeded node and
                # skip nodes after a filter node matched
                if first_hit >= 0 and first_hit < i:
                    # we will insert an exceeded node after inserting the callee
                    if self.config.debug:
                        self.dbg_print("Skipping inserting process (%d/%d) first_hit: %d, i: %d" % (i+1, len(r), first_hit, i))
                    break
                elif first_hit >= 0 and first_hit >= i:
                    if self.config.debug:
                        self.dbg_print("Skipping inserting process (%d/%d) first_hit: %d, i: %d" % (i+1, len(r), first_hit, i))
                    # if the path is already existent, just stop it.
                    if not self.config.skip_caller and caller in self.filtered_nodes and caller in self.nodes:
                        break
                    elif self.config.skip_caller and callee in self.filtered_nodes and callee in self.nodes:
                        break

                #################################
                #
                # for caller functions
                #
                if self.config.debug:
                    self.dbg_print("------------- processing the caller (%x)" % caller)
                # upper or lower function is exceeded of maximum limitation.
                if not self.config.skip_caller:
                    if caller == ida_idaapi.BADADDR:
                        if prev_callee in self.nodes:
                            prev_callee_id = self.nodes[prev_callee]
                            dst = self.find_dst_node_from_edges(prev_callee_id)
                            if dst < 0 and prev_callee not in self.additional_trace_points:
                                caller_id = self.add_node(prev_callee, self.exceeded_node_symbol, self.default_color, node_type="Exceeded Node")
                                self.add_edge(prev_callee_id, caller_id)
                            else:
                                caller_id = None
                        elif prev_callee in self.exceeded_nodes:
                            caller_id = self.exceeded_nodes[prev_callee]
                            if prev_id is not None:
                                self.add_edge(prev_id, caller_id)
                        elif prev_callee not in self.additional_trace_points:
                            caller_id = self.add_node(prev_callee, self.exceeded_node_symbol, self.default_color, node_type="Exceeded Node")
                            if prev_id is not None:
                                self.add_edge(prev_id, caller_id)
                        else:
                            caller_id = None
                    # for existing nodes (general nodes)
                    elif caller in self.nodes:
                        caller_id = self.nodes[caller]
                        dst = self.find_dst_node_from_edges(caller_id, self.exceeded_node_symbol)
                        if dst >= 0:
                            dst_ea = self.exceeded_node_ids[dst]
                            if self.config.debug: self.dbg_print("dst exceeded_node %d, %x" % (dst, dst_ea))
                            # dst_ea is callee. continue the replacing process.
                            if dst_ea in self.func_relations:
                                pass
                            # dst_ea is caller. caller's child is not caller. we do not replace.
                            else:
                                dst = -1
                        # if a callee has a "..." node, replace it with actual a function pointer.
                        if dst >= 0:
                            line = self.get_space_removed_disasm(caller)
                            color = self.get_color(callee_func_type, caller)
                            # change caller_id to the replaced node
                            caller_id = dst
                            dst_ea = self.replace_node(dst, line, color, start_ea, caller, node_type="Caller")
                        else:
                            self.update_related_nodes(caller, start_ea)
                            if prev_id is not None and prev_callee != caller:
                                self.add_edge(prev_id, caller_id)
                    # for new nodes to be added
                    else:
                        line = self.get_space_removed_disasm(caller)
                        color = self.get_color(callee_func_type, caller)
                        if prev_callee in self.nodes:
                            prev_callee_id = self.nodes[prev_callee]
                            dst = self.find_dst_node_from_edges(prev_callee_id, self.exceeded_node_symbol)
                            if self.config.debug: self.dbg_print("dst exceeded_node %d" % (dst))
                        else:
                            dst = -1
                        if dst < 0 and callee in self.nodes:
                            callee_id = self.nodes[callee]
                            dst = self.find_src_node_from_edges(callee_id, self.exceeded_node_symbol)
                            if self.config.debug: self.dbg_print("dst exceeded_node %d (callee: %x)" % (dst, callee))
                        
                        if dst >= 0:
                            dst_ea = self.exceeded_node_ids[dst]
                            if self.config.debug: self.dbg_print("dst exceeded_node %d (dst=ea: %x)" % (dst, dst_ea))
                            # dst_ea is callee. continue the replacing process.
                            if dst_ea in self.func_relations:
                                if self.config.debug: self.dbg_print("dst is callee %d, %x" % (dst, dst_ea))
                                pass
                            # dst_ea is caller. caller's child is not caller. we do not replace.
                            else:
                                if self.config.debug: self.dbg_print("dst is caller: %d, %x" % (dst, dst_ea))
                                dst = -1

                        add_node_flag = False
                        # if a callee has a "..." node, replace it with actual a function pointer.
                        if dst >= 0:
                            caller_id = dst
                            src_ea = self.replace_node(dst, line, color, start_ea, caller, node_type="Caller")
                            srcs = self.find_src_nodes_from_edges(dst)
                            if prev_id not in srcs:
                                add_node_flag = True
                        # create a new node for a new function
                        else:
                            caller_id = self.add_node(caller, line, color, start_ea, node_type="Caller")
                            add_node_flag = True
                            
                        if prev_id is not None and add_node_flag:
                            self.add_edge(prev_id, caller_id)
                    if self.config.debug:
                        self.dbg_print("updating prev_id, was: ", prev_id, "now: ", caller_id)
                    prev_id = caller_id

                    # insert an exceeded node if the caller node is in the filter list.
                    if first_hit >= i and caller in self.filtered_nodes:
                        nid = self.add_node(caller, self.exceeded_node_symbol, self.default_color, node_type="Exceeded Node")
                        self.add_edge(prev_id, nid)
                        if self.config.debug:
                            self.dbg_print("updating prev_id, was: ", None, "now: ", prev_id)
                        prev_id = None
                        break
                
                ##########################
                #
                # for callee functions
                #
                if self.config.debug:
                    self.dbg_print("++++++++++ processing the callee (%x)" % callee)
                if callee != ida_idaapi.BADADDR:
                    if self.config.debug:
                        self.dbg_print("callee (%x) is not BADADDR" % callee)
                    if callee in self.nodes:
                        if self.config.debug:
                            self.dbg_print("callee (%x) is in nodes" % callee)
                        callee_id = self.get_node_id(callee, start_ea)
                        if not self.config.skip_caller:
                            if self.config.debug:
                                self.dbg_print("skip caller is disabled and callee (%x) is in nodes" % callee)
                        else:
                            if self.config.debug:
                                self.dbg_print("skip caller is enabled and callee (%x) is in nodes" % callee)
                            if prev_id is not None and prev_callee != callee:
                                self.add_edge(prev_id, callee_id)
                            else:
                                pass
                            
                    elif self.config.skip_caller:
                        line = self.get_callee_name(callee, callee_func_type)
                        color = self.get_color(callee_func_type, callee)
                        if self.config.debug:
                            self.dbg_print("skip_caller is enabled. callee (%x)" % callee)
                        if callee in self.nodes:
                            if self.config.debug:
                                self.dbg_print("callee (%x) is in nodes" % callee)
                            callee_id = self.get_node_id(callee, start_ea)
                        elif next_caller in self.exceeded_nodes:
                            if self.config.debug:
                                self.dbg_print("callee (%x) is in exceeded nodes" % callee)
                            callee_id = self.exceeded_nodes[next_caller]
                            n_ea = self.replace_node(callee_id, line, color, start_ea, callee, node_type="Callee")
                        elif next_caller in self.exceeded_nodes and callee not in self.nodes:
                            callee_id = self.exceeded_nodes[next_caller]
                            dst_ea = self.replace_node(callee_id, line, color, start_ea, callee, node_type="Callee")
                            dst = -1
                        # find a set of callee and caller but they are not connected yet because the max path limitation is exceeded.
                        elif callee in self.func_relations and len(self.func_relations[callee]['parents']) >= 1:
                            if self.config.debug:
                                self.dbg_print("callee (%x) is not in nodes and exceeded nodes" % callee)
                            dst = -1
                            for tmp_ea in [self.func_relations[callee]['parents'][x][0] for x in self.func_relations[callee]['parents']]:
                                if self.config.debug:
                                    self.dbg_print("tmp_ea:", hex(tmp_ea).rstrip("L"), ", prev_callee:", hex(prev_callee).rstrip("L"), ", callee:", hex(callee).rstrip("L"))
                                if tmp_ea != prev_callee:
                                    if self.config.debug:
                                        self.dbg_print("tmp_ea (%x) is a parent of callee (%x) and tmp_ea != prev_callee (%x)" % (tmp_ea, callee, prev_callee))
                                    if tmp_ea in self.func_relations and callee in [self.func_relations[tmp_ea]['children'][x][0] for x in self.func_relations[tmp_ea]['children']]:
                                        if self.config.debug:
                                            self.dbg_print("tmp_ea (%x) is a child of callee (%x)" % (tmp_ea, callee))
                                        if tmp_ea in self.nodes:
                                            if tmp_ea not in self.filtered_nodes:
                                                tmp_dst = self.find_dst_node_from_edges(self.nodes[tmp_ea], self.exceeded_node_symbol)
                                                if tmp_dst >= 0:
                                                    if len(self.func_relations[tmp_ea]["children"]) > 1:
                                                        dst = -1
                                                        if self.config.debug:
                                                            self.dbg_print("tmp_ea's (%x) destination is an exceeded node (%d), but there are two or more nodes. I do not process it." % (tmp_ea, tmp_dst))
                                                        break
                                                    else:
                                                        if self.config.debug:
                                                            self.dbg_print("tmp_ea's (%x) destination is an exceeded node (%d)" % (tmp_ea, tmp_dst))
                                                        dst = tmp_dst
                                                        break
                                            else:
                                                if self.config.debug:
                                                    self.dbg_print("tmp_ea (%x) is in filtered nodes list. skip it." % tmp_ea)
                                        elif tmp_ea in self.exceeded_nodes:
                                            if self.config.debug:
                                                self.dbg_print("tmp_ea (%x) is in exceeded nodes" % tmp_ea)
                                            dst = self.exceeded_nodes[tmp_ea]
                                            break
                                        else:
                                            if self.config.debug:
                                                self.dbg_print("tmp_ea (%x) is not displayed yet." % tmp_ea)
                            if dst >= 0:
                                if self.config.debug:
                                    self.dbg_print("found tmp_ea (%x) has a destination" % tmp_ea)
                                callee_id = dst
                                if tmp_ea in self.exceeded_nodes:
                                    dst_ea = self.replace_node(dst, line, color, start_ea, callee, node_type="Callee")
                                else:
                                    if self.config.debug:
                                        self.dbg_print("tmp_ea (%x) is not in exceeded nodes" % tmp_ea)
                                    pass
                            else:
                                if self.config.debug:
                                    self.dbg_print("callee (%x) does not have any destinations. just add it as a node" % callee)
                                callee_id = self.add_node(callee, line, color, start_ea, node_type="Callee")
                        # create a new node for a new function
                        else:
                            if self.config.debug:
                                self.dbg_print("callee (%x) is a new node. add it as a node" % callee)
                            callee_id = self.add_node(callee, line, color, start_ea, node_type="Callee")
                        
                    else:
                        func_name = self.get_callee_name(callee, callee_func_type)
                        color = self.get_color(callee_func_type, callee)
                        callee_id = self.add_node(callee, func_name, color, start_ea, node_type="Callee")
                    if prev_id is not None:
                        self.add_edge(prev_id, callee_id)
                    if self.config.debug:
                        self.dbg_print("updating prev_id, was: ", prev_id, "now: ", callee_id)
                    prev_id = callee_id
                    
                    # insert an exceeded node if the callee node is in the filter list.
                    if first_hit >= i and callee in self.filtered_nodes and next_caller is not None:
                        dst = self.find_dst_node_from_edges(callee_id, self.exceeded_node_symbol)
                        if dst < 0 and callee not in self.additional_trace_points:
                            nid = self.add_node(callee, self.exceeded_node_symbol, self.default_color, node_type="Exceeded Node")
                            self.add_edge(callee_id, nid)
                        break
                    # skip underneath nodes of library functions for simple layout
                    elif (self.skip_api and callee_func_type in [FT_API]) or (self.skip_lib and callee_func_type in [FT_LIB]):
                        dst = self.find_dst_node_from_edges(callee_id)
                        # there are no nodes under this callee at the moment.
                        if dst < 0 and callee not in self.additional_trace_points:
                            if next_caller is not None:
                                # if there are no nodes under the node yet and there
                                # is a next node, insert an exceeded node.
                                caller_id = self.add_node(callee, self.exceeded_node_symbol, self.default_color, node_type="Exceeded Node")
                                self.add_edge(prev_id, caller_id)
                                break
                            else:
                                # no more nodes on the result. Nothing to do. Continue processing...
                                pass
                        # there is a node at least.
                        else:
                            if self._nodes[dst][0] == self.exceeded_node_symbol:
                                break
                            else:
                                # there is a node. In this case, this might be a situation,
                                # that a user expanded an exceeded node.
                                # we do not need to do anything. Contineue processing...
                                pass
                    
                    # there is a node. We do not need to do anything.
                    else:
                        pass
                # for caller skip mode
                elif self.config.skip_caller:
                    if self.config.debug:
                        self.dbg_print("skip caller mode is enabled. this path is for exceeded nodes.")
                    # for exceeded nodes (...)
                    # caller is BADADDR means this path is exceeded of the path length limitation.
                    if caller == ida_idaapi.BADADDR:
                        insert_flag = False
                        if self.config.debug:
                            self.dbg_print("caller is BADADDR. prev_caller (%x)" % prev_callee)
                        if prev_callee in self.nodes:
                            if self.config.debug:
                                self.dbg_print("prev_callee (%x) is in nodes" % prev_callee)
                            prev_callee_id = self.nodes[prev_callee]
                            dst = self.find_dst_node_from_edges(prev_callee_id)
                            if dst < 0:
                                if self.config.debug:
                                    self.dbg_print("prev_callee (%x) does not have any destination nodes." % prev_callee)
                                insert_flag = True
                                if prev_callee in self.additional_trace_points:
                                    insert_flag = False
                            else:
                                if self.config.debug:
                                    self.dbg_print("prev_callee (%x) has a destination node in nodes or exceeded_nodes" % (prev_callee))
                                callee_id = dst
                        else:
                            if self.config.debug:
                                self.dbg_print("prev_callee (%x) has no destinations or has an exceeded node" % prev_callee)
                            insert_flag = True

                        # find a set of callee and caller but they are not connected yet because the max path limitation is exceeded.
                        if insert_flag and prev_callee in self.func_relations and len(self.func_relations[prev_callee]['children']) == 1:
                            if self.config.debug:
                                self.dbg_print("insert_flag is true")
                            dst = -1
                            for tmp_ea in [self.func_relations[prev_callee]['children'][x][0] for x in self.func_relations[prev_callee]['children']]:
                                if self.config.debug:
                                    self.dbg_print("tmp_ea:", hex(tmp_ea).rstrip("L"), ", prev_callee:", hex(prev_callee).rstrip("L"), ", callee:", hex(callee).rstrip("L"))
                                if tmp_ea in self.nodes:
                                    src_eas = [self.node_ids[x] for x in self.find_src_nodes_from_edges(self.nodes[tmp_ea]) if x in self.node_ids]
                                    if prev_callee not in src_eas:
                                        if self.config.debug:
                                            self.dbg_print("tmp_ea (prev_callee's child (%x)) is not a parent of tmp_ea" % (tmp_ea))
                                        if tmp_ea in self.func_relations and prev_callee in [self.func_relations[tmp_ea]['parents'][x][0] for x in self.func_relations[tmp_ea]['parents']]:
                                            if self.config.debug:
                                                self.dbg_print("prev_callee (%x) is in a tmp_ea's (%x) parent" % (prev_callee, tmp_ea))
                                            dst = self.nodes[tmp_ea]
                                            if prev_callee in self.nodes:
                                                if self.config.debug:
                                                    self.dbg_print("prev_callee (%x) is already displayed as a node" % prev_callee)
                                                callee_id = self.nodes[prev_callee]
                                                self.add_edge(callee_id, dst)
                                            elif prev_callee in self.additional_trace_points:
                                                insert_flag = False
                                                dst = -1
                                                if self.config.debug:
                                                    self.dbg_print("prev_callee (%x) is in additional trace points" % prev_callee)
                                                pass
                                            else:
                                                dst = -1
                                                if self.config.debug:
                                                    self.dbg_print("prev_callee (%x) is not displayed yet." % prev_callee)
                                            break
                                else:
                                    if tmp_ea in self.func_relations and prev_callee in [self.func_relations[tmp_ea]['parents'][x][0] for x in self.func_relations[tmp_ea]['parents']]:
                                        if self.config.debug:
                                            self.dbg_print("prev_callee (%x) is a tmp_ea's (%x) child" % (prev_callee, tmp_ea))
                                        if prev_callee in self.nodes and tmp_ea in self.exceeded_nodes:
                                            dst = self.exceeded_nodes[tmp_ea]
                                            if self.config.debug:
                                                self.dbg_print("prev_callee (%x) is already displayed as a node" % prev_callee)
                                            self.add_edge(callee_id, dst)
                                            break
                                        elif prev_callee in self.additional_trace_points:
                                            insert_flag = False
                                            dst = -1
                                            if self.config.debug:
                                                self.dbg_print("prev_callee (%x) is in additional trace points" % prev_callee)
                                            break
                            if dst >= 0:
                                callee_id = dst
                                self.update_related_nodes(self.node_ids[callee_id], start_ea)
                                insert_flag = False
                                if self.config.debug:
                                    self.dbg_print("callee (%x) has a destination" % callee)
                            elif not insert_flag:
                                callee_id = None
                                pass
                            else:
                                insert_flag = True
                        
                        if insert_flag:
                            callee_id = self.add_node(prev_callee, self.exceeded_node_symbol, self.default_color, node_type="Exceeded Node")
                            prev_callee_id = self.nodes[prev_callee]
                            self.add_edge(prev_callee_id, callee_id)
                            
                    # for displaying indirect calls even if it's in skip caller mode while show_indirect_calls flag is enabled.
                    #if self.config.skip_caller and self.config.show_indirect_calls:
                    if self.config.show_indirect_calls:
                        if callee_func_type == FT_MEM and caller != ida_idaapi.BADADDR:
                            line = self.get_space_removed_disasm(caller)
                            color = self.get_color(callee_func_type, caller)
                            nid = self.add_node(caller, line, color, start_ea, node_type="Caller")
                            self.add_edge(prev_id, nid)
                            callee_id = nid

                    # for indirect call but it has a pointer to an API
                    if callee == ida_idaapi.BADADDR:
                        if caller != ida_idaapi.BADADDR:
                            _, func_type, opn, func_name = self.func_relations[prev_callee]['children'][caller]
                            if func_name:
                                found_flag = False
                                for i, (txt, color) in enumerate(self._nodes):
                                    if func_name == ida_lines.tag_remove(txt):
                                        found_flag = True
                                        callee_id = i
                                        break
                                if not found_flag:
                                    callee_id = self.add_node(ida_idaapi.BADADDR, self.color_callee_str(func_name, func_type), self.get_color(func_type), start_ea, node_type="Dynamic Call")
                                    tif = tinfo.get_tinfo_by_name(func_name)
                                    if tif:
                                        tinfo.apply_tinfo_to_ea(tif, caller, opn)
                                prev_callee_id = self.nodes[prev_callee]
                                self.add_edge(prev_callee_id, callee_id)
                    
                    if self.config.debug:
                        self.dbg_print("updating prev_id, was: ", prev_id, "now: ", callee_id)
                    prev_id = callee_id
                
                # caller skip mode is disabled and callee is BADADDR
                elif callee == ida_idaapi.BADADDR:
                    # for indirect call but it has a pointer to an API
                    if caller != ida_idaapi.BADADDR:
                        _, func_type, opn, func_name = self.func_relations[prev_callee]['children'][caller]
                        if func_name:
                            found_flag = False
                            for i, (txt, color) in enumerate(self._nodes):
                                if func_name == ida_lines.tag_remove(txt):
                                    found_flag = True
                                    callee_id = i
                                    break
                            if not found_flag:
                                callee_id = self.add_node(ida_idaapi.BADADDR, self.color_callee_str(func_name, func_type), self.get_color(func_type), start_ea, node_type="Dynamic Call")
                                tif = tinfo.get_tinfo_by_name(func_name)
                                if tif:
                                    tinfo.apply_tinfo_to_ea(tif, caller, opn)
                            caller_id = self.nodes[caller]
                            self.add_edge(caller_id, callee_id)
                
                            if self.config.debug:
                                self.dbg_print("updating prev_id, was: ", prev_id, "now: ", callee_id)
                            prev_id = callee_id
                    
                # insert an exceeded node behind the callee if the callee node is in the filter list.
                if first_hit >= i and callee in self.filtered_nodes:
                    dst = self.find_dst_node_from_edges(callee_id, self.exceeded_node_symbol)
                    if dst < 0 and callee not in self.additional_trace_points:
                        nid = self.add_node(callee, self.exceeded_node_symbol, self.default_color, node_type="Exceeded Node")
                        self.add_edge(callee_id, nid)
                
                if i < self.max_depth - 1:
                    # adding strings nodes
                    self.insert_string_node(callee)
                    
                    # adding global/static variable nodes
                    self.insert_var_node(callee, func_type="gvars", node_type="Global/Static Vars Ref", node_color=self.gvars_color, loc_cache=self.gvars_contents, show_flag=self.config.show_gvars_nodes, skip_content=False)
                    
                    # adding structure member access nodes
                    self.insert_var_node(callee, func_type="struct_offsets", node_type="Struct Members", node_color=self.stroff_color, loc_cache=self.stroff_contents, show_flag=self.config.show_stroff_nodes, skip_content=True)
                    
                    # adding comments nodes
                    self.insert_comment_node(callee)
        
        if self.config.debug: self.dbg_print("#################### Finished processing for children at %x #########################" % start_ea)
    
    def draw_call_tree(self):
        if self.config.debug:
            self.dbg_print("############################## start_ea: %x" % self.start_ea)
        
        # for parents tracing
        t1 = time.time()
        if not self.skip_parents:
            if self.config.debug:
                self.dbg_print("############################## parents: ")
            self.draw_parents_call_tree(self.start_ea, self.end_ea, self.max_depth)
        t2 = time.time()
        if self.config.debug: self.dbg_print("parents time: %d" % (t2-t1))

        # for children tracing
        if not self.skip_children:
            if self.config.debug:
                self.dbg_print("############################## children: ")
            self.draw_children_call_tree(self.start_ea, self.end_ea, self.max_depth)
        t3 = time.time()
        if self.config.debug: self.dbg_print("children time: %d" % (t3-t2))

        # for expanded nodes
        for p in self.additional_trace_points:
            if self.config.debug:
                self.dbg_print("############################## additional tracing point at %x" % p)
            direction = self.additional_trace_points[p]
            if direction == "parents":
                if not self.skip_parents:
                    self.draw_parents_call_tree(p, self.end_ea, max_recursive=self.max_depth)
            elif direction == "children":
                if not self.skip_children:
                    self.draw_children_call_tree(p, self.end_ea, max_recursive=self.max_depth)

        if self.config.debug:
            self.dbg_print("############################## finished to build call tree process.")
    
    def change_widget_icon(self, icon_data=None, bg_change=False, w=None):
        if icon_data is None:
            icon_data = self.icon.icon_data
        if w is None:
            w = self.GetWidget()
        return self.icon.change_widget_icon(w, icon_data, bg_change)
        
    def zoom_graph(self, w=None, zoom=1, x=None, y=None):
        if w is None:
            w = self.GetWidget()
        if w is None:
            return None
        w_gli = ida_moves.graph_location_info_t()
        if ida_graph.viewer_get_gli(w_gli, w, ida_graph.GLICTL_CENTER):
            if x is not None:
                w_gli.orgx = x
            if y is not None:
                w_gli.orgy = y
            w_gli.zoom = zoom
            ida_graph.viewer_set_gli(w, w_gli, ida_graph.GLICTL_CENTER)
            
    def show_graph(self, zoom=1):
        if self.start_ea not in self.func_relations:
            ida_kernwin.msg("Must be in a function" + os.linesep)
            return False
        
        # display IDA View window
        if not ida_kernwin.find_widget(self.ida_view):
            ida_kernwin.msg("View %s not available. Opening.%s" % (self.ida_view, os.linesep))
            ida_kernwin.open_disasm_window(self.ida_view[-1])
        
        # show the call graph
        r = self.show()
        #ida_kernwin.create_empty_widget(self.title, self.act_icon)

        # if show() is failed or the widget instance does not exsit, do not continue.
        if not r:
            ida_kernwin.msg("Failed to display the call tree graph.%s" % (os.linesep))
            self.close()
            return False
        if not self.GetWidget():
            # if self.GetWidget() returns None, do not show the graph.
            ida_kernwin.msg("Failed to display the call tree graph because it failed to get the widget instance.%s" % (os.linesep))
            self.close()
            return False
        
        # use the option not to close by pressing ESC key
        ida_kernwin.display_widget(self.GetWidget(), ida_kernwin.WOPN_NOT_CLOSED_BY_ESC, None)
        
        # zoom the call graph
        self.zoom_graph(self.GetWidget(), zoom=zoom)
        
        # docking to IDA View-A
        if self.parent is not None:
            ida_kernwin.set_dock_pos(self.title, self.parent.title, ida_kernwin.DP_TAB)
        else:
            ida_kernwin.set_dock_pos(self.title, self.ida_view, ida_kernwin.DP_RIGHT)
        
        # use the option not to close by pressing ESC key
        ida_kernwin.display_widget(self.GetWidget(), ida_kernwin.WOPN_NOT_CLOSED_BY_ESC, None)
        
        # center the start address
        nid = self.nodes[self.start_ea]
        self.do_center_node(nid)
        # color all nodes
        self.color_all_nodes()
        
        # try to automatically detect dark mode for the first execution
        self.config.dark_mode = self.is_dark_mode()
        self.change_widget_icon(bg_change=self.config.dark_mode)
        self.refresh()
        
        return r

def exec_cto(cto_data=None, debug=False):
    if debug or ("g_debug" in globals() and g_debug):
        debug = True
    try:
        r = ida_auto.auto_wait()
        if r:
            cto = CallTreeOverviewer(ida_kernwin.get_screen_ea(), cto_data=cto_data , debug=debug)
        else:
            ida_kernwin.msg("IDA is still in automatic analysis and you have canceled the plugin execution. Do it later again if you need.%s" % (os.linesep))
    except Exception as e:
        exc_type, exc_obj, tb = sys.exc_info()
        lineno = tb.tb_lineno
        ida_kernwin.msg("Got a unexpected error (%s: %s) (%d)%s" % (str(type(e)), e, lineno, os.linesep))
        traceback.print_exc()
        
    return cto

def main():
    global g_cto
    g_cto = exec_cto()

if __name__ == '__main__':
    main()
