import ida_kernwin
import ida_nalt
import ida_lines
import ida_funcs
import ida_idaapi
import ida_name
import ida_entry
import ida_bytes
import ida_ua
import idc
import idautils
import ida_auto

from PyQt5 import QtGui, QtCore, QtWidgets
import sip

import os

#import get_func_relation
#import jump
#import config_base
import cto_base
import syncui
import utils
import xor_loop_detector
#ida_idaapi.require("get_func_relation")
#ida_idaapi.require("jump")
#ida_idaapi.require("config_base")
ida_idaapi.require("cto_base")
ida_idaapi.require("syncui")
ida_idaapi.require("utils")
ida_idaapi.require("xor_loop_detector")

g_ida_view = "IDA View-A"

if not hasattr(ida_kernwin, "WOPN_NOT_CLOSED_BY_ESC"):
    setattr(ida_kernwin, "WOPN_NOT_CLOSED_BY_ESC", 0x100) # 7.5 lacks the definition

class MyFilterProxyModel(QtCore.QSortFilterProxyModel):
    itemDataChanged = QtCore.pyqtSignal(QtCore.QModelIndex, str, str, int)
    
    def __init__(self, parent=None):
        super(MyFilterProxyModel, self).__init__(parent)
        self.idx_set = set([])
        
        try:
            # the option is available only 5.10 and above
            self.setRecursiveFiltering(True)
        except AttributeError:
            #self.__index__ = self.___index__
            self.filterAcceptsRow = self._filterAcceptsRow
            
    # for observing renaming events
    def setData(self, index, value, role=QtCore.Qt.EditRole):
        oldvalue = index.data(role)
        result = super(MyFilterProxyModel, self).setData(index, value, role)
        if result and value != oldvalue:
            self.itemDataChanged.emit(index, oldvalue, value, role)
        return result

    """
    # for incremental filtering
    def ___index__(self):
        super(MyFilterProxyModel, self).__index__(self)
    """
        
    # for incremental filtering
    def _filterAcceptsRow(self, row, parent):
        res = super(MyFilterProxyModel, self).filterAcceptsRow(row, parent)
        idx = self.sourceModel().index(row, 0, parent)
        
        # for getting all the child nodes after matching a parent
        if parent in self.idx_set:
            res = True
        if res and idx.isValid():
            self.idx_set.add(idx)
            
        # find child nodes recursively
        if idx.isValid() and self.sourceModel().hasChildren(idx):
            num_items = self.sourceModel().rowCount(idx)
            for i in range(num_items):
                res = res or self.filterAcceptsRow(i, idx)
            
        return res
"""
# use these filters for later version of Qt
https://doc.qt.io/qt-5/qregularexpression.html

setFilterRegularExpression(const QRegularExpression &regularExpression)

QRegularExpression::CaseInsensitiveOption

QRegularExpression re("Qt rocks", QRegularExpression::CaseInsensitiveOption);

"""

"""
class MyStandardItemModel(QtGui.QStandardItemModel):
    itemDataChanged = QtCore.pyqtSignal(QtGui.QStandardItem, int)

    def setData(self, index, value, role=QtCore.Qt.EditRole):
        oldvalue = index.data(role)
        result = super(StandardItemModel, self).setData(index, value, role)
        if result and value != oldvalue:
            self.itemDataChanged.emit(self.itemFromIndex(index), role)
        return result
    def itemChanged(self, item):
        print("item changed")
"""
    
class MyWidget(QtWidgets.QTreeView):
    key_pressed = QtCore.pyqtSignal(QtGui.QKeyEvent)
    current_changed = QtCore.pyqtSignal(QtCore.QModelIndex, QtCore.QModelIndex)
    state_changed = QtCore.pyqtSignal(str)
    after_filtered = QtCore.pyqtSignal(str)
    item_changed = QtCore.pyqtSignal(QtCore.QModelIndex, str, str)
    
    def __init__(self):
        #super(MyWidget, self).__init__(self)
        QtWidgets.QTreeView.__init__(self)
        
        # sort setting
        self.setSelectionBehavior(QtWidgets.QAbstractItemView.SelectRows)
        self.setSortingEnabled(True)
        self.sortByColumn(1, QtCore.Qt.AscendingOrder)

        # generate source model
        self.model = QtGui.QStandardItemModel()
        #self.model = MyStandardItemModel()
        self.model.setHorizontalHeaderLabels(['Name','Address', 'CRefs', 'BBs'])
        #self.tree.setModel(self.model)
        #self.model.itemDataChanged.connect(self.handleItemDataChanged)
        
        # set proxy model for filter
        self.proxy_model = MyFilterProxyModel()
        # check all columns
        self.proxy_model.setFilterKeyColumn(-1)
        """
        self.proxy_model = QtCore.QSortFilterProxyModel()
        try:
            # the option is available only 5.10 and above
            self.proxy_model.setRecursiveFiltering(True)
        except AttributeError:
            self.proxy_model = MyFilterProxyModel()
        """

        # connect tree view with source item model through proxy model
        self.setModel(self.proxy_model)
        self.proxy_model.setSourceModel(self.model)
        self.proxy_model.itemDataChanged.connect(self.handleItemDataChanged)

        # set selection model for synchronizing with ida
        self.sel_model = QtCore.QItemSelectionModel(self.proxy_model)
        self.setSelectionModel(self.sel_model)
        
        # --------------------------------------
        # Create line edit widget for filter
        self.filter = QtWidgets.QLineEdit()
        self.filter.setToolTip('You can filter with this bar by inputting a keyword with regex and case sensitive options.')
        # --------------------------------------
        # Create check boxes
        self.regex_box = QtWidgets.QCheckBox("RegEx")
        self.cs_box = QtWidgets.QCheckBox("CS")
        self.clear_btn = QtWidgets.QPushButton("X")
        self.clear_btn.setContentsMargins(0,0,0,0)
        self.clear_btn.setFixedWidth(25)
        #self.cs_box.setCheckState(QtCore.Qt.Checked)
        
        # Create parent widget
        self.pfilter = QtWidgets.QWidget()
        filter_layout = QtWidgets.QHBoxLayout(self.pfilter)
        filter_layout.setContentsMargins(0,0,0,0)
        filter_layout.addWidget(self.clear_btn)
        filter_layout.addWidget(self.filter)
        filter_layout.addWidget(self.regex_box)
        filter_layout.addWidget(self.cs_box)

        # hook events of itself and filter box
        self.pfilter.installEventFilter(self)
        for c in self.pfilter.children():
            if c.isWidgetType():
                c.installEventFilter(self)
        self.installEventFilter(self)
        
        # set hooks for incremental search, regex and case sensitive check box state chenged
        self.filter.textChanged.connect(self.filterChanged)
        self.regex_box.stateChanged.connect(self.regexBoxChangedAction)
        self.cs_box.stateChanged.connect(self.csBoxChangedAction)
        self.state_changed.connect(self.filterChanged)
        self.clear_btn.pressed.connect(self.clear_text_and_hide_bar)
        
    def clear_text_and_hide_bar(self):
        self.pfilter.hide()
        self.filter.setText("")
        self.setFocus()
        
    def currentChanged(self, current, previous):
        QtWidgets.QTreeView.currentChanged(self, current, previous)
        self.current_changed.emit(current, previous)
        #print(current, previous)
        
    def handleItemDataChanged(self, idx, old_val, new_val, role):
        if role == QtCore.Qt.EditRole:
            #print(idx, old_val, new_val, role)
            if idx is not None and idx.isValid():
                idx = self.proxy_model.mapToSource(idx)
            if idx is not None and idx.isValid():
                # send the single to the parent widget
                self.item_changed.emit(idx, old_val, new_val)
                #item = self.model.itemFromIndex(idx)
                #print(item)
            
    def eventFilter(self, src, event):
        flag = False
        if event.type() == QtCore.QEvent.KeyPress:
            flag = self.on_key_pressed(src, event)
        return flag
        
    def on_key_pressed(self, src, key_event):
        #QtCore.Qt.CTRL | QtCore.Qt.ALT | QtCore.Qt.SHIFT
        #print('key pressed: %i, %i' % (key_event.key(), key_event.modifiers()))
        
        flag = True
        key = key_event.key()
        state = int(key_event.modifiers())
        c = chr(key & 0xff)
        
        # show/hide filter box for all widgets
        if c == 'F' and state == QtCore.Qt.CTRL:
            self.pfilter.setVisible(not self.pfilter.isVisible())
            if self.pfilter.isVisible():
                self.filter.setFocus()
            else:
                self.setFocus()
        # handling arrow keys for the widget
        elif key in [QtCore.Qt.Key_Left, QtCore.Qt.Key_Up, QtCore.Qt.Key_Right, QtCore.Qt.Key_Down]:
            return False
        # ESC for hiding filter box
        elif src != self and key == QtCore.Qt.Key_Escape:
            self.clear_text_and_hide_bar()
        # for the filter and check box widgets, disable to pass the input to the tree,
        # instead pass it to the filter box 
        elif src != self:
            flag = False
        # for the tree widget, call key press event handler for the tree
        else:
            # send the event to the parent plugin form
            self.key_pressed.emit(key_event)
        return flag
        
    def regexBoxChangedAction(self, state):
        self.use_regex = self.regex_box.isChecked()
        text = self.filter.text()
        if text:
            self.state_changed.emit(text)
    
    def csBoxChangedAction(self, state):
        self.use_cs = self.cs_box.isChecked()
        text = self.filter.text()
        if text:
            self.state_changed.emit(text)

    def filterChanged(self, text):
        #if not text:
        #    text = ""
        #self.sel_model.clearSelection()
        self.sel_model.clearCurrentIndex()
        self.proxy_model.idx_set = set([])
        
        cs = QtCore.Qt.CaseInsensitive
        if self.cs_box.isChecked():
            cs = QtCore.Qt.CaseSensitive
            
        regex = QtCore.QRegExp.FixedString
        if self.regex_box.isChecked():
            regex = QtCore.QRegExp.RegExp2
            
        regExp = QtCore.QRegExp(
            text,
            cs,
            regex
        )
        if regExp.isValid():
            self.proxy_model.setFilterRegExp(regExp)
        
        self.after_filtered.emit(text)
        
class cto_func_lister_t(cto_base.cto_base, ida_kernwin.PluginForm):
    imports = {}
    imports_ids = {}
    exports = {}
    exports_ids = {}
    funcs = {}
    func_ids = {}
    callers = {}
    caller_ids = {}
    default_bg = None
    selected_bg = None
    title = "CTO Function Lister"
    
    def __init__(self, cto_data=None, debug=False):

        # wait for auto analysis
        r = ida_auto.auto_wait()
        
        # init super class
        ida_kernwin.PluginForm.__init__(self)
        cto_base.cto_base.__init__(self, cto_data, debug)

        self.ida_view = g_ida_view
        self.selected_bg = self.get_selected_bg(0x99999999)

        # observing "IDA View-A" window
        class my_ui_hooks_t(syncui.my_ui_hooks_t):
            def _log(self, *msg):
                if self.v().config.debug:
                    self.v().dbg_print(">>> MyUiHook: %s" % " ".join([str(x) for x in msg]))
            def refresh(self):
                self.v().refresh(ida_kernwin.get_screen_ea())
                
        class my_view_hooks_t(syncui.my_view_hooks_t):
            def _log(self, *msg):
                if self.v().config.debug:
                    self.v().dbg_print(">>> MyViewHook: %s" % " ".join([str(x) for x in msg]))
                    
            def update_widget_b_ea(self, now_ea, was_ea):
                if now_ea != was_ea:
                    if was_ea != ida_idaapi.BADADDR:
                        self.v().deselect_item_by_ea(was_ea)
                        
                    if now_ea != ida_idaapi.BADADDR:
                        self.v().expand_item_by_ea(now_ea)

        # Note that this is still in the init method of the CallTreeOverviewer class
        # get the UI and View Hooks
        self.my_ui_hooks = my_ui_hooks_t(self)
        self.my_view_hooks = my_view_hooks_t(self)
        
    def exec_ui_action(self, action, w=None):
        if w is None:
            w = self.GetWidget()
        return self._exec_ui_action(action, w)

    def get_focus(self, w=None):
        if w is None:
            w = self.GetWidget()
        if w is None:
            return False
        return ida_kernwin.activate_widget(w, True)
        
    def clear_item_ids(self):
        imports = {}
        imports_ids = {}
        exports = {}
        exports_ids = {}
        funcs = {}
        func_ids = {}
        callers = {}
        caller_ids = {}

    def delete_func_idx_in_cache(self, idx, del_itself=True):
        # remove child indices
        root_item = self.model.itemFromIndex(idx)
        func_ea, _, _ = self.get_ea_by_idx(idx)
        for item in self.iter_items(root_item):
            curr_idx = self.model.indexFromItem(item)
            caller, _, _ = self.get_ea_by_idx(curr_idx)
            keyword = item.parent().text()
            
            self.delete_idx_in_cache(curr_idx, self.callers, self.caller_ids)
            
        # remove the index of the target funciton
        if del_itself:
            idx_addr = idx.sibling(idx.row(), 1)
            self.delete_idx_in_cache(idx, self.funcs, self.func_ids)
            self.delete_idx_in_cache(idx_addr, self.funcs, self.func_ids)

    def delete_idx_in_cache(self, idx, ea_dict, idx_dict):
        if idx in idx_dict:
            ea = idx_dict[idx]
            ea = idx_dict.pop(idx)
            if ea in ea_dict:
                ea_idx = ea_dict[ea]
                if ea_idx == idx:
                    ea_dict.pop(ea)
        
    def delete_row(self, idx):
        self.model.beginRemoveRows(idx.parent(), idx.row(), idx.row())
        self.model.removeRow(idx.row(), idx.parent())
        self.model.endRemoveRows()
        
    def delete_child_rows(self, parent):
        row_count = self.model.rowCount(parent)
        self.model.beginRemoveRows(parent, 0, row_count-1)
        #self.model.clear()
        self.model.removeRows(0, row_count, parent)
        self.model.endRemoveRows()
    
    def clear_tree(self):
        self.clear_item_ids()
        row_count = self.model.rowCount()
        self.model.beginRemoveRows(QtCore.QModelIndex(), 0, row_count-1)
        #self.model.clear()
        self.model.removeRows(0, row_count)
        self.model.endRemoveRows()
        #self.proxy_model.invalidate()
        
    def update_callee_function(self, ea):
        f = ida_funcs.get_func(ea)
        if f:
            func_ea = f.start_ea
        if func_ea in self.func_relations:
            for caller_ea in self.func_relations[func_ea]['children']:
                if caller_ea == ea:
                    callee_ea = self.func_relations[func_ea]["children"][caller_ea][0]
                    if callee_ea != ida_idaapi.BADADDR and callee_ea in self.func_relations:
                        self.update_function(callee_ea)
        
    def update_function(self, ea):
        idx = self.get_idx_by_ea_and_eatype(ea, "func")
        if idx is not None and ea in self.func_relations:
            rownum = idx.row()
            self.clear_function(idx)
            # add new data to the tree again
            item = self.model.itemFromIndex(idx)
            parent = item.parent()
            self.PopulateFuncTree(parent, ea, rownum)
            # change the callee data 
            for caller_ea in self.func_relations[ea]["parents"]:
                #print("%x" % caller_ea)
                callee_ea = self.func_relations[ea]["parents"][caller_ea][0]
                #print("  %x" % callee_ea)
                idx = self.get_idx_by_ea_and_eatype(callee_ea, "func")
                root_item = self.model.itemFromIndex(idx)
                for item in self.iter_items(root_item):
                    item_ea, _, _ = self.get_ea_by_item(item)
                    idx = self.model.indexFromItem(item)
                    #print("item_ea %x" % item_ea)
                    if caller_ea == item_ea and idx.column() == 0:
                        #print("caller_ea:%x" % caller_ea)
                        text = idc.generate_disasm_line(item_ea, 0)
                        item.setText(text)
                        item.setToolTip(text)
        
    def clear_function(self, idx):
        if idx is not None:
            self.delete_func_idx_in_cache(idx)
            self.delete_row(idx)
            #self.delete_child_rows(idx)
            
    def refresh(self, ea=ida_idaapi.BADADDR, center=False):
        orig_ea = ea
        if orig_ea == ida_idaapi.BADADDR:
            orig_ea = ida_kernwin.get_screen_ea()
        f = ida_funcs.get_func(ea)
        if f:
            ea = f.start_ea
        #self.create_tree()
        self.copy_cache_data()
        if ea == ida_idaapi.BADADDR:
            self.clear_tree()
            self.PopulateTree()
        else:
            # ToDo: if ea is string, check string and update string name ...
            self.update_callee_function(orig_ea)
            self.update_function(ea)
        self.jumpto(idc.get_inf_attr(idc.INF_MIN_EA))
        self.jumpto(orig_ea)
        self.expand_item_by_ea(orig_ea)
            
    def force_reload(self):
        self.refresh()
        return True
        
    # utilities
    # ============================
    def get_ea_by_item(self, item):
        ea = ida_idaapi.BADADDR
        name = ""
        mod_name = ""
        _ord = -1
        idx = self.model.indexFromItem(item)
        if idx in self.func_ids:
            ea = self.func_ids[idx]
        elif idx in self.caller_ids:
            ea = self.caller_ids[idx]
        elif idx in self.imports_ids:
            ea, name, _ord, mod_name = self.imports_ids[idx]
        elif idx in self.exports_ids:
            ea, name, _ord, mod_name = self.exports_ids[idx]
        return ea, name, idx
        
    def get_ea_by_idx(self, idx):
        ea = ida_idaapi.BADADDR
        name = ""
        mod_name = ""
        _ord = -1
        if idx in self.func_ids:
            ea = self.func_ids[idx]
        elif idx in self.caller_ids:
            ea = self.caller_ids[idx]
        elif idx in self.imports_ids:
            ea, name, _ord, mod_name = self.imports_ids[idx]
        elif idx in self.exports_ids:
            ea, name, _ord, mod_name = self.exports_ids[idx]
        return ea, name, mod_name
        
    def expand_parents(self, idx):
        if idx is None:
            return False
        idx = idx.parent()
        while idx.isValid():
            if not self.tree.isExpanded(idx):
                self.tree.expand(idx)
            idx = idx.parent()
            
    def get_idx_by_ea_and_eatype(self, ea, ea_type):
        idx = None
        ea_type = ""
        if ea in self.funcs:
            idx = self.funcs[ea]
            ea_type = "func"
        elif ea in self.callers:
            idx = self.callers[ea]
            ea_type = "caller"
        elif ea in self.imports:
            idx = self.imports[ea]
            ea_type = "import"
        elif ea in self.exports:
            idx = self.exports[ea]
            ea_type = "export"
        return idx
        
    def iter_items(self, root):
        def recurse(parent):
            for row in range(parent.rowCount()):
                for column in range(parent.columnCount()):
                    child = parent.child(row, column)
                    yield child
                    if child.hasChildren():
                        for item in recurse(child):
                            yield item
        if root is not None:
            for item in recurse(root):
                yield item
    
    def get_all_idx_by_ea(self, ea):
        idx = None
        for ea_type in ["func", "caller", "import", "export"]:
            idx = self.get_idx_by_ea_and_eatype(ea, ea_type)
            if idx is not None:
                yield idx, ea_type
        
    def _get_idx_by_ea(self, ea):
        idx = None
        ea_type = ""
        if ea in self.funcs:
            idx = self.funcs[ea]
            ea_type = "func"
        elif ea in self.callers:
            idx = self.callers[ea]
            ea_type = "caller"
        elif ea in self.imports:
            idx = self.imports[ea]
            ea_type = "import"
        elif ea in self.exports:
            idx = self.exports[ea]
            ea_type = "export"
        return idx, ea_type
        
    def get_idx_by_ea(self, ea):
        idx = None
        ea_type = ""
        idx, ea_type = self._get_idx_by_ea(ea)
        return idx, ea_type

    def deselect_item_by_ea(self, ea):
        idx, _ = self.get_idx_by_ea(ea)
        if idx is not None and idx.isValid():
            idx = self.proxy_model.mapFromSource(idx)
        if idx is not None and idx.isValid():
            self.sel_model.select(idx, QtCore.QItemSelectionModel.Deselect|QtCore.QItemSelectionModel.Rows)
            item = self.model.itemFromIndex(idx)
            if item is not None:
                item.setBackground(self.default_bg)

    def expand_item_by_ea(self, ea):
        # check if we need to expand or not
        # This is mainly for Exports/EPs.
        same_flag = False
        curr_idx = self.tree.currentIndex()
        if curr_idx is not None and curr_idx.isValid():
            curr_idx = self.proxy_model.mapToSource(curr_idx)
        if curr_idx is not None and curr_idx.isValid():
            curr_ea, _, _ = self.get_ea_by_idx(curr_idx)
            if curr_ea == ea:
                same_flag = True
                #return

        if same_flag:
            idx = curr_idx
        else:
            idx, _ = self.get_idx_by_ea(ea)
            
        if idx is not None and idx.isValid():
            idx = self.proxy_model.mapFromSource(idx)
                
        # Expand the tree.
        if idx is not None and idx.isValid():
            #if not self.tree.isExpanded(idx):
            #    self.tree.expand(idx)
            self.sel_model.select(idx, QtCore.QItemSelectionModel.Select|QtCore.QItemSelectionModel.Rows)
            #self.tree.setCurrentIndex(idx)
            self.tree.scrollTo(idx, QtWidgets.QAbstractItemView.EnsureVisible)
            self.expand_parents(idx)
            item = self.model.itemFromIndex(idx)
            if item is not None:
                item.setBackground(self.selected_bg)

    def jump_to_idx(self, idx):
        ea, name, mod_name = self.get_ea_by_idx(idx)
        if ea != ida_idaapi.BADADDR:
            self.jumpto(ea)
            self.expand_item_by_ea(ea)
            
    def jump_to_item(self, item, column_no=0):
        ea, name, idx = self.get_ea_by_item(item)
        if ea != ida_idaapi.BADADDR:
            self.jumpto(ea)
            self.expand_item_by_ea(ea)
            
    def jump_to_callee(self, idx):
        if not idx.isValid():
            return
        idx = self.proxy_model.mapToSource(idx)
        item = self.model.itemFromIndex(idx)
        if idx in self.caller_ids:
            idx_caller = idx.sibling(idx.row(), 0)
            caller = self.caller_ids[idx_caller]
            #print("%x" % caller)
            
            if not idx_caller.isValid():
                return
            
            idx_key = idx_caller.parent()
            if not idx_key.isValid():
                return
            
            item_key = self.model.itemFromIndex(idx_key)
            keyword = item_key.text()
            func_idx = idx_key.parent()
            
            if func_idx.isValid() and func_idx in self.func_ids:
                func_ea = self.func_ids[func_idx]
                if func_ea in self.func_relations and keyword in self.func_relations[func_ea] and caller in self.func_relations[func_ea][keyword]:
                    callee = self.func_relations[func_ea][keyword][caller][0]
                    self.jumpto(callee)
                    self.expand_item_by_ea(callee)
            
    @staticmethod
    def parse_rgba32(color=0xffffffff):
        r = color & 0xff
        g = (color >> 8) & 0xff
        b = (color >> 16) & 0xff
        a = (color >> 24) & 0xff
        return r, g, b, a
    
    def get_selected_bg(self, color=0xffffffff):
        brush = QtGui.QBrush(QtGui.QColor(*self.parse_rgba32(color)))
        return brush
        
    # actions and events
    # ---------------------------------------------------
    def on_dbl_clk(self, idx):
        if self.config.debug: self.dbg_print("double click", idx)
        self.jump_to_callee(idx)
                            
        #print(item.text(0))
        
    def on_clk(self, item, column_no):
        pass
        #ea, name, idx = self.get_ea_by_item(item)
        #if ea != ida_idaapi.BADADDR:
        #    self.jumpto(ea)
            
        #print(idx.internalId())
        #print(idx.row())
        #print("click", item.text(0), idx, column_no)
        
    def on_curr_item_changed(self, curr, prev):
        #print(curr, prev)
        if prev is not None and prev.isValid():
            prev = self.proxy_model.mapToSource(prev)
            if prev.isValid():
                item = self.model.itemFromIndex(prev)
                if item is not None:
                    item.setBackground(self.default_bg)
                #self.sel_model.clearSelection()
                self.sel_model.select(prev, QtCore.QItemSelectionModel.Deselect|QtCore.QItemSelectionModel.Rows)
        if curr is not None and curr.isValid() and curr != prev:
            curr = self.proxy_model.mapToSource(curr)
            if curr.isValid():
                self.jump_to_idx(curr)
                item = self.model.itemFromIndex(curr)
                if item is not None:
                    item.setBackground(self.selected_bg)
                self.tree.setFocus()
                #self.tree.setCurrentIndex(curr)
                self.sel_model.select(curr, QtCore.QItemSelectionModel.Select|QtCore.QItemSelectionModel.Rows)

    def on_item_changed(self, idx, old_val, new_val):
        ea, _name, _mod_name = self.get_ea_by_idx(idx)
        if ea != ida_idaapi.BADADDR and old_val != new_val:
            #r = ida_name.set_name(ea, new_val, ida_name.SN_NOCHECK|ida_name.SN_NOWARN|ida_name.SN_FORCE)
            r = ida_name.set_name(ea, new_val, ida_name.SN_NOCHECK|ida_name.SN_NOWARN)
            if not r:
                ida_kernwin.msg("Could not rename because of invalid characters%s" % os.linesep)
                # revert back to the old val
                item = self.tree.model.itemFromIndex(idx)
                item.setText(old_val)
            else:
                name = ida_name.get_name(ea)
                if name != new_val:
                    item = self.tree.model.itemFromIndex(idx)
                    if name == old_val:
                        ida_kernwin.msg("Could not rename because of invald name or reserved prefix%s" % os.linesep)
                        # revert balck to the old val
                        item.setText(old_val)
                    else:
                        # change the name to modified name that the specific charasters are replaced
                        item.setText(name)
                        # refresh other cto instanses
                        self.refresh_all(ea)
                else:
                    # refresh other cto instances
                    self.refresh_all(ea)
            
    def buildContextMenu(self, event):
        cmenu = QtWidgets.QMenu(self.tree)
        item = self.tree.itemAt(event)
        idx = self.model.indexFromItem(item)
        #print("right-click", item.text(0), idx)
        """
        index = self.ui.treeWidget.indexAt(event)

        if not index.isValid():
            return

        item = self.ui.treeWidget.itemAt(event)

        """
        
        newAct = cmenu.addAction("New")
        opnAct = cmenu.addAction("Open")
        quitAct = cmenu.addAction("Quit")
        action = cmenu.exec_(self.tree.mapToGlobal(event))
        #print("right-click")
        if action == quitAct:
            self.Close(0)
        elif action == newAct:
            print("new")
        elif action == opnAct:
            print("open")
            
    def get_qwidget(self, w, max_try=100):
        r = None
        # find the widget
        widget = sip.wrapinstance(int(w), QtWidgets.QWidget)
        find_flag = False
        i = 0
        while i < max_try and widget and type(widget) != QtWidgets.QMainWindow:
            if type(widget) == QtWidgets.QWidget:
                find_flag = True
                break
            widget = widget.parent()
            i += 1
        
        if not find_flag:
            return r
        else:
            r = widget
        return r
    
    def on_key_pressed(self, key_event):
        # for state
        SHIFT = QtCore.Qt.SHIFT
        ALT = QtCore.Qt.ALT
        CTRL = QtCore.Qt.CTRL
        ESC_KEY = QtCore.Qt.Key_Escape
        ENTER_KEY = QtCore.Qt.Key_Enter
        RETURN_KEY = QtCore.Qt.Key_Return
        
        #QtCore.Qt.CTRL | QtCore.Qt.ALT | QtCore.Qt.SHIFT
        if self.config.debug: self.dbg_print('key pressed: %x, %x' % (key_event.key(), int(key_event.modifiers())))
        key = key_event.key()
        state = int(key_event.modifiers())
        c = chr(key & 0xff)
        
        w = ida_kernwin.find_widget(self.ida_view)
        # for IDA history (back)
        if key == ESC_KEY:
            self.exec_ui_action("Return", w=w)
            self.expand_item_by_ea(ida_kernwin.get_screen_ea())
            self.tree.setFocus()
        # for IDA history (forward)
        elif key in [ENTER_KEY, RETURN_KEY] and state == CTRL:
            self.exec_ui_action("UndoReturn", w=w)
            self.expand_item_by_ea(ida_kernwin.get_screen_ea())
            self.tree.setFocus()
        # toggle Debug message
        elif c == 'D' and state == 0:
            self.config.debug = not self.config.debug
            ida_kernwin.msg("debugging %sabled%s" % ("en" if self.config.debug else "dis", os.linesep))
        elif c == "R" and state == 0:
            self.refresh()
            ida_kernwin.msg("refreshed" + os.linesep)
            #key_event.setAccepted(True)
            #return 1
        # Force refresh
        elif c == 'F' and state == 0:
            if self.force_reload():
                ida_kernwin.msg("Force reloaded." + os.linesep)
            else:
                ida_kernwin.msg("Not reloaded." + os.linesep)
        # Update func relations
        elif c == 'U' and state == 0:
            ida_kernwin.show_wait_box("Wait for updating the cache")
            #self.cache_update()
            self.update_data()
            ## remove past navigation history of this graph.
            #self.exec_ui_action("EmptyStack")
            self.refresh_all()
            ida_kernwin.msg("the caches of the function relationships and the referred string were Updated." + os.linesep)
            ida_kernwin.hide_wait_box()
        # Update func relations partially
        elif c == 'U' and state == SHIFT:
            ea = ida_kernwin.get_screen_ea()
            self.partial_cache_update(ea)
            #self.exec_ui_action("EmptyStack")
            self.refresh_all(ea)
            ida_kernwin.msg("the caches of the function relationships and the referred string were Updated partially." + os.linesep)
        # Update func relations partially
        elif c == 'U' and state == CTRL:
            self.cache_cmt_update()
            self.refresh_all()
            ida_kernwin.msg("the caches related to comments were Updated." + os.linesep)
        # Help
        elif c == 'H' and state == 0:
            self.print_help()
            """
        # darK mode
        elif c == 'K' and state == 0:
            self.config.dark_mode = not self.config.dark_mode
            #self.change_widget_icon(bg_change=self.config.dark_mode)
            #self.color_settings()
            self.refresh()
            ida_kernwin.msg("darK mode %sabled%s" % ("en" if self.config.dark_mode else "dis", os.linesep))
            """
        # go to an address or an address of a function name
        elif c == 'G' and state == 0:
            self.exec_ida_ui_action("JumpAsk")
        # rename a function
        elif c == 'N' and state == 0:
            flag = self.check_and_rename_var()
        elif c == 'P' and state == ALT:
            self.check_and_rename_func_info()
        # repeatable comment
        elif c == ';':
            self.check_and_add_rcmt()
        # comment
        elif c == ':':
            self.check_and_add_cmt()
        # apply structure
        elif c == 'T' and state == 0:
            self.check_and_apply_strunct()
        # show xrefs to
        elif c == 'X' and state == 0:
            self.check_xrefs()
        # detect xor loops
        elif c == 'X' and state == CTRL:
            for func_ea, ea, annotation_type in xor_loop_detector.find_xor_loop():
                ida_kernwin.msg("%x: %s, %x: %s%s" % (func_ea, annotation_type, ea, idc.generate_disasm_line(ea, 0), os.linesep))
            self.cache_cmt_update()
            self.refresh_all()
        # jump to callee on a caller
        elif key in [ENTER_KEY, RETURN_KEY] and state == 0:
            curr_idx = self.tree.currentIndex()
            self.jump_to_callee(curr_idx)
            
        self.get_focus(w=self.GetWidget())
        self.tree.setFocus()
        self.expand_item_by_ea(ida_kernwin.get_screen_ea())
            
    def print_help(self):
        ida_kernwin.msg("""
[How to use]
- You can think this is an enhanced version of "Functions" view.

[Shortcuts]
H: Help that you are looking.
R: Refresh the call graph manually.
F: Currently, this is the same as the refresh.
U: Update function relationships which is a cache and used for building the call tree.
Shift+U: Update function relationships partially. It updates only the node on the caches
   and its parent function and its parent functions's children if the node is a caller.
Ctrl+U: Update all comment caches. This is useful for collecting some tools'a results such as
   ironstrings and findcrypt.py.
N: reName a function (this option redirects to IDA View-A so that you can use it transparently).
G: Go to a place (this option redirects to IDA View-A so that you can use it transparently).
X: display Xrefs (this option redirects to IDA View-A so that you can use it transparently).
T: apply a sTructure member to an operand (this option redirects to IDA View-A so that
   you can use it transparently).
;: make repeatable comment (this option redirects to IDA View-A so that you can use it
   transparently).
:: make comment (this option redirects to IDA View-A so that you can use it transparently).
Alt+P: edit function (this option redirects to IDA View-A so that you can use it transparently).
Ctrl+X: detect Xor instructions in a loop.
ESC: hide filter bar and clear text.
Ctrl+F: display/hide Filter bar.
D: enable/disable Debug mode
""")
        
    """
    # setFilterRegularExpression does not exist in this PyQt5
    def filterChanged(self, text):
        #if not text:
        #    text = ""
        #self.sel_model.clearSelection()
        self.sel_model.clearCurrentIndex()
        regExp = QtCore.QRegularExpression(
            text,
            QtCore.QRegularExpression.CaseInsensitiveOption,
        )
        if regExp.isValid():
            self.proxy_model.setFilterRegularExpression(regExp)

        # if it empty, expand the current location and move to it
        # we can do it always, but it's a little bit heavy.
        if text == '':
            ea = ida_kernwin.get_screen_ea()
            if ea != ida_idaapi.BADADDR:
                self.expand_item_by_ea(ea)
        self.filter.setFocus()
    """

    def after_filtered(self, text):
        # if it empty, expand the current location and move to it
        # we can do it always, but it's a little bit heavy.
        if text == '':
            ea = ida_kernwin.get_screen_ea()
            if ea != ida_idaapi.BADADDR:
                self.expand_item_by_ea(ea)
        self.tree.filter.setFocus()
        
    def OnCreate(self, form):
        """
        Called when the plugin form is created
        """
        
        # Get parent widget
        self.parent = self.FormToPyQtWidget(form)
        
        self.create_tree()
        
    def create_tree(self):
        # =================================
	
        # Create tree control
        self.tree = MyWidget()
        self.model = self.tree.model
        self.proxy_model = self.tree.proxy_model
        self.sel_model = self.tree.sel_model
        
        # =============================
        # for actions like when clicking, right-clicking or for events when changing selected items
        # focus an item (for sync with IDA View-A by jumping to the corresponding EA)
        self.tree.current_changed.connect(self.on_curr_item_changed)
        
        # for hooking renaming events
        self.tree.item_changed.connect(self.on_item_changed)
        
        # shortcut keys for passing to IDA
        self.tree.key_pressed.connect(self.on_key_pressed)
        
        # processes afer filtering
        self.tree.after_filtered.connect(self.after_filtered)
        
        # click
        #self.tree.clicked.connect(self.on_clk)
        
        # double-click
        self.tree.doubleClicked.connect(self.on_dbl_clk)
        
        # right-click
        #self.tree.setContextMenuPolicy(QtCore.Qt.CustomContextMenu)
        #self.tree.customContextMenuRequested.connect(self.buildContextMenu)
        
        # =========================
        # Create layout
        layout = QtWidgets.QVBoxLayout()
        layout.setContentsMargins(2,0,0,0)
        layout.addWidget(self.tree)
        layout.addWidget(self.tree.pfilter)

        # =========================
        # build the function list
        self.PopulateTree()
        self.tree.reset()
        # Populate PluginForm
        self.parent.setLayout(layout)

        # adjust header length manually
        self.tree.header().setCascadingSectionResizes(True)
        self.tree.header().setSectionResizeMode(0, self.tree.header().Interactive)
        self.tree.header().setSectionResizeMode(1, self.tree.header().Interactive)
        self.tree.header().setStretchLastSection(False)
        self.tree.header().resizeSection(0, 180)
        for i in range(2,4):
            self.tree.resizeColumnToContents(i)
        #self.tree.setAutoScroll(True)
        
        # =========================
        # move to the current screen ea location
        ea = ida_kernwin.get_screen_ea()
        if ea != ida_idaapi.BADADDR:
            self.expand_item_by_ea(ea)

        # focus the tree widget
        self.tree.setFocus()

    def OnClose(self, form):
        """
        Called when the plugin form is closed
        """
        self.close()
        
    def close(self):
        # for closing cached data of cto_base class
        self.close_data()
        if hasattr(self, "sd"):
            self.sd.close()
            
        # unhook UI and View hooks
        if self.config.debug: self.dbg_print("Unhooking ui and view hooks%s" % (os.linesep))
        self.my_ui_hooks.unhook()
        self.my_view_hooks.unhook()
        if self.config.debug: self.dbg_print("Unhooked ui and view hooks%s" % (os.linesep))
        
        # close tempfile for debug log
        if self.f:
            self.f.close()
            self.f = None

    def __del__(self):
        self.close()
        
    def Show(self):
        """Creates the form is not created or focuses it if it was"""
        return ida_kernwin.PluginForm.Show(self,
                               "CTO Function Lister",
                               options = ida_kernwin.PluginForm.WOPN_PERSIST)
    
    #-----------------------------------------------------
    def imports_names_cb(self, ea, name, ordinal):
        self.items.append((ea, str(ordinal) if not name else name, ordinal))
        # True -> Continue enumeration
        return True
        
    def BuildImports(self):
        tree = {}
        nimps = ida_nalt.get_import_module_qty()
        
        for i in range(0, nimps):
            name = ida_nalt.get_import_module_name(i)
            if not name:
                continue
            # Create a list for imported names
            self.items = []

            # Enum imported entries in this module
            ida_nalt.enum_import_names(i, self.imports_names_cb)

            if name not in tree:
                tree[name] = []
            tree[name].extend(self.items)
            
        return tree
        
    def BuildExports(self):
        return list(idautils.Entries())

    def PopulateFuncTree(self, root, func_ea, row=-1):
        #func_name, ifunc, idx = self.RegisterFuncToTree(root, func_ea)
        func_name = self.get_name(func_ea)
        func_name, ifunc, idx = self.RegisterFuncToTree(root, func_ea, func_name, self.funcs, self.func_ids, row=row)
        for keyword in self.func_relations[func_ea]:
            if keyword == "func_type":
                continue

            #func_name = self.get_name(func_ea)
            ikey = QtGui.QStandardItem("%s" % (keyword))
            ikey.setEditable(False)
            ifunc.appendRow(ikey)
        
            for caller in sorted(self.func_relations[func_ea][keyword]):
                disasm, icaller, idx = self.RegisterCallerToTree(ikey, caller, keyword, func_ea)
                
    def RegisterCallerToTree(self, ikey, caller, keyword, func_ea):
        disasm = idc.generate_disasm_line(caller, 0)
        icaller_name = QtGui.QStandardItem("%s" % (disasm))
        icaller_addr = QtGui.QStandardItem("%x" % (caller))
        icaller_name.setToolTip(disasm)
        icaller_name.setEditable(False)
        icaller_addr.setEditable(False)
        ikey.appendRow((icaller_name, icaller_addr))
        idx = self.model.indexFromItem(icaller_name)
        idx_addr = self.model.indexFromItem(icaller_addr)
        self.caller_ids[idx] = caller
        self.caller_ids[idx_addr] = caller
        f = ida_funcs.get_func(caller)
        if keyword != "parents" or (keyword == "parents" and f.start_ea == func_ea):
            #self.caller_ids[idx] = caller
            #self.caller_ids[idx_addr] = caller
            self.callers[caller] = idx
        return disasm, icaller_name, idx

    def get_name(self, func_ea):
        f = ida_funcs.get_func(func_ea)
        if f:
            func_name = ida_funcs.get_func_name(f.start_ea)
        else:
            func_name = ida_name.get_name(func_ea)
        return func_name
    
    def RegisterFuncToTree(self, parent, func_ea, func_name, ea_dict, idx_dict, other_data=None, row=-1):
        ifunc_name = QtGui.QStandardItem("%s" % (func_name))
        ifunc_addr = QtGui.QStandardItem("%x" % (func_ea))
        ifunc_xref_cnt = QtGui.QStandardItem("%d" % (utils.count_xref(func_ea)))
        ifunc_bb_cnt = QtGui.QStandardItem("%d" % (utils.count_bbs(func_ea)))
        ifunc_addr.setEditable(False)
        ifunc_xref_cnt.setEditable(False)
        ifunc_bb_cnt.setEditable(False)
        if row >= 0:
            parent.insertRow(row, (ifunc_name, ifunc_addr, ifunc_xref_cnt, ifunc_bb_cnt))
        else:
            parent.appendRow((ifunc_name, ifunc_addr, ifunc_xref_cnt, ifunc_bb_cnt))
        
        idx = self.model.indexFromItem(ifunc_name)
        idx_addr = self.model.indexFromItem(ifunc_addr)
        
        idx_dict[idx] = func_ea
        if other_data is not None:
            idx_dict[idx] = [func_ea]
            idx_dict[idx].extend(other_data)
            
        idx_dict[idx_addr] = func_ea
        if other_data is not None:
            idx_dict[idx_addr] = [func_ea]
            idx_dict[idx_addr].extend(other_data)
            
        ea_dict[func_ea] = idx
        
        return func_name, ifunc_name, idx
    
    def find_toplevel_child(self, text):
        for i in range(self.tree.topLevelItemCount()):
            item = self.tree.topLevelItem(i)
            if item.text(0) == text:
                return item
        return None
    
    def PopulateTree(self):
        # Build functions
        root = QtGui.QStandardItem("Functions")
        root.setEditable(False)
        self.default_bg = root.background()
        self.model.appendRow(root)
        
        #for func_ea in sorted(self.func_relations):
        for func_ea in idautils.Functions():
            self.PopulateFuncTree(root, func_ea)

        # Build imports
        root = QtGui.QStandardItem("Imports")
        root.setEditable(False)
        self.model.appendRow(root)

        for dll_name, imp_entries in self.BuildImports().items():
            imp_dll = QtGui.QStandardItem(dll_name)
            imp_dll.setEditable(False)
            root.appendRow(imp_dll)
            for imp_ea, imp_name, imp_ord in imp_entries:
                func_name, ifunc, idx = self.RegisterFuncToTree(imp_dll, imp_ea, imp_name, self.imports, self.imports_ids, other_data=(imp_name, imp_ord, dll_name))
                
        # Build exports
        root = QtGui.QStandardItem("Exports")
        root.setEditable(False)
        self.model.appendRow(root)

        for exp_i, exp_ord, exp_ea, exp_name in self.BuildExports():
            name = idc.get_module_name(exp_ea)
            if exp_name is None:
                exp_name = ida_name.get_name(exp_ea)
            func_name, ifunc, idx = self.RegisterFuncToTree(root, exp_ea, exp_name, self.exports, self.exports_ids, other_data=(exp_name, exp_ord, name))

    def show(self):
        # show the list
        r = self.Show()

        if r:
            # use the option not to close by pressing ESC key
            ida_kernwin.display_widget(self.GetWidget(), ida_kernwin.WOPN_NOT_CLOSED_BY_ESC, None)
            
            ida_kernwin.set_dock_pos(self.title, "Functions window", ida_kernwin.DP_TAB)
        return r

# --------------------------------------------------------------------------
def exec_cto_function_lister(cto_data=None, debug=False):
    cto_func_lister = cto_func_lister_t(cto_data=cto_data, debug=debug)

    r = cto_func_lister.show()
    if r:
        return cto_func_lister
    return None

def main():
    global cto_func_lister
    try:
        cto_func_lister.Close(0)
        del cto_func_lister
        cto_func_lister = cto_func_lister_t()
    except:
        cto_func_lister = cto_func_lister_t()
    #cto_func_lister = exec_cto_function_lister()

# --------------------------------------------------------------------------
if __name__ == '__main__':
    main()
