import ida_kernwin
import ida_ua
import ida_idaapi
import ida_funcs
import ida_name
import ida_bytes

import os
import sys
import traceback

g_ida_view = "IDA View-A"

# Temoprary classes for UI and view hooks.
# Their classes are only available in init function to avoid to forget unhooking and deleting.
# Their instances are available while CallTreeOverviewer instance is available because they are generated at the end of init function below.
class my_ui_hooks_t(ida_kernwin.UI_Hooks):
    def __init__(self, g, title=g_ida_view):
        ida_kernwin.UI_Hooks.__init__(self)
        self.hook()
        # let's use weakrefs, so as soon as the last ref to
        # the 'CallTreeOverviewer' instance is dropped, the 'my_ui_hooks_t'
        # instance hooks can be automatically un-hooked, and deleted.
        # (in other words: avoid circular reference.)
        import weakref
        self.v = weakref.ref(g)
        self.title = title
        self.cmdname = "<no command>"
        self.ctx = None
        self.cur_ea = ida_idaapi.BADADDR
        self.cur_func = None
        self.cur_value = ida_idaapi.BADADDR
        self.func_name = ""
        self.value_name = ""
        self.make_name_line = ""
        self.func_cmt = ""
        self.cmt = ""
        self.func_rcmt = ""
        self.rcmt = ""
        self.line = ""
            
    def _log(self, *msg):
        ida_kernwin.msg(">>> MyUiHook: %s%s" % (" ".join([str(x) for x in msg], os.linesep)))

    def is_ea_to_be_processed(self, ea):
        r = False
        if hasattr(self.v(), "nodes"):
            if ea in self.v().nodes:
                r = True
        f = ida_funcs.get_func(ea)
        if ea in self.v().func_relations:
            r = True
        elif f and f.start_ea in self.v().func_relations:
            for ft in self.v().func_relations[f.start_ea]:
                if ft == "func_type":
                    continue
                if ea in self.v().func_relations[f.start_ea][ft]:
                    r = True
                    break
                if ea in [self.v().func_relations[f.start_ea][ft][x][0] for x in self.v().func_relations[f.start_ea][ft]]:
                    r = True
                    break
        elif ea in self.v().import_eas:
            r = True
        elif ea in self.v().string_eas:
            r = True
        return r
    
    def is_ea_to_be_processed_cmt(self, ea):
        r = False
        f = ida_funcs.get_func(ea)
        if f and f.start_ea in self.v().func_relations:
            r = True
        elif ea in self.v().import_eas:
            r = True
        return r
    
    def preprocess_action(self, name):
        if self.v().cto_data["master"] != id(self.v()):
            return 0
        self._log("IDA preprocessing command: %s" % name)
        self.cmdname = name
        ea = ida_kernwin.get_screen_ea()
        f = ida_funcs.get_func(ea)
        if self.cmdname == 'MakeName' and self.is_ea_to_be_processed(ea):
            if f and f.start_ea == ea and self.cur_func is not None:
                if hasattr(self.cur_func, "start_ea") and self.cur_func.start_ea == ea:
                    self.func_name = ida_funcs.get_func_name(ea)
            if self.cur_value != ida_idaapi.BADADDR and self.is_ea_to_be_processed(self.cur_value):
                self.value_name = ida_funcs.get_func_name(self.cur_value)
            if not self.value_name:
                self.value_name = ida_name.get_name(self.cur_value)
            if not self.value_name:
                self.value_name = self.v().get_highlighted_name(ea, w=ida_kernwin.find_widget(self.v().ida_view))
            self.make_name_line = ida_kernwin.get_curline()
        elif self.cmdname == 'MakeRptCmt':
            if self.is_ea_to_be_processed_cmt(ea):
                if f and f.start_ea == ea and self.cur_func is not None:
                    if hasattr(self.cur_func, "start_ea") and self.cur_func.start_ea == ea:
                        self.func_cmt = ida_funcs.get_func_cmt(f, 1)
                self.rcmt = ida_bytes.get_cmt(ea, 1)
            elif self.is_ea_to_be_processed(ea):
                self.rcmt = ida_bytes.get_cmt(ea, 1)
        elif self.cmdname == 'MakeComment':
            if self.is_ea_to_be_processed_cmt(ea):
                if f and f.start_ea == ea and self.cur_func is not None:
                    if hasattr(self.cur_func, "start_ea") and self.cur_func.start_ea == ea:
                        self.func_rcmt = ida_funcs.get_func_cmt(f, 0)
                self.cmt = ida_bytes.get_cmt(ea, 0)
            elif self.is_ea_to_be_processed(ea):
                self.cmt = ida_bytes.get_cmt(ea, 0)
        elif self.cmdname == 'OpStructOffset':
            if f and self.is_ea_to_be_processed(f.start_ea):
                self.line = ida_kernwin.get_curline()
        return 0
            
    def postprocess_action(self):
        if self.v().cto_data["master"] != id(self.v()):
            return 0
        self._log("IDA finished processing command: %s" % self.cmdname)
        ea = ida_kernwin.get_screen_ea()
        f = ida_funcs.get_func(ea)
        refresh_flag = False
        #history_clear = False
        #if self.cmdname == 'MakeName' and self.is_ea_to_be_processed(ea):
        if self.cmdname in ['MakeName', "hx:Rename"] and self.is_ea_to_be_processed(ea):
            if f and f.start_ea == ea and self.cur_func is not None:
                if hasattr(self.cur_func, "start_ea") and self.cur_func.start_ea == ea:
                    func_name = ida_funcs.get_func_name(ea)
                    if self.func_name is not None and func_name != self.func_name:
                        refresh_flag = True
                    self.func_name = None
            value_name = ""
            if self.cur_value != ida_idaapi.BADADDR and self.is_ea_to_be_processed(self.cur_value):
                value_name = ida_funcs.get_func_name(self.cur_value)
            if not value_name:
                value_name = ida_name.get_name(self.cur_value)
            if not value_name:
                value_name = self.v().get_highlighted_name(ea, w=ida_kernwin.find_widget(self.v().ida_view))
            if self.value_name is not None and value_name != self.value_name:
                refresh_flag = True
            self.value_name = None
                
            # for renaming a struct member name
            make_name_line = ida_kernwin.get_curline()
            if self.make_name_line is not None and make_name_line != self.make_name_line:
                callee, func_type, opn, _func_name = self.v().get_callee_info(ea)
                if callee is not None:
                    optype = idc.get_operand_type(ea, i)
                    if optype in [ida_ua.o_displ, ida_ua.o_phrase]:
                        #self.v().update_caller_tif(ea)
                        self.update_tif(ea)
                        refresh_flag = True
                        #history_clear = True
            self.make_name_line = None
        elif self.cmdname == 'MakeRptCmt':
            if self.is_ea_to_be_processed_cmt(ea):
                if f and f.start_ea == ea and self.cur_func is not None:
                    if hasattr(self.cur_func, "start_ea") and self.cur_func.start_ea == ea:
                        func_rcmt = ida_funcs.get_func_cmt(f, 1)
                        if self.func_rcmt is not None and func_rcmt != self.func_rcmt:
                            self.v().partial_cache_update(f.start_ea)
                            refresh_flag = True
                            #history_clear = True
                        self.func_rcmt = None
                rcmt = ida_bytes.get_cmt(ea, 1)
                if self.rcmt is not None and rcmt != self.rcmt:
                    refresh_flag = True
                    self.v().partial_cache_update(f.start_ea)
                    #history_clear = True
                self.rcmt = None
            elif self.is_ea_to_be_processed(ea):
                rcmt = ida_bytes.get_cmt(ea, 1)
                if self.rcmt is not None and rcmt != self.rcmt:
                    refresh_flag = True
                    self.v().partial_cache_update(f.start_ea)
                    #history_clear = True
                self.rcmt = None
        elif self.cmdname == 'MakeComment':
            if self.is_ea_to_be_processed_cmt(ea):
                if f and f.start_ea == ea and self.cur_func is not None:
                    if hasattr(self.cur_func, "start_ea") and self.cur_func.start_ea == ea:
                        func_cmt = ida_funcs.get_func_cmt(f, 0)
                        if self.func_cmt is not None and func_cmt != self.func_cmt:
                            self.v().partial_cache_update(f.start_ea)
                            refresh_flag = True
                            #history_clear = True
                        self.func_cmt = None
                cmt = ida_bytes.get_cmt(ea, 0)
                if self.cmt is not None and cmt != self.cmt:
                    #self.v().update_caller_tif(ea, cmt)
                    self.update_tif(ea, cmt)
                    refresh_flag = True
                    #history_clear = True
                self.cmt = None
            elif self.is_ea_to_be_processed(ea):
                cmt = ida_bytes.get_cmt(ea, 0)
                if self.cmt is not None and cmt != self.cmt:
                    #self.v().update_caller_tif(ea, cmt)
                    self.update_tif(ea, cmt)
                    refresh_flag = True
                    #history_clear = True
                self.cmt = None
        elif self.cmdname == 'OpStructOffset':
            if f and self.is_ea_to_be_processed(f.start_ea):
                line = ida_kernwin.get_curline()
                if self.line is not None and line != self.line:
                    self.update_tif(ea)
                    refresh_flag = True
                    #history_clear = True
                self.line - None
        elif self.cmdname == 'SetColors':
            self.chk_dark_mode()
            
        #if history_clear:
        #    self.clear_history()
        if refresh_flag:
            #self.refresh()
            self.v().refresh_all(ea)
        #self.v().get_focus(self.v().GetWidget())
        return 0

    def refresh(self):
        self.v().refresh()
        
    def update_tif(self, ea, name=None):
        self.v().update_caller_tif(ea, name)
    
    def clear_history(self):
        pass
    
    def chk_dark_mode(self):
        return False

    """
    def screen_ea_changed(self, ea, prev_ea):
        self._log("Screen EA has been changed from %x to %x" % (prev_ea, ea))
        pass
    """
            
    def updating_actions(self, ctx):
        if self.v().cto_data["master"] != id(self.v()):
            return 0
        #self._log("Updating actions")
        #self._log(ctx.action, hex(ctx.cur_ea).rstrip("L"), ctx.cur_func, hex(ctx.cur_value).rstrip("L"))
        self.cur_ea = ctx.cur_ea
        self.cur_func = ctx.cur_func
        self.cur_value = ctx.cur_value
               
    """
    def updated_actions(self):
        #self._log("Updated actions")
        #self._log(hex(self.cur_ea).rstrip("L"), self.cur_func, hex(self.cur_value).rstrip("L"))
        pass
    """
            
# observing "IDA View-A" window
class my_view_hooks_t(ida_kernwin.View_Hooks):
    def __init__(self, g, title=g_ida_view):
        ida_kernwin.View_Hooks.__init__(self)
        self.hook()
        # let's use weakrefs, so as soon as the last ref to
        # the 'CallTreeOverviewer' instance is dropped, the 'my_view_hooks_t'
        # instance hooks can be automatically un-hooked, and deleted.
        # (in other words: avoid circular reference.)
        import weakref
        self.v = weakref.ref(g)
        self.title = title

    def _log(self, *msg):
        ida_kernwin.msg(">>> MyViewHook: %s%s" % (" ".join([str(x) for x in msg], os.linesep)))

    def _view_loc_changed(self, w, now, was):
        # for hooking and observing IDA View-A and synchronize with CTO's node
        if ida_kernwin.find_widget(self.title) == w:
            self._log("changed to %x from %x" % (now.place().toea(), was.place().toea()))
            # color the corresponded CTO's node
            self.update_widget_b(now, was)
                
        # for hooking and observing the CTO window and synchronize with IDA View-A's ea
        elif self.v().GetWidget() == w:
            self.view_loc_change_on_widget_b(now, was)
                    
    def view_loc_changed(self, w, now, was):
        try:
            self._view_loc_changed(w, now, was)
        except Exception as e:
            exc_type, exc_obj, tb = sys.exc_info()
            lineno = tb.tb_lineno
            ida_kernwin.msg("Got a unexpected error (%s: %s) (%d)%s" % (str(type(e)), e, lineno, os.linesep))
            if self.v().config.debug:
                self.v().dbg_print(traceback.format_exc())
            else:
                traceback.print_exc()
                    
    def update_widget_b(self, now, was):
        now_ea = now.place().toea()
        was_ea = was.place().toea()
        self.update_widget_b_ea(now_ea, was_ea)
                
    def update_widget_b_ea(self, now_ea, was_ea):
        pass
    
    def view_loc_change_on_widget_b(self, now, was):
        pass
    
                            
    def _view_click(self, w, ve):
        pass
                
    def view_click(self, w, ve):
        try:
            self._view_click(w, ve)
        except Exception as e:
            exc_type, exc_obj, tb = sys.exc_info()
            lineno = tb.tb_lineno
            ida_kernwin.msg("Got a unexpected error (%s: %s) (%d)%s" % (str(type(e)), e, lineno, os.linesep))
            if self.v().config.debug:
                self.v().dbg_print(traceback.format_exc())
            else:
                traceback.print_exc()
