# Binary loading

Before doing anything with a `quokka` file, you need to load it.

!!! tip
	For the remaining of this tutorial, I will be using an interactive IPython
	shell.

To load a program, `quokka` needs both the export file **and** the original 
binary. Indeed, to reduce the size of the export, information directly available
in the binary are not exported.

```python
import quokka
from pathlib import Path

samples = Path('docs/samples')
assert samples.is_dir(), "You should run this code in Quokka main directory"

prog = quokka.Program(samples / "qb-crackme.quokka", samples / "qb-crackme")

# Let's check if it worked
print(f'Program base address is 0x{prog.base_address:x}')
# Should print: Program base address is 0x8048000.
```

## Explore a bit the `Program` object

We now have in `prog` a complete program representation.
Using `dir(prog)`, you can see the different possibilities.

For instance, you can list the strings found in the program using :

```python
import quokka
prog = quokka.Program("docs/samples/qb-crackme.quokka", "docs/samples/qb-crackme")
print(f"Found {len(prog.strings)} in {prog.export_file.stem}")
```

!!! note "Using the protobuf directly"
	Of note, the protobuf generated by the plugin is available in `prog.proto`.
	You should not access it directly because it's a rather compact format not 
	well documented.


## Export and load
If the IDA plugin has been installed you can also create the exported directly using the convenient method `from_binary`.

```python
import quokka

prog = quokka.Program.from_binary('docs/samples/qb-crackme')
assert prog is not None, "Unable to export qb-crackme"
```

## Tips & tricks
By default, IDA will store the database of the exported file next to the binary
file. However, this is not convenient when dealing with read only filesystems. 
Use the ```database_file``` option to control where it should be stored.

```python
import quokka

prog = quokka.Program.from_binary('/bin/ls')
# Will fail because /bin is not writable
assert prog is None, "/bin is writable -- this is unusual"

prog = quokka.Program.from_binary('/bin/ls',
                                  output_file='docs/samples/ls.quokka',                                      
                                  database_file='docs/samples/ls.i64')
assert prog is not None, "Unable to export ls"
```

The `debug` parameter of the same method will control the output. One of its 
effect is that it will print the exporter output and the command used to launch 
it.