import os
import re
import subprocess
import collections
#
import ida_typeinf
import idaapi
import idautils
import idc
#
from data.tester import (
    split_db_fns,
    validate_map,
    validate_map_eq,
    validate_ref_addr,
    validate_ref_idb
)
#
from utils.plg_utils import get_fenced, strip_comments


def get_type_def(type_name):
    type_id = idc.get_struc_id(type_name)
    type_size = idc.get_struc_size(type_id)

    struct_def = []
    struct_def.append("{:08X} struct {} // sizeof=0x{:X}".format(0, type_name, type_size))
    struct_def.append("{:08X} {{".format(0))

    for m_offset, m_name, m_size in idautils.StructMembers(type_id):
        m_name_id = idc.get_member_id(type_id, m_offset)
        udm = ida_typeinf.udm_t()
        ida_typeinf.get_udm_by_fullname(udm, idc.get_struc_name(m_name_id))
        m_type = udm.type

        struct_def.append("{:08X}     {} {};".format(m_offset, m_type, m_name))

    struct_def.append("{:08X} }};".format(type_size))
    return "\n".join(struct_def)

def split(strng, sep, pos):
    strng = strng.split(sep)
    return sep.join(strng[:pos]), sep.join(strng[pos:])

def fix_count(env_desc, val):
    return 'many' if int(val) > 1 else 'single'

def fix_title(env_desc, val):
    return '"{}"'.format(val)

def fix_idle(env_desc, val):
    return val

def fix_types(env_desc, val):
    type_maps = val.split(';')
    types_to_vars = collections.defaultdict(list)

    for t_map in type_maps:
        symb_name, type_name = t_map.split(':')
        types_to_vars[type_name].append(symb_name)

    result = dict(types_to_vars)
    res = ""
    for typ, vrs in result.items():
        is_many = len(vrs) > 1
        symb_type = "The {} `{}` {} of type `{}`.\n".format(['variable', 'variables'][is_many], ", ".join(vrs), ['is', 'are'][is_many], typ)
        struct_def = get_type_def(type_name)
        type_defn = "The definition of the type `{}` is as follows:\n```\n{}\n```\n\n".format(type_name, struct_def)
        res += "{}{}".format(symb_type, type_defn)

    return res

def fix_assign(env_desc, val):
    if not any(s in val for s in [':', '=']):
        return None
    return re.sub(r',(?=\S)', ', ', val.replace(':', ' = ').replace(';', ','))

def fix_funcs(env_desc, val):
    if validate_ref_idb(val):
        fn_path = val.split(';')
        for p in fn_path:
            db_path, fn_refs = split_db_fns(p)

            script_dir = os.path.dirname(os.path.realpath(__file__))
            ida_path = env_desc.ida_exe
            script_path = os.path.join(script_dir, 'idbref.py')
            script_arg = fn_refs
            temp_arg = os.path.join(script_dir, 'idbref.txt')
            idb_path = db_path

            script_with_args = f'{script_path} {script_arg} {temp_arg}'
            command = [
                ida_path,
                '-A',
                f'-S{script_with_args}',
                idb_path
            ]

            process = subprocess.Popen(
                command,
                stdout=subprocess.PIPE,
                stderr=subprocess.PIPE,
                text=True
            )

            stdout, stderr = process.communicate()
            with open(temp_arg, "r") as f:
                md_text = f.read()
            os.remove(temp_arg)
            return strip_comments(md_text)
    else:
        md_text = ""
        for r in val.split(','):
            fn_addr = None
            if r[0].isdigit():
                fn_addr = int(r, 16)
            else:
                fn_addr = idc.get_name_ea_simple(r)
            if not fn_addr:
                return None
            fn_text = str(idaapi.decompile(fn_addr))
            md_text += get_fenced(fn_text)
        return strip_comments(md_text)
