#!/usr/bin/python
###########################################################
# Copyright (c) 2024
# Sergejs 'HRLM' Harlamovs <harlamism@gmail.com>
# Licensed under the MIT License. All rights reserved.
###########################################################

import collections
import dotenv
import inspect
import json
import logging
import os
import sys

lib_qt = None
try:
    from PyQt5 import QtCore, QtGui, QtWidgets
    lib_qt = "pyqt5"
except ImportError:
    try:
        from PySide import QtCore, QtGui
        from PySide import QtGui as QtWidgets
        lib_qt = "pyside"
    except ImportError:
        pass

# Make sub-modules discoverable.
SCRIPT_DIR = os.path.dirname(os.path.realpath(__file__))
if os.path.basename(SCRIPT_DIR) == 'idaxlm':
    sys.path.append(SCRIPT_DIR)
else:
    sys.path.append(os.path.join(SCRIPT_DIR, 'idaxlm'))
sys.path.append(os.path.join(SCRIPT_DIR, 'idaxlm', 'data'))

is_ida = True
try:  # IDA imports that are neutral to version changes.
    import idc
    import idaapi
    from idaapi import plugin_t, plugmod_t, PluginForm
    from idaxlm.shims import ida_shims
except ImportError:
    is_ida = False

    # Standalone-run caps.
    class plugin_t:
        pass
    class PluginForm:
        pass

from shims.qt_shims import (
    QCoreApplication,
    QFontDatabase,
    QIcon,
    QMessageBox,
    QTranslator
)
from ui import idaxlm_gui, ida_actions
# Resources are imported implicitly.
from assets import resource

env_path = os.path.join(SCRIPT_DIR, '.env')
dotenv.load_dotenv(dotenv_path=env_path)


class CustomLogger(logging.Logger):
    def makeRecord(self, name, level, fn, lno, msg, args, exc_info, func=None, extra=None, sinfo=None):
        # Create a default log record.
        record = super().makeRecord(name, level, fn, lno, msg, args, exc_info, func, extra, sinfo)

        # Get the outer frames.
        outer_frames = inspect.getouterframes(inspect.currentframe())

        # Assemble class hierarchy.
        hierarchy = []
        for frame_info in outer_frames:
            frame = frame_info.frame
            # Extract the class name if the frame is within a class context.
            if 'self' in frame.f_locals:
                class_name = frame.f_locals['self'].__class__.__name__
                if class_name != self.__class__.__name__:
                    hierarchy.append(class_name)
                    if int(os.getenv('DEBUG_MODE', 0)) == 0:
                        break
            elif 'cls' in frame.f_locals:
                class_name = frame.f_locals['cls'].__name__
                hierarchy.append(class_name)

        # Add hierarchy information to the log record.
        record.class_hierarchy = ' -> '.join(hierarchy)
        record.name = ' -> '.join(hierarchy)
        return record


class LoggingHandler():
    def __init__(self):
        super(LoggingHandler, self).__init__()
        formatter = logging.Formatter('[%(asctime)s][%(name)s][%(levelname)s] %(message)s')

        # Create a Handler and set its formatter.
        console_handler = logging.StreamHandler()
        console_handler.setFormatter(formatter)
        self.log = CustomLogger(self.__class__.__name__)
        self.log.setLevel(logging.DEBUG)
        self.log.addHandler(console_handler)


class ScriptEnv(LoggingHandler):
    def __init__(self, is_ida, lib_qt):
        super(ScriptEnv, self).__init__()
        # Generic environment.
        self.is_ida = is_ida
        self.lib_qt = lib_qt
        self.run_mode = 'script'
        self.dir_script = SCRIPT_DIR
        self.ver_py = sys.version_info[0]
        # IDA-specific environment.
        self.detectEnv()

    def __repr__(self):
        return self.get_dump(is_banner=True)

    def dump(self, is_banner=False):
        d_text = self.get_dump(is_banner=is_banner)
        if self.is_ida:
            ida_shims.msg(d_text)
        else:
            print(d_text)

    def get_dump(self, is_banner=False):
        env_repr = []
        if is_banner:
            env_repr.append(self.get_banner())

        env_repr.append('ENVIRONMENT\n')
        for prop in collections.OrderedDict(sorted(self.__dict__.items())):
            if prop not in ['log']:
                value = getattr(self, prop)
                if isinstance(value, list):
                    for i, v in enumerate(value):
                        dots = '.' * (80 - (len(prop) + len(str(v))) + 4)
                        prop = " " * len(prop) if i > 0 else prop
                        env_repr.append("{} {} {}".format(prop, dots, v))
                else:
                    dots = '.' * (80 - (len(prop) + len(str(value))) + 4)
                    env_repr.append("{} {} {}".format(prop, dots, value))
        env_repr.append('\n')
        return "\n".join(env_repr)

    def get_banner(self):
        banner = r"""
         _   ___    __        _     _
        | | | | \  / /\  \_/ | |   | |\/|
        |_| |_|_/ /_/--\ / \ |_|__ |_|  |
        """
        return banner

    def get_script_mode(self):
        mode = 'script'
        if idaapi.IDA_SDK_VERSION >= 720:
            if __name__ == "__main__":
                mode = "script"
            elif __name__.startswith('__plugins__'):
                mode = "plugin"
        else:
            plugin_dirs = ida_shims.get_ida_subdirs('plugins')
            if SCRIPT_DIR in plugin_dirs:
                mode = 'plugin'
            else:
                mode = 'script'
        return mode

    def get_plugin_ort(self):
        plg_loc = None
        plg_scope = 'None'
        g_plg_path = os.path.join(self.dir_plugin[0], 'idaxlm')
        l_plg_path = os.path.join(self.dir_plugin[1], 'idaxlm')
        if os.path.isdir(g_plg_path):
            plg_loc = self.dir_plugin[0]
            plg_scope = 'global'
        elif os.path.isdir(l_plg_path):
            plg_loc = self.dir_plugin[1]
            plg_scope = 'local'
        return (plg_loc, plg_scope)

    def detectEnv(self):
        if self.is_ida:
            self.ver_sdk = idaapi.IDA_SDK_VERSION
            self.ida_sample = ida_shims.get_input_file_path()
            self.dir_plugin = ida_shims.get_ida_subdirs("plugins")
            self.feat_asmtil = self.ver_sdk >= 840
            self.feat_bookmarks = self.ver_sdk >= 760
            # C++ class hierarchy and virtual function recognition.
            self.feat_cpp_oop = self.ver_sdk >= 720
            self.feat_folders = self.ver_sdk >= 750
            self.feat_golang = self.ver_sdk >= 760
            self.feat_goomba = self.ver_sdk >= 830
            # Point in time at which IDA v6.95 API compatibility was dropped.
            self.feat_ida6 = self.ver_sdk < 740
            # "IdaOnIda64" - 32-bit file analysis in IDA64 by default.
            self.feat_ioi64 = self.ver_sdk >= 820
            self.feat_img_search = self.ver_sdk >= 820
            self.feat_lumina = self.ver_sdk >= 720
            self.feat_makesig = self.ver_sdk >= 840
            # Microcode feature was introduced in v7.1, improved in v7.2
            self.feat_microcode = self.ver_sdk >= 710
            self.feat_microcode_new = self.ver_sdk >= 720
            self.feat_python3 = self.ver_sdk >= 740
            self.feat_rust = self.ver_sdk >= 840
            self.feat_swift = self.ver_sdk >= 820
            self.feat_undo = self.ver_sdk >= 730
            self.ida_arch = "x64" if idc.__EA64__ else "x86"
            self.ida_exe = sys.executable
            self.ida_kernel = idaapi.get_kernel_version()
            self.ida_module = os.path.basename(self.ida_sample) if self.ida_sample else None
            self.idb_path = ida_shims.get_idb_path()
            self.is_dbg = idaapi.is_debugger_on()
            self.lib_qt = self.lib_qt = "pyside" if self.ver_sdk < 690 else "pyqt5"
            self.platform = sys.platform
            plg_loc, plg_scope = self.get_plugin_ort()
            self.plg_loc = plg_loc
            self.plg_scope = plg_scope
            self.run_mode = self.get_script_mode()
            self.ver_hexrays = idaapi.get_hexrays_version() if idaapi.init_hexrays_plugin() else None


def common_init():
    env_desc = ScriptEnv(is_ida, lib_qt)
    return env_desc


__AUTHOR__ = "Sergejs 'HRLM' Harlamovs"
PLUGIN_NAME = "IDAxLM"
PLUGIN_HOTKEY = 'Ctrl+Alt+J'
PLUGIN_VERSION = '0.9'
PLUGIN_TITLE = '{0} v{1}'.format(PLUGIN_NAME, PLUGIN_VERSION)
PLUGIN_URL = "https://github.com/harlamism/IDAxLM"
PLUGIN_INFO = 'For usage see: <a href="{0}">{0}</a>'.format(PLUGIN_URL)


# Install the UI hook.
ui_hook = ida_actions.MenuHook()
ui_hook.hook()


class IdaXlmForm(PluginForm):
    def __init__(self, env_desc):
        super(IdaXlmForm, self).__init__()
        self.env_desc = env_desc
        self.icon = QIcon(':/idaxlm/icon_64.png')
        self.qss = os.path.join(SCRIPT_DIR, 'assets', 'style.qss')

    def OnCreate(self, form):
        self.env_desc.dump(True)

        app = QCoreApplication
        translator = QTranslator()
        translator.load('idaxlm/assets/i18n/tr_cn', os.path.dirname(__file__))
        app.installTranslator(translator)

        if self.env_desc.lib_qt == 'pyqt5':
            self.parent = self.FormToPyQtWidget(form)
        elif self.env_desc.lib_qt == 'pyside':
            self.parent = self.FormToPySideWidget(form)
        self.parent.setWindowTitle(PLUGIN_NAME)
        self.parent.setWindowIcon(self.icon)

        # The environment snapshot is propagated to the GUI.
        self.dialog = idaxlm_gui.IdaXlmDialog(self.env_desc)
        full_qss = open(self.qss).read()
        if int(os.getenv('PLUGIN_CUSTOM_FONT', 0)) == 1:

            font_path = os.path.join(SCRIPT_DIR, 'assets', 'font.ttf')
            font_id = QFontDatabase.addApplicationFont(font_path)
            if font_id != -1:
                font_name = QFontDatabase.applicationFontFamilies(font_id)[0]
                font_size = 12 if self.env_desc.platform == 'darwin' else 12
                font_qss = """
                    * {{
                        font-family: '{}';
                        font-size: {}px;
                    }}
                """.format(font_name, font_size)
                full_qss = font_qss + full_qss
        self.dialog.setStyleSheet(full_qss)
        layout = QtWidgets.QVBoxLayout()
        layout.addWidget(self.dialog)
        layout.setSpacing(0)
        layout.setContentsMargins(0, 0, 0, 0)
        self.parent.setLayout(layout)

    def OnClose(self,form):
        pass

    def Show(self, title):
        PluginForm.Show(self, title, options = PluginForm.WOPN_DP_BOTTOM | PluginForm.WOPN_PERSIST)
        for target in ["Output", "Pseudocode-A"]:
            dwidget = idaapi.find_widget(target)
            if dwidget:
                idaapi.set_dock_pos('IDAxLM', target, idaapi.DP_INSIDE)
                break


# class MyPlugmod(plugmod_t):
#     def __del__(self):
#         print(">>> MyPlugmod: destructor called.")
#
#     def run(self, arg):
#         print(">>> MyPlugmod.run() is invoked with argument value: {arg}.")
#         # for func_ea in idautils.Functions():
#         #     func_name = ida_funcs.get_func_name(func_ea)
#         #     print(f">>>MyPlugmod: Function{func_name} at address {func_ea:x}")

class IdaXlmPlugin(plugin_t):
    flags = idaapi.PLUGIN_KEEP  # idaapi.PLUGIN_MULTI, idaapi.PLUGIN_MOD
    comment = "IDA Language Models Toolset"
    help = "Edit->Plugin->IDAxLM or {}.".format(PLUGIN_HOTKEY)
    wanted_name = PLUGIN_NAME
    wanted_hotkey = PLUGIN_HOTKEY

    def init(self):
        super(IdaXlmPlugin, self).__init__()
        self.icon_id = 0
        ida_shims.msg("%s %s loaded\n" % (self.wanted_name, PLUGIN_VERSION))
        env_desc = common_init()
        self.f = IdaXlmForm(env_desc)
        ida_actions.ActionReg.register_actions(self.f)
        return idaapi.PLUGIN_KEEP  # MyPlugmod()  # return idaapi.PLUGIN_MULTI  # idaapi.PLUGIN_KEEP, idaapi.PLUGIN_MOD

    # Method triggered by hotkey press.
    def run(self, arg):
        self.f.Show('IDAxLM')
        return

    def term(self):
        ida_actions.ActionReg.unregister_actions()


def PLUGIN_ENTRY():
    return IdaXlmPlugin()
