#!/usr/bin/python
#############################################################################
# Copyright (c) 2024
# Sergejs 'HRLM' Harlamovs <harlamism@gmail.com>
# Licensed under the MIT License. All rights reserved.
#############################################################################
# Shim file for compatibility with IDA versions 6.x through 9.X
# If it is required, avoid calling the IDA API directly.
#
# The official Hex-Rays documentation:
# https://hex-rays.com
#  - /products/ida/support/idapython_docs/
#  - /products/ida/support/ida74_idapython_no_bc695_porting_guide/
#  - /products/ida/support/ida74_idapython_no_bc695_porting_guide.shtml
# https://docs.hex-rays.com
#  - /pre-release/developer-guide/idapython/idapython-porting-guide-ida-9
#
# For API implementation details and comments
# refer to corresponding modules in <IDA_PATH>/python/.
#############################################################################

import os
import sys

# IDA imports that are neutral to version changes.
import idc
import idaapi
import idautils

# Version specific IDA modules.
def try_import(module_name):
    try:
        return __import__(module_name)
    except ImportError:
        return None

idapy_modules = [
    'ida_allins',
    'ida_auto',
    'ida_bitrange',
    'ida_bytes',
    'ida_dbg',
    'ida_dirtree',
    'ida_diskio',
    'ida_entry',
    'ida_enum',
    'ida_expr',
    'ida_fixup',
    'ida_fpro',
    'ida_frame',
    'ida_funcs',
    'ida_gdl',
    'ida_graph',
    'ida_hexrays',
    'ida_ida',
    'ida_idaapi',
    'ida_idc',
    'ida_idd',
    'ida_idp',
    'ida_ieee',
    'ida_kernwin',
    'ida_lines',
    'ida_loader',
    'ida_merge',
    'ida_mergemod',
    'ida_moves',
    'ida_nalt',
    'ida_name',
    'ida_netnode',
    'ida_offset',
    'ida_pro',
    'ida_problems',
    'ida_range',
    'ida_regfinder',
    'ida_registry',
    'ida_search',
    'ida_segment',
    'ida_segregs',
    'ida_srclang',
    'ida_strlist',
    'ida_struct',
    'ida_tryblks',
    'ida_typeinf',
    'ida_ua',
    'ida_xref',
]

for name in idapy_modules:
    globals()[name] = try_import(name)


def _get_fn_by_version(ida_ver, nxt_fnc, prv_fnc, nxt_lib, prv_lib=None):
    """
    Determines the appropriate function to call based on the IDA version.

    :param ida_ver:  Backward incompatible version.
    :param nxt_fnc:  New function name.
    :param prv_fnc:  Old function name.
    :param nxt_lib:  New library ref.
    :param prv_lib:  Old library ref.

    :return: The function to call based on the IDA version.
    """
    is_ida_nxt = idaapi.IDA_SDK_VERSION >= ida_ver
    use_lib = nxt_lib if is_ida_nxt else (prv_lib if prv_lib else nxt_lib)
    use_fnc = nxt_fnc if is_ida_nxt else prv_fnc

    try:
        return getattr(use_lib, use_fnc)
    except AttributeError:
        raise Exception("{} is not a valid function in {}".format(use_fnc, use_lib))


def add_bpt(ea):
    fn = _get_fn_by_version(700, 'add_bpt', 'AddBpt', idc)
    return fn(ea)

def add_func(ea):
    fn = _get_fn_by_version(700, 'add_func', 'MakeFunction', ida_funcs, idc)
    return fn(ea)

def ask_file(for_saving, default, dialog):
    fn = _get_fn_by_version(700, 'ask_file', 'AskFile', ida_kernwin, idc)
    return fn(for_saving, default, dialog)

def ask_ident(default, prompt):
    fn = _get_fn_by_version(700, 'ask_str', 'AskIdent', ida_kernwin, idc)
    if idaapi.IDA_SDK_VERSION >= 700:
        return fn(default, ida_kernwin.HIST_IDENT, prompt)
    return fn(default, prompt)

def ask_yn(default, format_str):
    fn = _get_fn_by_version(700, 'ask_yn', 'AskYN', ida_kernwin, idc)
    return fn(default, format_str)

def can_decode(ea):
    fn = _get_fn_by_version(700, 'can_decode', 'decode_insn', ida_ua, idaapi)
    return fn(ea)

def choose_func(title):
    fn = _get_fn_by_version(700, 'choose_func', 'ChooseFunction', idc)
    return fn(title)

def create_dword(ea):
    fn = _get_fn_by_version(700, 'create_data', 'MakeDword', ida_bytes, idc)
    if idaapi.IDA_SDK_VERSION >= 700:
        return fn(ea, ida_bytes.FF_DWORD, 4, idaapi.BADADDR)
    return fn(ea)

def create_insn(ea):
    fn = _get_fn_by_version(700, 'create_insn', 'MakeCode', idc)
    return fn(ea)

def create_strlit(start, length):
    fn = _get_fn_by_version(700, 'create_strlit', 'MakeStr', ida_bytes, idc)
    if idaapi.IDA_SDK_VERSION >= 700:
        return fn(start, length, ida_nalt.STRTYPE_C)
    return fn(start, idc.BADADDR)

def decode_insn(ea):
    fn = _get_fn_by_version(700, 'decode_insn', 'decode_insn', ida_ua, idaapi)
    if idaapi.IDA_SDK_VERSION >= 700:
        insn = ida_ua.insn_t()
        fn(insn, ea)
        return insn
    fn(ea)
    return idaapi.cmd

def define_local_var(start, end, location, name):
    fn = _get_fn_by_version(700, 'define_local_var', 'MakeLocal', idc)
    return fn(start, end, location, name)

def find_func_end(ea):
    fn = _get_fn_by_version(700, 'find_func_end', 'FindFuncEnd', idc)
    return fn(ea)

def find_text(ea, y, x, searchstr, flag):
    fn = _get_fn_by_version(700, 'find_text', 'FindText', ida_search, idc)
    if idaapi.IDA_SDK_VERSION >= 700:
        return fn(ea, y, x, searchstr, flag)
    return fn(ea, flag, y, x, searchstr)

def generate_disasm_line(ea, flags):
    fn = _get_fn_by_version(700, 'generate_disasm_line', 'GetDisasmEx', idc)
    return fn(ea, flags)

def get_bookmark(index):
    fn = _get_fn_by_version(700, 'get_bookmark', 'GetMarkedPos', idc)
    return fn(index)

def get_bookmark_desc(index):
    fn = _get_fn_by_version(700, 'get_bookmark_desc', 'GetMarkComment', idc)
    return fn(index)

def get_bytes(func_addr, func_size):
    fn = _get_fn_by_version(700, 'get_bytes', 'GetManyBytes', idc)
    return fn(func_addr, func_size)

def get_func_cmt(ea, repeatable):
    fn = _get_fn_by_version(700, 'get_func_cmt', 'GetFunctionCmt', idc)
    return fn(ea, repeatable)

def get_color(ea, what):
    fn = _get_fn_by_version(700, 'get_color', 'GetColor', idc)
    return fn(ea, what) & 0xFFFFFF

def get_current_widget():
    fn = _get_fn_by_version(700, 'get_current_widget', 'get_current_tform', idaapi)
    return fn()

def get_first_seg():
    fn = _get_fn_by_version(700, 'get_first_seg', 'FirstSeg', idc)
    return fn()

def get_flags(ea):
    fn = _get_fn_by_version(700, 'get_flags', 'get_flags_novalue', ida_bytes)
    return fn(ea)

def get_full_flags(ea):
    fn = _get_fn_by_version(700, 'get_full_flags', 'getFlags', ida_bytes, idaapi)
    return fn(ea)

def get_func_attr(ea, attr):
    fn = _get_fn_by_version(700, 'get_func_attr', 'GetFunctionAttr', idc)
    return fn(ea, attr)

def get_func_flags(ea):
    fn = _get_fn_by_version(700, 'get_func_attr', 'GetFunctionFlags', idc)
    if idaapi.IDA_SDK_VERSION >= 700:
        return fn(ea, idc.FUNCATTR_FLAGS)
    return fn(ea)

def get_func_name(ea):
    fn = _get_fn_by_version(700, 'get_func_name', 'GetFunctionName', idc)
    return fn(ea)

def get_func_off_str(ea):
    fn = _get_fn_by_version(700, 'get_func_off_str', 'GetFuncOffset', idc)
    return fn(ea)

def get_highlighted_identifier():
    fn = _get_fn_by_version(700, 'get_highlight', 'get_highlighted_identifier', ida_kernwin, idaapi)
    if idaapi.IDA_SDK_VERSION >= 700:
        viewer = ida_kernwin.get_current_viewer()
        highlight = fn(viewer)
        if highlight and highlight[1]:
            return highlight[0]
    return fn()

def get_idb_path():
    fn = _get_fn_by_version(700, 'get_idb_path', 'GetIdbPath', idc)
    return fn()

def get_input_file_path():
    fn = _get_fn_by_version(700, 'get_input_file_path', 'GetInputFilePath', idaapi, idc)
    return fn()

def get_name(ea):
    fn = _get_fn_by_version(700, 'get_name', 'Name', idc)
    if idaapi.IDA_SDK_VERSION > 700:
        return fn(ea, ida_name.GN_VISIBLE)
    return fn(ea)

def get_name_ea(afrom, fname):
    fn = _get_fn_by_version(700, 'get_name_ea', 'LocByNameEx', idaapi, idc)
    return fn(afrom, fname)

def get_name_ea_simple(name):
    fn = _get_fn_by_version(700, 'get_name_ea_simple', 'LocByName', idc)
    return fn(name)

def get_next_seg(ea):
    fn = _get_fn_by_version(700, 'get_next_seg', 'NextSeg', idc)
    return fn(ea)

def get_operand_type(head, opnd_index):
    fn = _get_fn_by_version(700, 'get_operand_type', 'GetOpType', idc)
    return fn(head, opnd_index)

def get_operand_value(ea, idx):
    fn = _get_fn_by_version(700, 'get_operand_value', 'GetOperandValue', idc)
    return fn(ea, idx)

def get_screen_ea():
    fn = _get_fn_by_version(700, 'get_screen_ea', 'ScreenEA', idc)
    return fn()

def get_segm_attr(segea, attr):
    fn = _get_fn_by_version(700, 'get_segm_attr', 'GetSegmentAttr', idc)
    return fn(segea, attr)

def get_segm_end(ea):
    fn = _get_fn_by_version(700, 'get_segm_end', 'SegEnd', idc)
    return fn(ea)

def get_segm_name(ea):
    fn = _get_fn_by_version(700, 'get_segm_name', 'SegName', idc)
    return fn(ea)

def get_segm_start(ea):
    fn = _get_fn_by_version(700, 'get_segm_start', 'SegStart', idc)
    return fn(ea)

def set_func_cmt(ea, cmt, repeatable):
    fn = _get_fn_by_version(700, 'set_func_cmt', 'SetFunctionCmt', idc)
    return fn(ea, cmt, repeatable)

def get_strlit_contents(str_obj):
    fn = _get_fn_by_version(700, 'get_strlit_contents', 'GetString', ida_bytes, idc)
    if idaapi.IDA_SDK_VERSION >= 700:
        return fn(str_obj.ea, str_obj.length, str_obj.strtype)
    return fn(str_obj.ea)

def get_switch_info(ea):
    fn = _get_fn_by_version(700, 'get_switch_info', 'get_switch_info_ex', idaapi)
    return fn(ea)

def get_wide_dword(ea):
    fn = _get_fn_by_version(700, 'get_wide_dword', 'Dword', idc)
    return fn(ea)

def get_widget_vdui(form_widget_ref):
    fn = _get_fn_by_version(700, 'get_widget_vdui', 'get_tform_vdui', idaapi)
    return fn(form_widget_ref)

def has_xref(ea):
    fn = _get_fn_by_version(700, 'has_xref', 'hasRef', ida_bytes, idaapi)
    return fn(ea)

def insn_t():
    prop = _get_fn_by_version(700, 'insn_t', 'cmd', idaapi)
    if idaapi.IDA_SDK_VERSION >= 700:
        return prop()
    else:
        return prop

def is_byte(flags):
    fn = _get_fn_by_version(700, 'is_byte', 'isByte', ida_bytes, idc)
    return fn(flags)

def is_char0(flags):
    fn = _get_fn_by_version(700, 'is_char0', 'isChar0', ida_bytes, idc)
    return fn(flags)

def is_code(flag):
    fn = _get_fn_by_version(700, 'is_code', 'isCode', ida_bytes, idaapi)
    return fn(flag)

def is_data(flag):
    fn = _get_fn_by_version(700, 'is_code', 'isData', ida_bytes, idaapi)
    return fn(flag)

def is_double(flag):
    fn = _get_fn_by_version(700, 'is_double', 'isDouble', ida_bytes, idaapi)
    return fn(flag)

def is_dword(flag):
    fn = _get_fn_by_version(700, 'is_dword', 'isDwrd', ida_bytes, idaapi)
    return fn(flag)

def is_enum0(flag):
    fn = _get_fn_by_version(700, 'is_enum0', 'isEnum0', ida_bytes, idaapi)
    return fn(flag)

def is_float(flag):
    fn = _get_fn_by_version(700, 'is_float', 'isFloat', ida_bytes, idaapi)
    return fn(flag)

def is_loaded(ea):
    fn = _get_fn_by_version(700, 'is_loaded', 'isLoaded', ida_bytes, idc)
    return fn(ea)

def is_off0(flag):
    fn = _get_fn_by_version(700, 'is_off0', 'isOff0', ida_bytes, idaapi)
    return fn(flag)

def is_qword(flag):
    fn = _get_fn_by_version(700, 'is_qword', 'isQwrd', ida_bytes, idaapi)
    return fn(flag)

def is_strlit(flags):
    fn = _get_fn_by_version(700, 'is_strlit', 'isASCII', ida_bytes, idc)
    return fn(flags)

def is_struct(flag):
    fn = _get_fn_by_version(700, 'is_struct', 'isStruct', ida_bytes, idaapi)
    return fn(flag)

def is_unknown(flags):
    fn = _get_fn_by_version(700, 'is_unknown', 'isUnknown', ida_bytes, idc)
    return fn(flags)

def is_word(flag):
    fn = _get_fn_by_version(700, 'is_word', 'isWord', ida_bytes, idaapi)
    return fn(flag)

def jumpto(ea, opnum=-1, uijmp_flags=0x0001):
    fn = _get_fn_by_version(700, 'jumpto', 'Jump', ida_kernwin, idc)
    if idaapi.IDA_SDK_VERSION >= 700:
        return fn(ea, opnum, uijmp_flags)
    return fn(ea)

def msg(message):
    fn = _get_fn_by_version(700, 'msg', 'Message', ida_kernwin, idc)
    return fn(message)

def next_addr(ea):
    fn = _get_fn_by_version(700, 'next_addr', 'NextAddr', ida_bytes, idc)
    return fn(ea)

def next_head(ea, maxea=4294967295):
    fn = _get_fn_by_version(700, 'next_head', 'NextHead', idc)
    return fn(ea, maxea)

def op_plain_offset(ea, n, base):
    fn = _get_fn_by_version(700, 'op_plain_offset', 'OpOff', idc)
    return fn(ea, n, base)

def print_insn_mnem(ea):
    fn = _get_fn_by_version(700, 'print_insn_mnem', 'GetMnem', idc)
    return fn(ea)

def print_operand(ea, n):
    fn = _get_fn_by_version(700, 'print_operand', 'GetOpnd', idc)
    return fn(ea, n)

def refresh_idaview_anyway():
    fn = _get_fn_by_version(700, 'refresh_idaview_anyway', 'Refresh', idaapi, idc)
    return fn()

def set_color(ea, what, color):
    fn = _get_fn_by_version(700, 'set_color', 'SetColor', idc)
    return fn(ea, what, color)

def set_func_attr(ea, flags, value):
    fn = _get_fn_by_version(700, 'set_func_attr', 'SetFunctionFlags', idc)
    if idaapi.IDA_SDK_VERSION >= 700:
        return fn(ea, 36, value)
    return fn(ea, 36, value)

def set_func_flags(ea, flags):
    fn = _get_fn_by_version(700, 'set_func_attr', 'SetFunctionFlags', idc)
    if idaapi.IDA_SDK_VERSION >= 700:
        return fn(ea, idc.FUNCATTR_FLAGS, flags)
    return fn(ea, flags)

def set_name(ea, name, flags):
    fn = _get_fn_by_version(700, 'set_name', 'MakeName', idc)
    if idaapi.IDA_SDK_VERSION >= 700:
        return fn(ea, name, flags)
    return fn(ea, name)

def ua_mnem(ea):
    fn = _get_fn_by_version(700, 'ua_mnem', 'GetMnem', ida_ua, idc)
    return fn(ea)

def start_ea(obj):
    if not obj:
        return None
    try:
        return obj.startEA
    except AttributeError:
        return obj.start_ea

def end_ea(obj):
    if not obj:
        return None
    try:
        return obj.endEA
    except AttributeError:
        return obj.end_ea

def get_operands(insn):
    if idaapi.IDA_SDK_VERSION >= 700:
        return insn.ops
    return idaapi.cmd.Operands

def get_canon_feature(insn):
    if idaapi.IDA_SDK_VERSION >= 700:
        return insn.get_canon_feature()
    return idaapi.cmd.get_canon_feature()

def get_ida_subdirs(sub_folder):
    if idaapi.IDA_SDK_VERSION > 700:
        return idaapi.get_ida_subdirs("plugins")
    else:
        USR_PLUGIN_PATH = os.path.join(idaapi.get_user_idadir(), sub_folder)
        SYS_PLUGIN_PATH = os.path.join(idaapi.idadir(idaapi.PLG_SUBDIR))
        return [USR_PLUGIN_PATH, SYS_PLUGIN_PATH]

def get_chunk_eas(func_addr):
    """
    Check if a function is divided into chunks.
    """
    func_iter = idaapi.func_tail_iterator_t(idaapi.get_func(func_addr))
    status = func_iter.main()
    while status:
        chunk = func_iter.chunk()
        yield (start_ea(chunk), end_ea(chunk))
        status = func_iter.next()

def _is_func_chunked(func_addr):
    for eas in get_chunk_eas(func_addr):
        return True
    return False

def calc_func_size(func_desc):
    if idaapi.IDA_SDK_VERSION >= 720:
        return ida_funcs.calc_func_size(func_desc)
    else:
        beg_addr = start_ea(func_desc)
        if _is_func_chunked(beg_addr):
            func_size = 0
            for beg, end in idautils.Chunks(beg_addr):
                func_size += end - beg
            return func_size
        else:
            end_addr = end_ea(func_desc)
            if end_addr > beg_addr:
                return end_addr - beg_addr
            else:
                return 0
