#!/usr/bin/python
#############################################################################
# Copyright (c) 2024
# Sergejs 'HRLM' Harlamovs <harlamism@gmail.com>
# Licensed under the MIT License. All rights reserved.
#############################################################################
# Shim file for UI compatibility across IDA versions 6.x to 9.X
# Supports PySide v1.x and PyQt v5.x
# If backward compatibility is required, avoid direct calls to Qt API.
#
# Documentation provided by Qt and Riverbank Computing Ltd.:
#  - https://doc.qt.io/qtforpython-5/
#  - https://srinikom.github.io/pyside-docs/
#############################################################################

is_ida = True
try:
    import idaapi
except ImportError:
    is_ida = False

def _get_qt_class(mdl_pyqt, cls_name=None, mdl_pysd=None):
    """
    Dynamically imports and returns a Qt class based on IDA version.
    """
    fromlist = [cls_name] if cls_name else [mdl_pyqt]
    if is_ida and idaapi.IDA_SDK_VERSION > 680:
        mdl_used = __import__('PyQt5.{}'.format(mdl_pyqt), fromlist=fromlist)
    else:
        if not mdl_pysd:
            mdl_pysd = mdl_pyqt
        mdl_used = __import__('PySide.{}'.format(mdl_pysd), fromlist=fromlist)
    if cls_name:
        try:
            return getattr(mdl_used, cls_name)
        except AttributeError:
            return None
    return mdl_used


# Modules
def get_QtCore():
    return _get_qt_class('QtCore')

def get_QtGui():
    return _get_qt_class('QtGui')

def get_QtWidgets():
    return _get_qt_class('QtWidgets')


# QtCore -> QtCore
def get_DescendingOrder():
    return _get_qt_class('QtCore', 'DescendingOrder')

def get_QAbstractItemModel():
    return _get_qt_class('QtCore', 'QAbstractItemModel')

def get_QAbstractListModel():
    return _get_qt_class('QtCore', 'QAbstractListModel')

def get_QByteArray():
    return _get_qt_class('QtCore', 'QByteArray')

def get_QCoreApplication():
    return _get_qt_class('QtCore', 'QCoreApplication')

def get_QEvent():
    return _get_qt_class('QtCore', 'QEvent')

def get_QMargins():
    return _get_qt_class('QtCore', 'QMargins')

def get_QMetaObject():
    return _get_qt_class('QtCore', 'QMetaObject')

def get_QModelIndex():
    return _get_qt_class('QtCore', 'QModelIndex')

def get_QPoint():
    return _get_qt_class('QtCore', 'QPoint')

def get_QPointF():
    return _get_qt_class('QtCore', 'QPointF')

def get_QRect():
    return _get_qt_class('QtCore', 'QRect')

def get_QSize():
    return _get_qt_class('QtCore', 'QSize')

def get_QThread():
    return _get_qt_class('QtCore', 'QThread')

def get_QTranslator():
    return _get_qt_class('QtCore', 'QTranslator')

def get_Qt():
    return _get_qt_class('QtCore', 'Qt')

def get_Signal():
    return _get_qt_class('QtCore', 'pyqtSignal')


# QtGui -> QtGui
def get_QBrush():
    return _get_qt_class('QtGui', 'QBrush')

def get_QColor():
    return _get_qt_class('QtGui', 'QColor')

def get_QCursor():
    return _get_qt_class('QtGui', 'QCursor')

def get_QFont():
    return _get_qt_class('QtGui', 'QFont')

def get_QFontDatabase():
    return _get_qt_class('QtGui', 'QFontDatabase')

def get_QIcon():
    return _get_qt_class('QtGui', 'QIcon')

def get_QImage():
    return _get_qt_class('QtGui', 'QImage')

def get_QMouseEvent():
    return _get_qt_class('QtGui', 'QMouseEvent')

def get_QPainter():
    return _get_qt_class('QtGui', 'QPainter')

def get_QPalette():
    return _get_qt_class('QtGui', 'QPalette')

def get_QPen():
    return _get_qt_class('QtGui', 'QPen')

def get_QPixmap():
    return _get_qt_class('QtGui', 'QPixmap')

def get_QStandardItem():
    return _get_qt_class('QtGui', 'QStandardItem')

def get_QStandardItemModel():
    return _get_qt_class('QtGui', 'QStandardItemModel')

def get_QTextBlockFormat():
    return _get_qt_class('QtGui', 'QTextBlockFormat')

def get_QTextCursor():
    return _get_qt_class('QtGui', 'QTextCursor')


# QtGui -> QtCore
def get_QItemSelection():
    return _get_qt_class('QtCore', 'QItemSelection', 'QtGui')

def get_QItemSelectionModel():
    return _get_qt_class('QtCore', 'QItemSelectionModel', 'QtGui')

def get_QStringListModel():
    return _get_qt_class('QtCore', 'QStringListModel', 'QtGui')


# QtGui -> QtWidgets
def get_QAbstractItemView():
    return _get_qt_class('QtWidgets', 'QAbstractItemView', 'QtGui')

def get_QAction():
    return _get_qt_class('QtWidgets', 'QAction', 'QtGui')

def get_QApplication():
    return _get_qt_class('QtWidgets', 'QApplication', 'QtGui')

def get_QCheckBox():
    return _get_qt_class('QtWidgets', 'QCheckBox', 'QtGui')

def get_QComboBox():
    return _get_qt_class('QtWidgets', 'QComboBox', 'QtGui')

def get_QCompleter():
    return _get_qt_class('QtWidgets', 'QCompleter', 'QtGui')

def get_QDialog():
    return _get_qt_class('QtWidgets', 'QDialog', 'QtGui')

def get_QFileDialog():
    return _get_qt_class('QtWidgets', 'QFileDialog', 'QtGui')

def get_QFrame():
    return _get_qt_class('QtWidgets', 'QFrame', 'QtGui')

def get_QGroupBox():
    return _get_qt_class('QtWidgets', 'QGroupBox', 'QtGui')

def get_QHBoxLayout():
    return _get_qt_class('QtWidgets', 'QHBoxLayout', 'QtGui')

def get_QHeaderView():
    return _get_qt_class('QtWidgets', 'QHeaderView', 'QtGui')

def get_QLabel():
    return _get_qt_class('QtWidgets', 'QLabel', 'QtGui')

def get_QLineEdit():
    return _get_qt_class('QtWidgets', 'QLineEdit', 'QtGui')

def get_QListView():
    return _get_qt_class('QtWidgets', 'QListView', 'QtGui')

def get_QListWidget():
    return _get_qt_class('QtWidgets', 'QListWidget', 'QtGui')

def get_QListWidgetItem():
    return _get_qt_class('QtWidgets', 'QListWidgetItem', 'QtGui')

def get_QMainWindow():
    return _get_qt_class('QtWidgets', 'QMainWindow', 'QtGui')

def get_QMenu():
    return _get_qt_class('QtWidgets', 'QMenu', 'QtGui')

def get_QMessageBox():
    return _get_qt_class('QtWidgets', 'QMessageBox', 'QtGui')

def get_QProgressBar():
    return _get_qt_class('QtWidgets', 'QProgressBar', 'QtGui')

def get_QPushButton():
    return _get_qt_class('QtWidgets', 'QPushButton', 'QtGui')

def get_QRadioButton():
    return _get_qt_class('QtWidgets', 'QRadioButton', 'QtGui')

def get_QScrollArea():
    return _get_qt_class('QtWidgets', 'QScrollArea', 'QtGui')

def get_QSizePolicy():
    return _get_qt_class('QtWidgets', 'QSizePolicy', 'QtGui')

def get_QSlider():
    return _get_qt_class('QtWidgets', 'QSlider', 'QtGui')

def get_QSpacerItem():
    return _get_qt_class('QtWidgets', 'QSpacerItem', 'QtGui')

def get_QSplitter():
    return _get_qt_class('QtWidgets', 'QSplitter', 'QtGui')

def get_QStackedWidget():
    return _get_qt_class('QtWidgets', 'QStackedWidget', 'QtGui')

def get_QStyle():
    return _get_qt_class('QtWidgets', 'QStyle', 'QtGui')

def get_QStyleFactory():
    return _get_qt_class('QtWidgets', 'QStyleFactory', 'QtGui')

def get_QStyleOptionComboBox():
    return _get_qt_class('QtWidgets', 'QStyleOptionComboBox', 'QtGui')

def get_QStyleOptionSlider():
    return _get_qt_class('QtWidgets', 'QStyleOptionSlider', 'QtGui')

def get_QStyledItemDelegate():
    return _get_qt_class('QtWidgets', 'QStyledItemDelegate', 'QtGui')

def get_QTabWidget():
    return _get_qt_class('QtWidgets', 'QTabWidget', 'QtGui')

def get_QTableWidget():
    return _get_qt_class('QtWidgets', 'QTableWidget', 'QtGui')

def get_QTableWidgetItem():
    return _get_qt_class('QtWidgets', 'QTableWidgetItem', 'QtGui')

def get_QTextBrowser():
    return _get_qt_class('QtWidgets', 'QTextBrowser', 'QtGui')

def get_QTextEdit():
    return _get_qt_class('QtWidgets', 'QTextEdit', 'QtGui')

def get_QToolButton():
    return _get_qt_class('QtWidgets', 'QToolButton', 'QtGui')

def get_QTreeView():
    return _get_qt_class('QtWidgets', 'QTreeView', 'QtGui')

def get_QTreeWidget():
    return _get_qt_class('QtWidgets', 'QTreeWidget', 'QtGui')

def get_QTreeWidgetItem():
    return _get_qt_class('QtWidgets', 'QTreeWidgetItem', 'QtGui')

def get_QVBoxLayout():
    return _get_qt_class('QtWidgets', 'QVBoxLayout', 'QtGui')

def get_QWidget():
    return _get_qt_class('QtWidgets', 'QWidget', 'QtGui')


# Modules
QtCore = get_QtCore()
QtGui = get_QtGui()
QtWidgets = get_QtWidgets()

# QtCore -> QtCore
DescendingOrder = get_DescendingOrder()
QAbstractItemModel = get_QAbstractItemModel()
QAbstractListModel = get_QAbstractListModel()
QByteArray = get_QByteArray()
QCoreApplication = get_QCoreApplication()
QEvent = get_QEvent()
QMargins = get_QMargins()
QMetaObject = get_QMetaObject()
QModelIndex = get_QModelIndex()
QPoint = get_QPoint()
QPointF = get_QPointF()
QRect = get_QRect()
QSize = get_QSize()
QThread = get_QThread()
QTranslator = get_QTranslator()
Qt = get_Qt()
Signal = get_Signal()

# QtGui -> QtGui
QBrush = get_QBrush()
QColor = get_QColor()
QCursor = get_QCursor()
QFont = get_QFont()
QFontDatabase = get_QFontDatabase()
QIcon = get_QIcon()
QImage = get_QImage()
QMouseEvent = get_QMouseEvent()
QPainter = get_QPainter()
QPalette = get_QPalette();
QPen = get_QPen()
QPixmap = get_QPixmap()
QStandardItem = get_QStandardItem()
QStandardItemModel = get_QStandardItemModel()
QTextBlockFormat = get_QTextBlockFormat()
QTextCursor = get_QTextCursor()

# QtGui -> QtCore
QItemSelection = get_QItemSelection()
QStringListModel = get_QStringListModel()
QItemSelectionModel = get_QItemSelectionModel()

# QtGui -> QtWidgets
QAbstractItemView = get_QAbstractItemView()
QAction = get_QAction()
QApplication = get_QApplication()
QCheckBox = get_QCheckBox()
QComboBox = get_QComboBox()
QCompleter = get_QCompleter()
QDialog = get_QDialog()
QFileDialog = get_QFileDialog()
QFrame = get_QFrame()
QGroupBox = get_QGroupBox()
QHBoxLayout = get_QHBoxLayout()
QHeaderView = get_QHeaderView()
QLabel = get_QLabel()
QLineEdit = get_QLineEdit()
QListView = get_QListView()
QListWidget = get_QListWidget()
QListWidgetItem = get_QListWidgetItem()
QMainWindow = get_QMainWindow()
QMenu = get_QMenu()
QMessageBox = get_QMessageBox()
QProgressBar = get_QProgressBar()
QPushButton = get_QPushButton()
QRadioButton = get_QRadioButton()
QScrollArea = get_QScrollArea()
QSizePolicy = get_QSizePolicy()
QSlider = get_QSlider()
QSpacerItem = get_QSpacerItem()
QSplitter = get_QSplitter()
QStackedWidget = get_QStackedWidget()
QStyle = get_QStyle()
QStyleFactory = get_QStyleFactory()
QStyleOptionComboBox = get_QStyleOptionComboBox()
QStyleOptionSlider = get_QStyleOptionSlider()
QStyledItemDelegate = get_QStyledItemDelegate()
QTabWidget = get_QTabWidget()
QTableWidget = get_QTableWidget()
QTableWidgetItem = get_QTableWidgetItem()
QTextBrowser = get_QTextBrowser()
QTextEdit = get_QTextEdit()
QToolButton = get_QToolButton()
QTreeView = get_QTreeView()
QTreeWidget = get_QTreeWidget()
QTreeWidgetItem = get_QTreeWidgetItem()
QVBoxLayout = get_QVBoxLayout()
QWidget = get_QWidget()
