import collections
import functools
import itertools
import idaapi
import ida_funcs
import idc
import json
#
from shims import ida_shims, qt_shims
from utils import ida_utils, plg_utils
from shims.qt_shims import QtGui, QtWidgets, QtCore

class PickFuncHandler(idaapi.action_handler_t):
    def __init__(self, func, name):
        idaapi.action_handler_t.__init__(self)
        self.func = func
        self.name = name

    def activate(self, ctx):
        # executing function when the menu item is selected
        eas = []
        for pfn_idx in ctx.chooser_selection:
            pfn = ida_funcs.getn_func(pfn_idx)
            if pfn:
                eas.append(pfn.start_ea)
        if len(eas) == 1:
            addr = eas[0]
            name = idaapi.get_func_name(addr)
            anchor = None
            if self.name == 'idaxlm:dir':
                anchor = ida_utils.get_parent_dir(addr)
            elif self.name == 'idaxlm:pfx':
                anchor = ida_utils.get_external_pfx(name, True)
            elif self.name == 'idaxlm:col':
                anchor = ida_utils.get_func_color(addr)

            if anchor:
                self.func(eas, anchor)
            else:
                self.func(eas)
        else:
            self.func(eas)

        return 1

    def update(self, ctx):
        # Keep the context menu always accessible.
        return idaapi.AST_ENABLE_ALWAYS


DecorationRole2 = QtCore.Qt.UserRole + 1000

class IconDelegate(QtWidgets.QStyledItemDelegate):
    def paint(self, painter, option, index):
        super(self.__class__, self).paint(painter, option, index)
        value = index.data(DecorationRole2)
        if value:
            mode = QtGui.QIcon.Normal

            if not (option.state & QtWidgets.QStyle.State_Enabled):
                mode = QtGui.QIcon.Disabled
            elif option.state & QtWidgets.QStyle.State_Selected:
                mode = QtGui.QIcon.Selected

            if isinstance(value, QtGui.QPixmap):
                icon = QtGui.QIcon(value)
                option.decorationSize = value.size() / value.devicePixelRatio()

            elif isinstance(value, QtGui.QColor):
                pixmap = QtGui.QPixmap(option.decorationSize)
                pixmap.fill(value)
                icon = QtGui.QIcon(pixmap)

            elif isinstance(value, QtGui.QImage):
                icon = QtGui.QIcon(QtGui.QPixmap.fromImage(value))
                option.decorationSize = value.size() / value.devicePixelRatio()

            elif isinstance(value, QtGui.QIcon):
                state =  QtGui.QIcon.On if option.state & QtWidgets.QStyle.State_Open else QtGui.QIcon.Off
                actualSize = option.icon.actualSize(option.decorationSize, mode, state)
                option.decorationSize = QtCore.QSize(min(option.decorationSize.width(), actualSize.width()), min(option.decorationSize.height(), actualSize.height()))

            r = QtCore.QRect(option.rect)  # QtCore.QPoint(), option.decorationSize
            vertical_center = option.rect.center().y()
            r.setSize(option.decorationSize)
            r.moveLeft(option.rect.left() + 10)  # Adjust padding from the left edge
            r.moveCenter(QtCore.QPoint(r.center().x() + 25, vertical_center))  # option.rect.center().x() + 60, vertical_center
            # r.moveCenter(option.rect.center())
            # r.setLeft(160)

            state = QtGui.QIcon.On if option.state & QtWidgets.QStyle.State_Open else QtGui.QIcon.Off
            icon.paint(painter, r, QtCore.Qt.AlignLeft | QtCore.Qt.AlignVCenter, mode, state)


class ActionReg():
    @classmethod
    def get_act_data(cls): #V
        # Create the following assets, for each new entry:
        #   - _action_<action_name> method
        #   - ./assets/img/icon_<action_name>.png
        raw_data = [
            ("dir", "Pick same folder functions"),
            ("pfx", "Pick same prefix functions"),
            ("col", "Pick same color functions"),
            ("rec", "Pick functions recursively"),
            ("sel", "Pick selected functions")
        ]

        for token, descr in raw_data:
            act_name = "idaxlm:{}".format(token)
            act_text = descr
            act_file = "icon_{}.png".format(token)
            act_icon = idaapi.load_custom_icon(':/idaxlm/{}'.format(act_file))
            act_func = getattr(cls, "_action_{}".format(token))
            yield (act_name, act_text, act_icon, act_func)

    @classmethod
    def register_actions(cls, gui): #V
        for act_name, act_text, act_icon, act_func in cls.get_act_data():
            handler = functools.partial(act_func, gui)
            # Construct action description w/o specifying keyboard shortcut.
            act_desc = idaapi.action_desc_t(
                act_name,
                act_text,
                PickFuncHandler(handler, act_name),
                None,
                "Pick the related functions",
                act_icon
            )
            assert idaapi.register_action(act_desc), "Action registration failed"

    @classmethod
    def unregister_actions(cls): #V
        for act_name, _, act_icon, _ in cls.get_act_data():
            idaapi.unregister_action(act_name)
            idaapi.free_custom_icon(act_icon)

    @classmethod
    def _action_dir(cls, gui, eas, anchor):
        addr = eas[0]
        addrs = ida_utils.get_folder_funcs(anchor)
        gui.Show('IDAxLM')

        # Detect already existing function clusters.
        ea_order = sorted(addrs)
        clu_id = plg_utils.get_int_list_sha1_trunc(ea_order)
        if not gui.dialog.ui.taskView.is_task_id_present(clu_id):
            ActionReg._add_to_gui(gui, 'dir', anchor, addr, addrs, ea_order, clu_id)

    @classmethod
    def _action_pfx(cls, gui, eas, anchor):
        addr = eas[0]
        addrs = ida_utils.get_prefix_funcs(anchor)
        gui.Show('IDAxLM')

        # Detect already existing function clusters.
        ea_order = sorted(addrs)
        clu_id = plg_utils.get_int_list_sha1_trunc(ea_order)
        if not gui.dialog.ui.taskView.is_task_id_present(clu_id):
            ActionReg._add_to_gui(gui, 'pfx', anchor, addr, addrs, ea_order, clu_id)

    @classmethod
    def _action_col(cls, gui, eas, anchor):
        addr = eas[0]
        addrs = ida_utils.get_color_funcs(anchor)
        gui.Show('IDAxLM')

        ea_order = sorted(addrs)
        clu_id = plg_utils.get_int_list_sha1_trunc(ea_order)
        if not gui.dialog.ui.taskView.is_task_id_present(clu_id):
            anchor_str = plg_utils.RgbColor(anchor, invert=True).get_to_str()
            ActionReg._add_to_gui(gui, 'col', anchor_str, addr, addrs, ea_order, clu_id)

    @classmethod
    def _action_sel(cls, gui, eas):
        addr = eas[0]
        gui.Show('IDAxLM')

        ea_order = sorted(eas)
        clu_id = plg_utils.get_int_list_sha1_trunc(ea_order)
        if not gui.dialog.ui.taskView.is_task_id_present(clu_id):

            if len(eas) > 1:
                ActionReg._add_to_gui(gui, 'sel', 'custom', addr, eas, ea_order, clu_id)
            else:
                ActionReg._add_to_gui(gui, 'sel', 'single', None, eas, ea_order, clu_id)

    @classmethod
    def _action_rec(cls, gui, eas):
        func_addr = eas[0]
        name = idaapi.get_func_name(func_addr)
        gui.Show('IDAxLM')
        # nodes_xref_down = ida_utils.graph_down(func_addr, depth=0, path=dict(), convert=True)
        nodes_xref_down = ida_utils.graph_down_simple(func_addr)
        nodes_order = ida_utils.get_order_func_defs(nodes_xref_down)
        ea_order = sorted(nodes_order)
        clu_id = plg_utils.get_int_list_sha1_trunc(ea_order)
        if not gui.dialog.ui.taskView.is_task_id_present(clu_id):
            ActionReg._add_to_gui_struct(gui, 'tri', name, func_addr, nodes_xref_down, nodes_order, clu_id)

    @classmethod
    def build_tree(cls, parent, node_id, data):
        """ Recursively build tree from the data dictionary. """
        if node_id not in data:
            return

        # Create a QStandardItem for the current node
        # item = qt_shims.QStandardItem(str(node_id))
        name = idaapi.get_func_name(node_id)
        child1 = qt_shims.QStandardItem(name)
        child2 = qt_shims.QStandardItem(hex(node_id))
        parent.appendRow([child1, child2])

        # Recursively add child items
        for child_id in data[node_id]:
            cls.build_tree(child1, child_id, data)


    @classmethod
    def _add_to_gui_struct(cls, gui, clu_type, clu_name, clu_org, clu_eas, ea_order, clu_id):
        # Create the model for the QTreeView
        # model = QStandardItemModel()
        # tree_view.setModel(model)
        # model.setHorizontalHeaderLabels(['Address'])  # Set header label

        # Store the created items in a dictionary to easily reference parents
        parent_item = gui.dialog.ui.taskView.model().invisibleRootItem()

        clu_title = "tid_{}".format(clu_id)
        clud_description = "{}: {}, org: {}, fns: {}".format(clu_type, hex(clu_org), clu_name, len(clu_eas))
        item1 = qt_shims.QStandardItem(clu_title)
        item2 = qt_shims.QStandardItem(clud_description)

        parent_item.appendRow([item1, item2])
        parent_item = item1

        top_level_nodes = [node_id for node_id in clu_eas if all(node_id not in children for children in clu_eas.values())]
        func_set = ida_utils.get_func_set(clu_eas)

        for node_id in top_level_nodes:
            cls.build_tree(parent_item, node_id, clu_eas)

        cls.storeFuncClu(gui, clu_title, clud_description, func_set, ea_order, clu_id)
        cls.updateIndexes(gui, item1.index())

    #    items_by_level = {}
    #    # Iterate through the data dictionary
    #    for level, addresses in clu_eas.items():
    #        for ea in addresses:
    #            # Create the QStandardItem for this address
    #            name = idaapi.get_func_name(ea)
    #            child1 = qt_shims.QStandardItem(name)
    #            child2 = qt_shims.QStandardItem(hex(ea))

    #            # If this is the root level (level 0), add it to the root
    #            if level == 0:
    #                parent_item.appendRow([child1, child2])
    #            else:
    #                # Get the parent item from the previous level
    #                parent_item = items_by_level[level - 1][-1]  # Get the last item from the previous level
    #                parent_item.appendRow([child1, child2])

    #            # Store this item in the dictionary by its level
    #            if level not in items_by_level:
    #                items_by_level[level] = []
    #            items_by_level[level].append(child1)

    @classmethod
    def _add_to_gui(cls, gui, clu_type, clu_name, clu_org, clu_eas, clu_order, clu_id):
        parent_item = gui.dialog.ui.taskView.model().invisibleRootItem()

        tree_view = gui.dialog.ui.taskView  # Adjust according to your actual widget structure
        delegate = IconDelegate(tree_view)
        tree_view.setItemDelegateForColumn(1, delegate)

        # Create the QStandardItem with text and icon
        clu_title = "tid_{}".format(clu_id)

        item1 = qt_shims.QStandardItem(clu_title)
        item2 = qt_shims.QStandardItem()

        clu_description = None
        if clu_type == 'col': # isinstance(clu_name, str):
            clu_description = "{}:  ,".format(clu_type)
        elif clu_type == 'pfx':
            clu_description = "{}: {}_,".format(clu_type, clu_name)
        else:
            clu_description = "{}: {},".format(clu_type, clu_name)

        if clu_org:
            org_name = idaapi.get_func_name(clu_org)
            clu_description += " org: {}".format(org_name)
        else:
            if len(clu_eas) > 1:
                clu_description += " org: -"
            else:
                clu_description += " org: unclustered"
        item2.setText("{}, fns: {}".format(clu_description, len(clu_eas)))
        if clu_type == 'col':
            # item1.setData(cls.createDotPixmap(), role=QtCore.Qt.DecorationRole)
            col = QtGui.QColor(plg_utils.RgbColor(clu_name).get_to_int())
            item2.setData(cls.createDotPixmap(col=col), role=DecorationRole2)

        # Add it to the model under the root item
        parent_item.appendRow([item1, item2])

        # Create a child item.
        for ea in clu_eas:
            name = idaapi.get_func_name(ea)
            child1 = qt_shims.QStandardItem(name)
            child2 = qt_shims.QStandardItem(hex(ea))
            item1.appendRow([child1, child2])

        cls.storeFuncClu(gui, clu_title, clu_description, clu_eas, clu_order, clu_id)
        cls.updateIndexes(gui, item1.index())

    @classmethod
    def updateIndexes(cls, gui, topIndex, task=None):
        model = gui.dialog.ui.taskView.model()
        link = model.link

        row_count = model.rowCount(topIndex)
        sup_text = model.data(topIndex)  # right column index is ensured by switching columns in `model.index(row, 1, topIndex)`

        id = None
        if not task:
            id = sup_text.split('/')[0].replace('tid_', '')
            task = id
            link['tasks'][task]['view'] = topIndex
        else:
            id = int(str(sup_text), 16)

        for row in range(row_count):
            index = model.index(row, 1, topIndex)

            cell_text = model.data(index)
            func_addr = int(str(cell_text), 16)
            link['tasks'][task]['funcs'][func_addr]['view'] = index
            link['funcs'][func_addr]['views'].add(index)

            cls.updateIndexes(gui, index, task)

    @classmethod
    def storeFuncClu(cls, gui, clu_name, clu_text, clu_eas, clu_order, clu_id):
        # Assume cluster function order is unique and, so unique is hash.
        link = gui.dialog.ui.taskView.model().link
        link['tasks'][clu_id] = {
            'order': clu_order,
            'funcs': {ea: {'flag': True, 'view': None} for ea in clu_eas},
            'view': None,
            'flag': True
        }
        for ea in clu_eas:
            # model.link['funcs'][ea]['flag'] = True
            link['funcs'][ea]['tasks'].add(clu_id)
            # 'links' is initialized but empty

    @classmethod
    def createRectPixmap(cls, col=QtGui.QColor(240,50,50)):
        px = QtGui.QPixmap(12,12)
        px.fill(QtCore.Qt.transparent)
        pxSize = px.rect().adjusted(1,1,-1,-1)
        painter = QtGui.QPainter(px)
        painter.setRenderHint(QtGui.QPainter.Antialiasing)
        painter.setBrush(col)
        painter.setPen(QtGui.QPen(QtGui.QColor(150,20,20), 1.25))
        painter.drawRect(pxSize)
        painter.end()
        return px

    @classmethod
    def createDotPixmap(cls, col=QtGui.QColor(128,128,128)):
        px = QtGui.QPixmap(11,11)
        px.fill(QtCore.Qt.transparent)
        pxSize = px.rect().adjusted(1,1,-1,-1)
        painter = QtGui.QPainter(px)
        painter.setRenderHint(QtGui.QPainter.Antialiasing)

        border_color = col.darker(150)
        painter.setPen(QtGui.QPen(border_color, 1.25))

        painter.setBrush(col)
        # painter.setPen(QtGui.QPen(QtGui.QColor(15,15,15), 1.25))
        painter.drawEllipse(pxSize)
        painter.end()
        return px

def inject_prefix_actions(form, popup, form_type):
    """
    Inject prefix actions to popup menu(s) based on context.
    """
    if form_type == idaapi.BWN_FUNCS:
        func_selection = ida_utils.get_selected_funcs()
        is_select_single = len(func_selection) == 1

        # Provide prefix/folder/color-based funjction cherry-picking options
        anchor_pfx, anchor_dir, anchor_col = None, None, None
        if is_select_single:
            addr = func_selection[0]
            name = idaapi.get_func_name(addr)
            anchor_pfx = ida_utils.get_external_pfx(name, True)
            anchor_dir = ida_utils.get_parent_dir(addr)
            anchor_col = ida_utils.get_func_color(addr)

        sep_elem = [(None, None, None, None)]
        # Add separator in front, because of extra context menu items if folder view
        for act_name, *_ in itertools.chain(sep_elem, list(ActionReg.get_act_data()), sep_elem):
            if act_name == 'idaxlm:pfx' and is_select_single and anchor_pfx == None:
                continue
            if act_name == 'idaxlm:dir' and is_select_single and anchor_dir == None:
                continue
            if act_name == 'idaxlm:col' and is_select_single and anchor_col == None:
                continue
            if act_name in ['idaxlm:pfx', 'idaxlm:dir', 'idaxlm:col', 'idaxlm:rec'] and is_select_single == False:
                continue
            idaapi.attach_action_to_popup(
                form,
                popup,
                act_name,
                "Delete function(s)...",  # "Crete folder with items..." in case of folder feature enabled
                idaapi.SETMENU_INS
            )

    return 0

# Hook the context menu to add the action when a function is right-clicked
class MenuHook(idaapi.UI_Hooks):
    def ready_to_run(self):
        """
        UI ready to run -- an IDA event fired when everything is spunup.

        NOTE: this is a placeholder func, it gets replaced on a live instance
        but we need it defined here for IDA 7.2+ to properly hook it.
        """
        pass

    def finish_populating_widget_popup(self, widget, popup):
        """
        Triggered when the context menu is about to be displayed in IDA Pro v7+.
        """
        # way to find what was selected beforehand and what context menu items to apply
        inject_prefix_actions(widget, popup, idaapi.get_widget_type(widget))
        return 0
