# -*- coding: utf-8 -*-
import inspect
import json
import os
import re
from collections import defaultdict, OrderedDict

from shims.qt_shims import (
    QAction,
    QApplication,
    QAbstractItemView,
    QAbstractListModel,
    QComboBox,
    QColor,
    QCompleter,
    QCursor,
    QEvent,
    QFont,
    QFontDatabase,
    QFrame,
    QGroupBox,
    QIcon,
    QHeaderView,
    QHBoxLayout,
    QItemSelection,
    QItemSelectionModel,
    QLabel,
    QLineEdit,
    QListView,
    QMargins,
    QMenu,
    QMessageBox,
    QMouseEvent,
    QPainter,
    QPalette,
    QPen,
    QPixmap,
    QPoint,
    QPointF,
    QProgressBar,
    QPushButton,
    QToolButton,
    QTreeWidget,
    QTreeWidgetItem,
    QRect,
    QScrollArea,
    QSize,
    QSizePolicy,
    QSpacerItem,
    QStandardItem,
    QStandardItemModel,
    QStringListModel,
    QStyle,
    QStyledItemDelegate,
    Qt,
    QTabWidget,
    QTableWidget,
    QTableWidgetItem,
    QTextCursor,
    QTextBlockFormat,
    QTextEdit,
    QTreeView,
    QThread,
    QVBoxLayout,
    QWidget,
    QListWidget,
    QListWidgetItem,
    Signal
)

from utils.qt_utils import (
    i18n,
    set_fixed_size,
    getTopParent
)

from shims import ida_shims
from utils import lm_utils, plg_utils, ida_utils
import idaapi
import idc


class CustomFont(QFont):
    def __init__(self, font_name="Arial", font_size=9, env=None):
        font_path = os.path.join(os.path.dirname(__file__), 'assets', 'font.ttf')
        font_id = QFontDatabase.addApplicationFont(font_path)
        font_name = QFontDatabase.applicationFontFamilies(font_id)[0]
        super(CustomFont, self).__init__(font_name, font_size)

class CustomCursor(QCursor):
    def __init__(self, cursor_type=Qt.PointingHandCursor):
        super(CustomCursor, self).__init__(cursor_type)

class ProgressIndicator(QWidget):
    def __init__(self, parent=None):
        super(ProgressIndicator, self).__init__(parent)
        layout = QVBoxLayout()
        layout.addWidget(self.initProgressBar(parent))
        layout.setContentsMargins(0, 0, 0, 0)
        self.setLayout(layout)
        self.setProgress(0)
        self._worker = Worker()
        self._worker.updateProgress.connect(self.setProgress)

    def initProgressBar(self, parent):
        progress = QProgressBar(parent)
        progress.setMinimumSize(QSize(0, 5))
        progress.setMaximumSize(QSize(16777215, 5))
        progress.setTextVisible(False)
        self._progress = progress
        return progress

    def setProgress(self, progress):
        if progress == 0:
            self.setVisible(False)
        elif progress == 100:
            self.setVisible(False)
            self._progress.setValue(0)
        else:
            self.setVisible(True)
            self._progress.setValue(progress)

    def updateProgress(self, progress):
        self._worker.updateProgress.emit(progress)


class ColorButton(QPushButton):
    def __init__(self, name, size=(30, 30), parent=None):
        QPushButton.__init__(self, parent=parent)
        self.setObjectName(name)
        self.setMinimumSize(QSize(*size))
        self.setMaximumSize(QSize(*size))
        self.setCheckable(True)
        self.setCursor(QCursor(Qt.PointingHandCursor))


class Worker(QThread):
    updateProgress = Signal(int)

    def __init__(self):
        QThread.__init__(self)

    def run(self):
        for i in range(1, 101):
            self.updateProgress.emit(i)
            # time.sleep(0.01)


class FilterInputGroup(QWidget):
    def __init__(self, names, pholder, env_desc, parent=None):
        super(FilterInputGroup, self).__init__(parent)

        self._items = OrderedDict()

        self.env_desc = env_desc
        is_unicode = isinstance(names, basestring) if env_desc.ver_py == 2 else isinstance(names, str)
        if is_unicode:
            self._has_state = False
            names = [names]
        elif isinstance(names, list) and len(names) == 2:
            self._has_state = True
            self._names = names
            self._state = False

        name = names[0]
        layout = QHBoxLayout()
        layout.setContentsMargins(0, 0, 0, 0)
        layout.setSpacing(0)
        layout.addWidget(self.initText(name, self))
        layout.addWidget(self.initSelect(name))
        layout.setStretch(0, 0)
        layout.setStretch(1, 5)
        layout.setStretch(2, 7)
        layout.setStretch(3, 0)
        self.setMinimumSize(QSize(0, 26))
        self.setMaximumSize(QSize(16777215, 26))
        self.setText("")
        self.setPlaceholder(pholder)
        self.setLayout(layout)

    def initText(self, name, parent=None):
        label = QPushButton(parent)
        label.setText(name)
        label.setMinimumSize(QSize(96, 26))
        label.setMaximumSize(QSize(96, 26))
        label.setProperty('class', 'select-head')
        self._label = label
        if self._has_state:
            self._label.setCursor(QCursor(Qt.PointingHandCursor))
            self._label.clicked.connect(self.toggleMode)
        return label

    def initSelect(self, name, parent=None):
        select = CheckableComboBox()
        select.setEnabled(True)
        select.setAutoFillBackground(False)
        select.setMinimumSize(QSize(16777215, 26))
        select.setMaximumSize(QSize(16777215, 26))
        select.lineEdit().setText("")
        self._select = select
        return select

    def setPlaceholder(self, pholder):
        self._select.lineEdit().setPlaceholderText(pholder)


    def addItemSimple(self, txt):
        self._select.addItem((txt, None), userData=None)


    def addItems(self, items, is_sorted=False):
        for tpl in items:
            self.addItemSimple(tpl)
        if len(items):
            self.setEnabled(True)
            if is_sorted:
                self.sortItems()
        else:
            self.setEnabled(False)

    def sortItems(self):
        self._items = OrderedDict(sorted(self._items.items()))
        self._select.sortItems()

    def addItem(self, item, is_sorted=False, is_unique=False):
        is_skip = False
        txt, num, col = None, None, None
        if isinstance(item, str):
            item = (item)
        for t in item:
            if isinstance(t, str):
                txt = t
            if isinstance(t, int):
                num = t
            if isinstance(t, tuple):
                col = t
        if txt in self._items:
            if not is_unique:
                self._items[txt] += num
            else:
                is_skip = True
        else:
            self._items[txt] = num
        if num:
            txt = '{} ({})'.format(txt, num)
        if not is_skip:
            self._select.addItem((txt, col), userData=None)
        if is_sorted:
            self.sortItems()

    def chgItems(self, changelog, is_sorted=False):
        rem_items = []
        for mod in changelog:
            for lbl, val in changelog[mod].items():
                idx = list(self._items.keys()).index(lbl) if lbl in self._items else -1
                if mod == 'sub':
                    self._items[lbl] -= val
                    if self._items[lbl] == 0:
                        self._items.pop(lbl)
                        sel_count = len(self.getData())
                        self._select.removeItem(idx)
                        if sel_count == 1:
                            self.setText('')
                        else:
                            self._select.updateLineEditField()
                        rem_items.append(lbl)
                        continue
                elif mod == 'add':
                    if lbl in self._items:
                        self._items[lbl] += val
                    else:
                        self._items[lbl] = val
                new_entry = '{} ({})'.format(lbl, self._items[lbl])
                self._select.chgItem(idx, new_entry)

        if is_sorted:
            self.sortItems()
        return rem_items

    def setEnabled(self, state=False):
        self._select.setEnabled(state)

    def removeSelf(self):
        self._label.setParent(None)
        self._select.setParent(None)
        self.setParent(None)

    def setText(self, text):
        self._select.lineEdit().setText(text)

    def getData(self):
        text_data = self._select.getData()
        data = []
        if text_data:
            entries = text_data.split('; ')
            for e in entries:
                data.append(e.split(' ')[0])
        return data

    def toggleMode(self):
        self._state = not self._state
        caption = self._names[int(self._state)].upper()
        self._label.setText(i18n(caption))

    def getState(self):
        return self._state


class TaskItemModel(QStandardItemModel):
    def __init__(self, parent=None):
        super(TaskItemModel, self).__init__(parent)

        self.dataChanged.connect(self.handle_data_changed)

        self.pick = {
            'tasks': defaultdict(lambda: {
                'name': '',
                'funcs': set(),
                'full': False
            }),
            'funcs': dict(),
            'name': ''
        }

        self.link = {
            'tasks': defaultdict(lambda: {
                'funcs': defaultdict(lambda: {
                    'flag': True,
                    'view': None,
                }),
                'flag': None,
                'view': None,
                'order': set()
            }),
            'funcs': defaultdict(lambda: {
                'tasks': set(),
                'views': set(),
                'links': {
                    'base': '',
                    'quiz': '',
                    'chat': '',
                    'done': ''
                },
            })
        }

    # --
    def clear_pick(self):
        self.pick['tasks'].clear()
        self.pick['funcs'].clear()
        self.pick['name'] = ''

    # --
    def update_pick(self, to_select, task_hash, task_name, func_addr=None, func_name=None):
        if to_select:
            # if func_addr:
            self.pick['funcs'][func_addr] = func_name
            self.pick['tasks'][task_hash]['name'] = task_name
            self.pick['tasks'][task_hash]['funcs'].add(func_addr)
            if len(self.pick['tasks'][task_hash]['funcs']) == len(self.link['tasks'][task_hash]['funcs'].keys()):
                self.pick['tasks'][task_hash]['full'] = True
            # else:
            #     for func_addr in self.link['tasks'][task_hash]['funcs'].keys():
            #         self.pick['funcs'][func_addr] = ''
            #         self.pick['tasks'][task_hash]['funcs'].add(func_addr)
            #     self.pick['tasks'][task_hash]['full'] = True
            #     self.pick['tasks'][task_hash]['name'] = task_name
        else:
            # if func_addr:
            self.pick['funcs'].pop(func_addr)
            self.pick['tasks'][task_hash]['funcs'].remove(func_addr)
            self.pick['tasks'][task_hash]['full'] = False
            if (len(self.pick['tasks'][task_hash]['funcs'])) == 0:
                self.pick['tasks'].pop(task_hash)
            # else:
            #     for func_addr in self.link['tasks'][task_hash]['funcs'].keys():
            #         self.pick['funcs'].pop(func_addr)
            #     self.pick['tasks'].pop(task_hash)

    def flags(self, index):
        if not index.isValid():
            return Qt.NoItemFlags

        # Allow editing only for topmost parent items and first column
        if index.column() == 0:  # First field
            parent = self.itemFromIndex(index).parent()
            if parent is None:  # Topmost parent (no parent item)
                return Qt.ItemIsSelectable | Qt.ItemIsEnabled | Qt.ItemIsEditable

        # For all other items, disable editing
        return Qt.ItemIsSelectable | Qt.ItemIsEnabled

    # Slot to handle data changes
    def handle_data_changed(self, top_left, bottom_right):
        # Efficiently handle changes without unnecessary iteration
        if top_left == bottom_right:  # Single cell edit
            value = self.data(top_left, Qt.EditRole)  # .model
            print(f"Updated value at row {top_left.row()}, column {top_left.column()}: {value}")
        else:
            print("Multiple cells updated, handle as needed.", top_left.row(), bottom_right.row(), top_left.column(), bottom_right.column())


    def updateRowState(self):
        for fn_idx in self.pick['funcs'][fn_addr]['ref_idx']:
            self.model.setRowColor(fn_idx.row(), is_excluded)
        # beg_idx = self.index(row, 0)
        # end_idx = self.index(row, self.columnCount() - 1)
        # self.dataChanged().emit(beg_idx, end_idx, [Qt.ForegroundRole])

    # def data(self, index, role):
    #     if role == Qt.ForegroundRole:
    #         return QColor(Qt.gray) if is_excluded else QColor(Qt.black)
    #     return super().data(index, role)

    def is_task_id_present(self, task_id):
        res = task_id in self.link['tasks']
        if res:
            response = QMessageBox.warning(
                None,
                "Same cluster",
                "Cluster with same function set already exists: tid_{}".format(task_id),
                QMessageBox.Ok
            )
        return res

    # --
    def update_pick_name(self):
        title = None
        func_count = len(self.pick['funcs'])
        if func_count > 0:
            is_set = all(task_desc['full'] for task_desc in self.pick['tasks'].values())
            if is_set:
                title = plg_utils.get_list_repr([task['name'] for task in self.pick['tasks'].values()])
            else:
                title = plg_utils.get_list_repr(self.pick['funcs'].values())
        else:
            title = 'none'
        self.pick['name'] = "selection ({}): {}".format(func_count, title)


class TaskView(QTreeView):
    pickUpdate = Signal(object)

    def __init__(self, parent=None):
        super(TaskView, self).__init__(parent)

        self.setSortingEnabled(True)
        self.setAlternatingRowColors(True)

        model = TaskItemModel()
        model.setHorizontalHeaderLabels(['Name', 'Address'])
        self.setModel(model)
        self.setColumnWidth(0, 280)
        self.setColumnWidth(1, 280)

        self.setSelectionMode(QAbstractItemView.ExtendedSelection)
        self.setSelectionBehavior(QTableWidget.SelectRows)
        self.selectionModel().selectionChanged.connect(self.onSelectionChanged)

        # self.setEditTriggers(QAbstractItemView.NoEditTriggers)
        self.installEventFilter(self)
        self.setContextMenuPolicy(Qt.CustomContextMenu)
        self.customContextMenuRequested.connect(self.showContextMenu)
        self.pickUpdate.emit(model.pick)

    # --
    def scrollTo(self, index, hint):
        """Prevent horizontal self-scrolling on row selection"""
        super().scrollTo(index, hint)
        self.horizontalScrollBar().setValue(self.horizontalScrollBar().minimum())

    # --
    def eventFilter(self, source, event):
        if source is self and event.type() == event.KeyPress and event.key() == Qt.Key_Delete:
            self.delete_selected_nodes()
            return True
        return super().eventFilter(source, event)


    # Dictionary of all parents within the selection.
    # No information collected, whether parent is `task` or `func`.
    # Each row is guaranteed to have parent - whether task of func.
    # Since there is no digging - childs and parents are relative.

    # it is impossible to select(!) just task, w/o selecting the functions

    # Sort child indexes by row in reverse order before deleting.
    # Bottom-up propagation.

    def delete_selected_nodes(self):
        model = self.model()

        removal = {
            'views': defaultdict(list),
            'funcs': defaultdict(list)
        }

        # Prepare data
        for view in self.selectedRows():  # selected rows is the ground truth for removal
            if view.isValid():            # gather information to be reflected to view and models
                supView = view.parent()
                if supView.isValid():
                    removal['views'][supView].append(view)
                    task_id, task_name, func_id, func_name = self.get_model_func(view)
                    removal['funcs'][func_id].append(task_id)

        # Delete items, grouped by parent, in reverse order
        for supIdx, subIds in removal['views'].items():
            subIds.sort(key=lambda x: x.row(), reverse=True)
            for childIdx in subIds:
                model.removeRow(childIdx.row(), childIdx.parent())  # model.removeRow(childIdx.row())  # ambiguity
                if model.rowCount(supIdx) == 0:
                    model.removeRow(supIdx.row())

        # self.reflectToData(removal_log)

    def reflectToData(self, removal_log):
        link = self.model().link
        for rem_desc in removal_log:
            task, func = rem_desc
            if func == None:
                link['tasks'].pop(task)
            else:
                func_tasks = link['funcs'][func]['tasks']
                if task in func_tasks and len(func_tasks) == 1:
                    link['funcs'].pop(func)
                    link['stats']['calc'] -= 1
                else:
                    link['funcs'][func]['tasks'].remove(task)

    def is_tree_node(self, index):
        return self.model().rowCount(index) != 0

    def get_node_index(self, index):
        if index.isValid():
            return index.sibling(index.row(), 0)
        else:
            return None

    def test(self):
        print("test")

    def showContextMenu(self, position):
        # Consider click index separately, because right-clicked row can be not pre-selected
        i_node = self.get_node_index(self.indexAt(position))  # i_line <-- i_cell

        if i_node:
            i_sels = self.selectedRows()

            pick = self.model().pick
            link = self.model().link

            is_node_shared = False
            for func in pick['funcs'].keys():
                if len(link['funcs'][func]['tasks']) > 1:
                    is_node_shared = True
                    break


            flags = {
                link['tasks'][task]['funcs'][func]['flag']
                for task in pick['tasks']
                for func in pick['tasks'][task]['funcs']
            }

            flag_state = int(next(iter(flags))) if len(flags) == 1 else -1


            # Click is made inside the tree view table
            contextMenu = QMenu(self)

            if self.is_tree_node(i_node) and self.hasParent(i_node):
                is_tree_complete = self.are_all_children_selected(i_node)
                toggleTrees = contextMenu.addAction(
                    "Deselect Tree" if is_tree_complete else "Select Tree"
                )
                toggleTrees.triggered.connect(
                    lambda: self.selectNodeChildren(i_node, not bool(is_tree_complete), i_node not in i_sels)
                )

            if flag_state in [0, -1]:
                toggleSelected = contextMenu.addAction("Enable Selected")
                toggleSelected.triggered.connect(
                    lambda: self.toggleEnabledItems(to_disable=False, all_occurences=False)
                )

                if is_node_shared:
                    toggleOccurences = contextMenu.addAction("Enable Occurrences")
                    toggleOccurences.triggered.connect(
                        lambda: self.toggleEnabledItems(to_disable=False, all_occurences=True)
                    )

            if flag_state in [1, -1]:
                toggleSelected = contextMenu.addAction("Disable Selected")
                toggleSelected.triggered.connect(
                    lambda: self.toggleEnabledItems(to_disable=True, all_occurences=False)
                )

                if is_node_shared:
                    toggleOccurences = contextMenu.addAction("Disable Occurrences")
                    toggleOccurences.triggered.connect(
                        lambda: self.toggleEnabledItems(to_disable=True, all_occurences=True)
                    )

            contextMenu.exec_(self.viewport().mapToGlobal(position))

        else:
            # Click is made outside the tree view table
            return


    def gray_out(self, index, is_excluded):
        for col in range(self.model().columnCount()):
            item = self.model().itemFromIndex(index.siblingAtColumn(col))
            if item:
                item.setForeground(Qt.gray)
                item.setForeground(Qt.gray if is_excluded else Qt.black)

    def toggleEnabledItems(self, to_disable, all_occurences=False):
        pick = self.model().pick
        link = self.model().link

        state = not to_disable
        indexes = set()
        if all_occurences:
            for func_addr in pick['funcs']:
                for task in link['funcs'][func_addr]['tasks']:
                    link['tasks'][task]['funcs'][func_addr]['flag'] = state
                indexes.update(link[func_addr]['views'])
        else:
            for task in pick['tasks']:
                for func_addr in pick['tasks'][task]['funcs']:
                    link['tasks'][task]['funcs'][func_addr]['flag'] = state
                    self.gray_out(link['tasks'][task]['funcs'][func_addr]['view'], to_disable)
                if pick['tasks'][task]['full'] == True:
                    self.gray_out(link['tasks'][task]['view'], to_disable)

        # self.updateRows()

    # --
    def isItemGrayedOut(self, index):
        item = self.model().itemFromIndex(index)
        return item.foreground().color() == Qt.gray if item else False

    def get_func_desc(self, subIndex):
        model = self.model()

        subItem = model.itemFromIndex(subIndex)
        supItem = getTopParent(subItem)
        task_id = supItem.text().split('/')[0]

        subIndex = subIndex.sibling(subIndex.row(), 1)
        subItem = model.itemFromIndex(sb_index)
        func_id = int(str(subItem.text()), 16)

        return (task_id, func_id)


    def reflToData(self, disable_log, state):
        for dis_desc in disable_log:
            task, func = dis_desc
            # self.link['stats']['calc'] += (1 if state else -1)
            self.link['tasks'][task]['funcs'][func] = state

    def toggleRowExclude(self, index, is_excluded):
        """Include/exclude a row but not its children."""
        if not index.isValid():
            return

        model = self.model()

        for col in range(model.columnCount()):
            item = model.itemFromIndex(index.siblingAtColumn(col))
            if item:
                item.setForeground(Qt.gray)


    def get_row_data(self, row_index):
        model = self.model()
        row_data = tuple(
            model.data(model.index(row_index.row(), col))
            for col in range(model.columnCount())
        )

    def get_cell_text(self, index):
        return self.model().itemFromIndex(index).text()

    def get_cell_origin(self, index):
        while index.parent().isValid():
            index = index.parent()
        return index

    def get_cell_next(self, index):
        return index.sibling(index.row(), 1)

    def get_task_id(self, task_name):
        return task_name.split('/')[0].replace('tid_', '')

    def get_func_id(self, func_name):
        return ida_utils.get_func_ea_by_ref(func_name)

    def get_cell_task_name(self, index):
        """Get task name from the specified table cell"""
        return self.get_cell_text(index)

    def get_cell_func_name(self, index):
        """Get function name from the specified table cell"""
        return self.get_cell_text(index)

    def get_cell_func_addr(self, index):
        """Get function address (hex) from the specified table cell"""
        return self.get_func_id(self.get_cell_text(index))

    def get_view_func(self, index):
        func_name = self.get_cell_func_name(index)
        func_id = self.get_cell_func_addr(self.get_cell_next(index))
        return (func_id, func_name)

    def get_view_task(self, index):
        task_name = self.get_cell_task_name(index)
        task_id = self.get_task_id(task_name)
        return (task_id, task_name)

    def is_view_func(self, index):
        return index.parent().isValid()

    def get_model_func(self, index):
        func_id, func_name = (
            self.get_view_func(index) if self.is_view_func(index) else (None, None)
        )
        task_id, task_name = self.get_view_task(self.get_cell_origin(index))
        return (task_id, task_name, func_id, func_name)

    def selectedRows(self):
        return self.selectionModel().selectedRows()

    def childRows(self, index):
        rowCount = self.model().rowCount(index)
        for rowNum in range(rowCount):
            yield index.child(rowNum, 0)  # child_index = model.index(row, 0, parent_index)

    def get_cell_childs(self, index):
        model = self.model()
        child_indexes = []

        rowCount = model.rowCount(parent_index)

        for rowNum in range(rowCount):
            # Get the child index at the given row
            childIdx = model.index(rowNum, 0, parent_index)
            if childIdx.isValid():
                child_indexes.append(childIdx)
                # Recursively get children of this child
                child_indexes.extend(get_cell_childs(tree_view, childIdx))

        return child_indexes

    def hasParent(self, rowIdx):
        return rowIdx.parent().isValid()

    def selectionRows(self, selection):  # QItemSelection
        for index in selection.indexes():
            if index.column() == 0:
                yield index

    def onSelectionChanged(self, selected, deselected):
        model = self.model()

        changes = [deselected, selected]

        for idx, sel in enumerate(changes):
            to_select = bool(idx)
            for index in self.selectionRows(sel):

                if not index.parent().isValid():
                    # If a task is (de)selected, (de)select all nested child functions
                    self.selectChildren(index, to_select)
                else:
                    # If all task's child functions are selected, select the task
                    # If any task's child function is deselected, deselect the task
                    self.selectionModel().blockSignals(True)
                    # Temporarily block signals when deselecting or selecting the parent
                    self.selectParent(index)
                    self.viewport().update()
                    self.selectionModel().blockSignals(False)

                    task_id, task_name, func_id, func_name = self.get_model_func(index)
                    model.update_pick(to_select, task_id, task_name, func_id, func_name)

        model.update_pick_name()
        self.pickUpdate.emit(model.pick)

    def selectSelChildren(self, selection, to_select):
        for index in selection:
            self.selectChildren(index, to_select)

    def selectNodeChildren(self, index, to_select, to_root):
        if to_root:
            model = self.model()
            flag = QItemSelectionModel.Select if to_select else QItemSelectionModel.Deselect
            self.selectionModel().select(index, flag | QItemSelectionModel.Rows)
        self.selectChildren(index, to_select)

    def selectChildren(self, index, to_select):
        model = self.model()
        flag = QItemSelectionModel.Select if to_select else QItemSelectionModel.Deselect

        for row in self.childRows(index):
            self.selectionModel().select(row, flag | QItemSelectionModel.Rows)
            self.selectChildren(row, to_select)

    def selectParent(self, func_index):
        task_index = self.get_cell_origin(func_index)
        to_select = self.are_all_children_selected(task_index)
        flag = QItemSelectionModel.Select if to_select else QItemSelectionModel.Deselect
        self.selectionModel().select(task_index, flag | QItemSelectionModel.Rows)

    def are_all_children_selected(self, parent_index):
        """Recursively check if all children of a given index are selected."""
        parent_item = self.model().itemFromIndex(parent_index)
        for row in range(parent_item.rowCount()):
            child_item = parent_item.child(row)
            child_index = self.model().indexFromItem(child_item)

            # If this child is not selected
            if not self.selectionModel().isSelected(child_index):
                return False

            # If this child has its own children, check them recursively
            if child_item.hasChildren() and not self.are_all_children_selected(child_index):
                return False

        return True

    def is_task_id_present(self, task_id):
        return self.model().is_task_id_present(task_id)


class ContextTab(QWidget):
    def __init__(self, parent=None, env=None):
        super(ContextTab, self).__init__(parent)
        self.env = env
        self.wgt = parent
        self.is_enabled = False
        layout = self.genLayout(parent)
        self.setLayout(layout)
        self.data = self.loadContextData()
        self._tool._cwContextTemplate._select.currentIndexChanged.connect(self.contextTypeChanged)
        self._tool.setSubmitHandler(self.submitContext)

    def genLayout(self, parent):
        layout = QVBoxLayout()
        self._table = ContextTable(parent)
        self._tool = ContextTool(parent, self.env)
        layout.addWidget(self._table)
        layout.addWidget(self._tool)
        return layout

    def contextTypeChanged(self):
        if self.is_enabled:
            self._tool._submit.setEnabled(True)

    def setToolEnabled(self, state):
        self.is_enabled = state
        self._tool._submit.setEnabled(state)

    def loadContextData(self):
        schemas = plg_utils.get_json('data/context.json')
        pickers = plg_utils.get_funcs('data/picker.py', 'get_')
        mutants = plg_utils.get_funcs('data/mutant.py', 'fix_')
        self._tool.loadData(schemas, pickers, mutants)
        return schemas

    def submitContext(self):
        tool_data = self._tool.getValues()  # {'temp': 'xrefs', 'handle': '', 'mutant': 'fix_count', 'prompt': 'This function is called by {} function(s).'}
        mutant_name = tool_data['mutant']
        module = plg_utils.import_path_simple('mutant')

        mutant_data = None
        mutant_func = None
        try:
            mutant_func = getattr(module, mutant_name)  # BB
            mutant_data = mutant_func(self.env, tool_data['handle'])
        except AttributeError:
            mutant_data = ""

        comp = {
            'temp': tool_data['temp'],
            'text': tool_data['prompt'].format(mutant_data)
        }

        self._table.addRow((*comp.values(),))

        comp_hash = plg_utils.get_dict_sha1_trunc(comp)
        self.wgt.bank['regs']['context'][comp_hash] = (*comp.values(),)

        # recompilation
        bag_key = self.wgt.bags['base']
        if not bag_key:
            bag_key = plg_utils.get_str_list_sha1_trunc([comp_hash])
            self.wgt.bank['bags']['context'][bag_key].add(comp_hash)
        else:
            regs = self.wgt.bank['bags']['context'][bag_key]
            self.wgt.bank['bags']['context'].pop(bag_key)
            regs.add(comp_hash)
            bag_key = plg_utils.get_str_list_sha1_trunc(regs)
            self.wgt.bank['bags']['context'][bag_key] = regs

        for func_addr in self.wgt.pick:
            self.wgt.gui.ui.taskView.model().link['funcs'][func_addr]['links']['base'] = bag_key

        self.wgt.bags['base'] = bag_key

        ## # if something has changed
        ## key = tool_data.pop('temp')
        ## selected_template = self._tool.selected_template
        ## if key != selected_template:
        ##     if self._tool.selected_template == qt_widgets.CustomCombo.NEW_LABEL[1:-1]:
        ##         self.insertContextData(key, tool_data)
        ##     else:
        ##         self.replaceContextData(selected_template, key, tool_data)


class PromptTab(QWidget):
    def __init__(self, parent=None, env=None):
        super(PromptTab, self).__init__(parent)
        self.wgt = parent
        self.is_enabled = False
        layout = self.genLayout(parent)
        self.setLayout(layout)
        self.data = self.load_data()
        self._tool.setSubmitHandler(self.submitPrompt)

    def genLayout(self, parent):
        layout = QVBoxLayout()
        self._table = PromptTable(parent)
        self._tool = PromptTool()
        layout.addWidget(self._table)
        layout.addWidget(self._tool)
        return layout

    def setToolEnabled(self, state):
        pass

    def load_data(self):
        schemas = plg_utils.get_json('data/prompt.json')
        self._tool.loadData(schemas)

    def submitPrompt(self):
        tool_data = self._tool.getValues()
        comp_data = {
            'temp': tool_data['temp'],
            'text': tool_data['comp']
        }

        table_row = ((*comp_data.values(),))
        self._table.addRow(table_row)

        comp_hash = plg_utils.get_dict_sha1_trunc(comp_data)
        self.wgt.bank['regs']['prompt'][comp_hash] = (*comp_data.values(),)

        # recompilation
        bag_key = self.wgt.bags['quiz']
        if not bag_key:
            bag_key = plg_utils.get_str_list_sha1_trunc([comp_hash])
            self.wgt.bank['bags']['prompt'][bag_key].add(comp_hash)
        else:
            regs = self.wgt.bank['bags']['prompt'][bag_key]
            self.wgt.bank['bags']['prompt'].pop(bag_key)
            regs.add(comp_hash)
            bag_key = plg_utils.get_str_list_sha1_trunc(regs)
            self.wgt.bank['bags']['prompt'][bag_key] = regs

        for func_addr in self.wgt.pick:
            self.wgt.gui.ui.taskView.model().link['funcs'][func_addr]['links']['quiz'] = bag_key

        self.wgt.bags['quiz'] = bag_key


        # self.job_data['funcs'][self.ctx_func['addr']]['prompt'][data['temp']] = data['comp']

        ## key = data.pop('temp')
        ## selected_template = self.ui.promptWidget.tabPrompt._tool.selected_template
        ## if key != selected_template:
        ##     if self.ui.promptWidget.tabPrompt._tool.selected_template == qt_widgets.CustomCombo.NEW_LABEL[1:-1]:
        ##         self.insertPromptData(key, data)
        ##     else:
        ##         self.replacePromptData(selected_template, key, data)


class ResultsTab(QWidget):
    def __init__(self, parent=None, env=None):
        super(ResultsTab, self).__init__(parent)
        self.wgt = parent
        self.is_enabled = False
        layout = self.genLayout(parent)
        self.setLayout(layout)
        self.loadPromptData()

    def genLayout(self, parent):
        layout = QVBoxLayout()
        self._screen = ResultsTree(parent)
        layout.addWidget(self._screen)
        # Tool is empty.
        self._tool = ResultsTool()
        layout.addWidget(self._tool)
        return layout

    def updateScreen(self, res):
        # self._screen.clear()
        root = self._screen.invisibleRootItem()
        tmp, txt, svc = res

        parent_item = None
        for i in range(root.childCount()):
            existing_item = root.child(i)
            if existing_item.text(0) == tmp:
                parent_item = existing_item
                break

        # If no parent item is found, create a new one
        if parent_item is None:
            parent_item = QTreeWidgetItem([tmp, ""])
            root.addChild(parent_item)

        # Check if the child with text 'svc' already exists
        child_item = None
        for i in range(parent_item.childCount()):
            existing_child = parent_item.child(i)
            if existing_child.text(0) == svc:
                child_item = existing_child
                break

        if child_item is None:
            # If no child exists, create a new one
            child_item = QTreeWidgetItem([svc, txt])
            parent_item.addChild(child_item)
        else:
            # If the child exists, update the text if necessary
            if child_item.text(1) != txt:
                child_item.setText(1, txt)

        for i in range(self._screen.topLevelItemCount()):
            parent_item = self._screen.topLevelItem(i)
            for j in range(parent_item.childCount()):  # Iterate through first-level children
                child_item = parent_item.child(j)
                # Set tooltip for the second column (index 1)
                child_item.setToolTip(1, child_item.text(1))


    def setToolEnabled(self, state):
        self.is_enabled = state

    def loadPromptData(self):
        lnk_data = plg_utils.get_json('data/linker.json')
        ppt_data = plg_utils.get_json('data/prompt.json')
        self._screen.loadData(ppt_data, lnk_data)


class AutoResizeTextEdit(QTextEdit):
    def __init__(self, owner, wgt, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.msg_owner = owner
        self.wgt = wgt
        # Super critical to have styles defined here.
        self.setProperty('class', '{}-prompt-v'.format(owner))

        # Configure QTextEdit
        self.setReadOnly(True)
        self.setAcceptRichText(False)
        self.setLineWrapColumnOrWidth(100)
        self.setVerticalScrollBarPolicy(Qt.ScrollBarAlwaysOff)
        self.installEventFilter(self)

        # Connect the document's size change signal to adjust the height.
        # While the idea was good, it messed up message positioning.
        # self.document().documentLayout().documentSizeChanged.connect(self.adjustHeight)

        # Create a QLabel for the icon
        if self.msg_owner == 'lm':
            self.icon_label = QLabel(self)
            self.icon_pixmap = QPixmap(':/idaxlm/icon_send.png')
            self.icon_size = 24  # Size of the icon (width and height)
            self.icon_label.setPixmap(self.icon_pixmap.scaled(self.icon_size, self.icon_size, Qt.KeepAspectRatio, Qt.SmoothTransformation))
            self.icon_label.setFixedSize(self.icon_size, self.icon_size)
            self.icon_label.setCursor(QCursor(Qt.PointingHandCursor))  # Set cursor to pointing hand
            self.update_icon_position()
            self.icon_label.mousePressEvent = self.handle_icon_click

    def setMdContent(self, md_content):
        prompt_html = plg_utils.render_md(md_content)
        self.md_content = md_content
        self.setHtmlContent(prompt_html)

    def setHtmlContent(self, html_content):
        self.blockSignals(True)
        self.setHtml(html_content)
        self.blockSignals(False)
        self.adjustHeight()
        # self._applyStyle()

    def getTempHeight(self):
        doc_size = self.document().size()
        doc_height = int(doc_size.height())
        padding = 48
        height = doc_height + padding
        return height

    def adjustHeight(self):
        # Calculate the required height based on the document's size.
        doc_size = self.document().size()
        doc_height = int(doc_size.height())
        padding = 34
        # Set the fixed height of the QTextEdit
        self.setFixedHeight(self.getTempHeight())  # + padding + self.frameWidth() * 2

    def sizeHint(self):
        doc_size = self.document().size()
        doc_height = int(doc_size.height())
        return QSize(super().sizeHint().width(), self.height())

    def _applyStyle(self):
        # if self._pending_style_class:
        # self.setProperty('class', 'self-prompt')  # self._pending_style_class
        self.style().unpolish(self)
        self.style().polish(self)

    def update_icon_position(self):
        # Position the icon in the top-right corner of the QTextEdit
        owner_pad = 0 if self.msg_owner == 'me' else 62
        corner_pad = 18
        self.icon_label.move(self.width() - self.icon_size - corner_pad - owner_pad, corner_pad)

    def resizeEvent(self, event):
        super().resizeEvent(event)
        if self.msg_owner == 'lm':
            self.update_icon_position()

    def handle_icon_click(self, event):
        if event.button() == Qt.LeftButton:

            xlm_head, xlm_text = self.md_content.split('\n\n', 1)
            xlm_name = xlm_head.replace('`', '')

            ida_head = "IDAxLM/{}".format(xlm_name)
            ida_code = plg_utils.format_code(xlm_text)

            ida_addr = self.wgt.pick[0]
            ida_cmnt = ida_shims.get_func_cmt(ida_addr, 0)

            xlm_cmnt = "---- {} ----\n\n{}\n----".format(ida_head, ida_code)

            if ida_head in ida_cmnt:
                # Replace the comment from the same service if it already exists
                ida_cmnt = re.sub(
                    r"---- {} ----.*?----".format(ida_head),
                    xlm_cmnt,
                    ida_cmnt,
                    flags=re.DOTALL)
            else:
                ida_cmnt = f"{ida_cmnt}\n{xlm_cmnt}".strip()

            ida_shims.set_func_cmt(ida_addr, ida_cmnt, 0)

            ida_shims.jumpto(ida_addr)
            ida_utils.refresh_ui()

    def eventFilter(self, obj, event):
        if event.type() == QEvent.ContextMenu and obj == self:
            margin_area = self.get_margin_area()
            if not margin_area.contains(event.pos()):
                self.show_context_menu(event.globalPos())
                return True  # Consume the event
        return super().eventFilter(obj, event)

    def get_margin_area(self):
        full_rect = self.rect()
        content_rect = self.contentsRect()
        margin_top = content_rect.top()
        margin_left = content_rect.left()
        margin_right = full_rect.right() - content_rect.right()
        margin_bottom = full_rect.bottom() - content_rect.bottom()

        return QRect(
            margin_left, margin_top, full_rect.width() - margin_right, full_rect.height() - margin_bottom
        )

    def show_context_menu(self, global_pos):
        # Find the parent QListWidget and the corresponding QListWidgetItem
        # parent_list = self.parent().parent()  # Assuming hierarchy: ChatList/QListWidget -> QWidget -> QTextEdit
        # item_index = parent_list.indexAt(self.parent().pos())
        # item = parent_list.item(item_index.row())  # <PyQt5.QtWidgets.QListWidgetItem object at 0x0000012C256476D0>
        # print("row: ", item_index.row(), self.pos())

        to_enable = self.property('class').endswith('-x')
        # to_enable = not bool(item.flags() & Qt.ItemIsEnabled)

        # Create the context menu
        context_menu = QMenu()
        action = QAction(['Disable', 'Enable'][int(to_enable)], self)
        # action.setEnabled(to_enable)  # Enable/disable based on item state
        context_menu.addAction(action)
        action.triggered.connect(lambda: self.set_item_flags(None, to_enable))

        # Show the context menu
        context_menu.exec(global_pos)

    def set_item_flags(self, item, to_enable):
        current_class = self.property('class')
        if to_enable:
            current_class = current_class.replace('-x', '-v')
        else:
            current_class = current_class.replace('-v', '-x')
        self.setProperty('class', current_class)

        # self.setProperty('class', class_name)
        # Trigger a re-evaluation of the stylesheet
        self.style().unpolish(self)
        self.style().polish(self)
        self.update()  # Force a repaint

        # if to_enable:
        #     item.setFlags(item.flags() | Qt.ItemIsEnabled)
        # else:
        #     item.setFlags(item.flags() & ~Qt.ItemIsEnabled)


class ChatTab(QWidget):
    def __init__(self, parent=None, env=None):
        super(ChatTab, self).__init__(parent)
        self.wgt = parent
        self.is_enabled = False
        layout = self.genLayout(parent)
        self.setLayout(layout)
        self._tool._submit.clicked.connect(self.message_to)
        self._tool._chatPrompt.textChanged.connect(self.enableButton)

    def genLayout(self, parent):
        layout = QVBoxLayout()
        self.table = ChatList()  # parent
        self.table.setVerticalScrollMode(QListWidget.ScrollPerPixel)
        # self.table = QListWidget()
        self.table.setSizePolicy(QSizePolicy.Expanding, QSizePolicy.Expanding)
        self.table.setUniformItemSizes(False)
        layout.addWidget(self.table, 4)
        self._tool = ChatTool(parent)
        self._tool.setSizePolicy(QSizePolicy.Expanding, QSizePolicy.Minimum)
        # smaller ratio will produce larger gap between the `table` and `tool`
        layout.addWidget(self._tool, 1)
        return layout

    def messageSend(self, owner, markup):
        text_edit = AutoResizeTextEdit(owner, self.wgt)
        text_edit.setMdContent(markup)

        list_item = QListWidgetItem(self.table)
        list_item.setFlags(Qt.NoItemFlags)
        # list_item.setFlags(list_item.flags() | Qt.ItemIsEnabled)
        list_item.setSizeHint(text_edit.sizeHint())

        self.table.addItem(list_item)
        self.table.setItemWidget(list_item, text_edit)
        self.table.updateGeometry()
        self.table.update()

    def message_to(self):
        prompt_text = self._tool._chatPrompt.toPlainText()
        history = self.get_chat_history()

        self._tool._chatPrompt.update_suggestions(prompt_text, is_dump=True)
        self._tool._chatPrompt.clear()
        # prompt_html = plg_utils.render_md(prompt_text)

        pat_func = r"<func:(0x[0-9A-Fa-f]+|screen|\w+)>"
        mat_func = re.findall(pat_func, prompt_text)

        for mat in mat_func:
            func_ref = mat
            func_ea = None
            if func_ref.startswith('0x'):
                func_ea = int(func_ref, 16)
            elif func_ref == 'screen':
                func_ea = ida_shims.get_screen_ea()
            else:
                func_ea = ida_shims.get_name_ea_simple(func_ref)

            func_dsc = idaapi.get_func(func_ea)
            if func_dsc:
                func_def = str(idaapi.decompile(ida_shims.start_ea(func_dsc)))
                lbl_func = "<func:{}>".format(func_ref)
                prompt_text = prompt_text.replace(lbl_func, "\n```cpp\n{}\n```\n".format(func_def))

        pat_type = r"<type:([a-zA-Z0-9_-]+)>"
        mat_type = re.findall(pat_type, prompt_text)

        for mat in mat_type:
            type_ref = mat
            type_def = ida_utils.get_type_def(type_ref)
            if type_def:
                lbl_type = "<type:{}>".format(type_ref)
                prompt_text = prompt_text.replace(lbl_type, "\n```cpp\n{}\n```\n".format(type_def))

        func_addr = self.wgt.pick[0]
        # if not self.wgt.job_data['funcs'][func_addr]['results']: ??
        func_text = None
        if len(history) == 1:
            code_text = lm_utils.strip_comments(str(idaapi.decompile(func_addr)))
            func_text = "Consider the following C function: \n\n```cpp\n{}\n```\n{}".format(code_text, prompt_text)  # cfuncptr_t
        else:
            func_text = prompt_text

        self.messageSend('me', func_text)  # prompt_text
        history.append({"role": "user", "content": func_text})

        session = []
        session.append({
            'owner': 'me',
            'engine': 'human',
            'text': func_text
        })

        for name in self.wgt.services:
            pvd_name, mdl_name = name.split('/')
            dialer = lm_utils.LmDialer()
            dialer.set_data(pvd_name, mdl_name)
            # func_wrap = self.assemblePrompt(func_text, func_desc['context'])
            gpt_response = dialer.postData(history, is_dry=False)

            gpt_response = "`{}`\n\n".format(name) + gpt_response

            self.messageSend('lm', gpt_response)

            session.append({
                'owner': 'lm',
                'engine': name,
                'text': gpt_response
            })

        reg_key = plg_utils.get_dict_sha1_trunc(session)
        self.wgt.bank['regs']['chat'][reg_key] = session

        bag_key = self.wgt.bags['chat']

        if not bag_key:
            bag_key = plg_utils.get_str_list_sha1_trunc([reg_key])
            self.wgt.bank['bags']['chat'][bag_key].add(reg_key)
        else:
            regs = self.wgt.bank['bags']['chat'][bag_key]
            self.wgt.bank['bags']['chat'].pop(bag_key)
            regs.add(reg_key)
            bag_key = plg_utils.get_str_list_sha1_trunc(regs)
            self.wgt.bank['bags']['chat'][bag_key] = regs

        for func_addr in self.wgt.pick:
            self.wgt.gui.ui.taskView.model().link['funcs'][func_addr]['links']['chat'] = bag_key

        self.wgt.bags['chat'] = bag_key


    def enableButton(self):
        set_enabled = self.is_enabled and self._tool._chatPrompt.toPlainText() != ""
        self._tool._submit.setEnabled(set_enabled)
    #  def message_from(self):
    #      self.model.add_message(USER_THEM, self.message_input.text())

    def setToolEnabled(self, state):
        self.is_enabled = state
        if state:
            if self._tool._chatPrompt.toPlainText() != "":
                self._tool._submit.setEnabled(True)
        else:
            self._tool._submit.setEnabled(False)

    def get_chat_history(self):
        history = [
            {"role": "system", "content": "You are a helpful reverse-engineering and programming assistant."}
        ]
        for row in range(self.table.count()):
            list_item = self.table.item(row)  # Get the QListWidgetItem
            text_edit = self.table.itemWidget(list_item)  # Get the widget (QTextEdit)
            if isinstance(text_edit, QTextEdit):  # Ensure it's a QTextEdit
                class_name = text_edit.property('class')
                if class_name.endswith('-x'):
                    continue
                is_user = 'me-prompt' in class_name
                role = "user" if is_user else "assistant"
                history.append({"role": role, "content": text_edit.toPlainText()})

        return history


class GhostTab(QWidget):
    def __init__(self, parent=None, env=None):
        super(GhostTab, self).__init__(parent)
        self.wgt = parent
        self.env = env


class CustomHeader(QHeaderView):
    def __init__(self, orientation, parent=None):
        super().__init__(orientation, parent)
        self.setSectionsClickable(False)
        self.setHighlightSections(False)  # Disable highlighting of headers


class ReadOnlyWidgetBase(QWidget):
    def __init__(self, *args, **kwargs):
        pass
        # This class is a base class and should not be instantiated directly.
        # It is meant to be subclassed (!).
        # super().__init__(*args, **kwargs)

    def paintEmptyMessage(self, event):
        painter = QPainter(self.viewport())
        painter.setPen(QColor(150, 150, 150))

        # Center the text in the lsit/table/tree.
        rect = self.viewport().rect()
        text = "{} data will appear here".format(self.contents)
        painter.drawText(rect, Qt.AlignCenter, text)

    def edit(self, index, trigger, event):
        # Disable editing completely.
        return False

class ReadOnlyTableWidget(QTableWidget, ReadOnlyWidgetBase):
    def __init__(self, *args, **kwargs):
        QTableWidget.__init__(self, *args, **kwargs)
        ReadOnlyWidgetBase.__init__(self, *args, **kwargs)

    def paintEvent(self, event):
        super().paintEvent(event)
        if self.rowCount() == 0:  # Check if the table is empty
            self.paintEmptyMessage(event)

class ReadOnlyListWidget(QListWidget, ReadOnlyWidgetBase):
    def __init__(self, *args, **kwargs):
        QListWidget.__init__(self, *args, **kwargs)
        ReadOnlyWidgetBase.__init__(self, *args, **kwargs)

    def paintEvent(self, event):
        super().paintEvent(event)
        if self.count() == 0:  # Check if the list is empty
            self.paintEmptyMessage(event)

class ReadOnlyTreeWidget(QTreeWidget, ReadOnlyWidgetBase):
    def __init__(self, *args, **kwargs):
        QTreeWidget.__init__(self, *args, **kwargs)
        ReadOnlyWidgetBase.__init__(self, *args, **kwargs)

    def paintEvent(self, event):
        super().paintEvent(event)
        if self.topLevelItemCount() == 0:  # Check if the tree is empty
            self.paintEmptyMessage(event)

class PromptTable(ReadOnlyTableWidget):
    def __init__(self, parent=None):
        super(PromptTable, self).__init__(parent)
        self.contents = "prompt"
        self.setColumnCount(2)
        self.setHorizontalHeaderLabels(["Template", "Assembled Text"])
        self.setSelectionBehavior(QTableWidget.SelectRows)
        self.installEventFilter(self)
        self.setHorizontalHeader(CustomHeader(Qt.Horizontal))
        self.setVerticalHeader(CustomHeader(Qt.Vertical))
        # self.horizontalHeader().setSectionResizeMode(QHeaderView.Stretch)
        self.horizontalHeader().setMinimumSectionSize(300)
        self.horizontalHeader().setSectionResizeMode(0, QHeaderView.ResizeToContents)
        self.horizontalHeader().setSectionResizeMode(1, QHeaderView.Stretch)

    def eventFilter(self, source, event):
        # self.paintEvent(event)
        if source is self and event.type() == QEvent.KeyPress and event.key() == Qt.Key_Delete:
            self.delete_selected_row()
            return True  # Event has been handled
        return super().eventFilter(source, event)

    def delete_selected_row(self):
        # Get the indices of the selected rows
        selected_rows = self.selectionModel().selectedRows()

        for index in selected_rows:
            self.removeRow(index.row())

    def is_unique_in_first_column(self, value):
        """Check if the given value is unique in the first column."""
        for row in range(self.rowCount()):
            if self.item(row, 0) and self.item(row, 0).text() == value:
                return False
        return True

    def addRow(self, data):
        if self.is_unique_in_first_column(data[0]):
            row_count = self.rowCount()
            self.insertRow(row_count)

            for col in range(len(data)):
                item = QTableWidgetItem(data[col])
                self.setItem(row_count, col, item)

class MessageModel(QAbstractListModel):
    def __init__(self, *args, **kwargs):
        super(MessageModel, self).__init__(*args, **kwargs)
        self.messages = []

    def data(self, index, role):
        if role == Qt.DisplayRole:
            # Here we pass the delegate the user, message tuple.
            return self.messages[index.row()]

    def rowCount(self, index):
        return len(self.messages)

    def add_message(self, who, text):
        """
        Add an message to our message list, getting the text from the QLineEdit
        """
        if text:  # Don't add empty strings.
            # Access the list via the model.
            self.messages.append((who, text))
            # Trigger refresh.
            self.layoutChanged.emit()

class ChatList(ReadOnlyListWidget):
    def __init__(self, parent=None):
        super(ChatList, self).__init__(parent)
        self.contents = "chat"


class PaddedItemDelegate(QStyledItemDelegate):
    def __init__(self, parent, col_width, col_margin=100):
        super().__init__(parent)
        self.col_width = col_width
        self.col_margin = col_margin  # Set the amount of spacing on the right

    def paint(self, painter, option, index):
        if index.column() == 1:
            rect = QRect(option.rect)
            rect.setWidth(self.col_width)

            # QApplication.palette().color(QPalette.Highlight)
            # option.palette.color(QPalette.Highlight)
            # option.palette.highlight().color()
            color_over_focus_v = QColor("#e5f3ff")
            color_over_focus_x = QColor("#cce8ff")
            color_sel_focus_x = QColor("#d9d9d9")
            color_sel_focus_v = QColor("#cde8ff")

            # Highlight the full item background, including hover effect
            if option.state & QStyle.State_Selected:
                if option.state & QStyle.State_Active:
                    painter.fillRect(rect, color_sel_focus_v)
                    if option.state & QStyle.State_MouseOver:
                        pen = QPen(QColor("#99d1ff"), 1)  # Blue border, 2px wide
                        painter.setPen(pen)
                        painter.drawLine(rect.topLeft(), rect.topRight())    # Top border
                        painter.drawLine(rect.topRight(), rect.bottomRight())  # Right border
                        painter.drawLine(rect.bottomRight(), rect.bottomLeft())  # Bottom border
                else:
                    if option.state & QStyle.State_MouseOver:
                        painter.fillRect(rect, color_over_focus_x)

                        # Draw a blue border around the selected item
                        pen = QPen(QColor("#99d1ff"), 1)  # Blue border, 2px wide
                        painter.setPen(pen)
                        painter.drawLine(rect.topLeft(), rect.topRight())    # Top border
                        painter.drawLine(rect.topRight(), rect.bottomRight())  # Right border
                        painter.drawLine(rect.bottomRight(), rect.bottomLeft())  # Bottom border

                    else:
                        painter.fillRect(rect, color_sel_focus_x)
            elif option.state & QStyle.State_MouseOver:
                if option.state & QStyle.State_Active:
                    painter.fillRect(rect, color_over_focus_v)
                else:
                    painter.fillRect(rect, color_over_focus_v)

        super().paint(painter, option, index)

    def initStyleOption(self, option, index):
        super().initStyleOption(option, index)
        # Reduce the drawable area -
        # Add right padding by adjusting the display rect
        if index.column() == 1:
            reduced_rect = option.rect  # QRect()
            reduced_rect.setRight(reduced_rect.right() - self.col_margin)
            # option.rect = reduced_rect

            option.state &= ~QStyle.State_Selected
            option.state &= ~QStyle.State_MouseOver


# class CustomDelegate(QStyledItemDelegate):
#     def __init__(self, margin=10, parent=None):
#         super().__init__(parent)
#         self.margin = margin

#     def paint(self, painter, option, index):
#         # Adjust the rectangle for margins
#         option.rect = option.rect.adjusted(self.margin, 0, -self.margin, 0)
#         super().paint(painter, option, index)


# class CustomDelegate(QStyledItemDelegate):
#     def __init__(self, margin=10, parent=None):
#         super().__init__(parent)
#         self.margin = margin
#
#     def paint(self, painter, option, index):
#         # Save the original rectangle for selection and margins
#         original_rect = option.rect
#
#         # Paint the selection rectangle (full row, including margins)
#         if option.state & QStyle.State_Selected:
#             painter.save()
#             painter.setBrush(option.palette.highlight())
#             painter.setPen(Qt.NoPen)
#             painter.drawRect(original_rect)
#             painter.restore()
#
#         # Adjust the rectangle for painting content
#         option.rect = option.rect.adjusted(0, 0, -self.margin, 0)
#
#         # Paint the content (text, icons, etc.) within the adjusted rectangle
#         super().paint(painter, option, index)

class MyDelegate(QStyledItemDelegate):
    def initStyleOption(self, opt, index):
        super().initStyleOption(opt, index)
        if (opt.state & QStyle.State_Selected
            and not opt.state & QStyle.State_Active
            and index.data(Qt.BackgroundRole) is not None
        ):
            opt.state &= ~QStyle.State_Selected


class ResultsTree(ReadOnlyTreeWidget):
    def __init__(self, parent=None):
        super(ResultsTree, self).__init__(parent)
        self.wgt = parent
        self.contents = "results"
        self.data = defaultdict(dict)
        self.setColumnCount(2)
        self.setAlternatingRowColors(False)
        # self.setColumnWidth(1, 320)
        self.adjustHeader()
        self.setContextMenuPolicy(Qt.CustomContextMenu)
        self.customContextMenuRequested.connect(self.showContextMenu)
        # self.setMouseTracking(True)
        # self.mouseMoveEvent = self.mouse_move_event
        # delegate = PaddedItemDelegate()
        # self.setItemDelegate(delegate)

        self.setItemDelegate(PaddedItemDelegate(self, self.columnWidth(1)))
        self.retranslateUi()

    def adjustHeader(self):
        header = self.header()
        header.setMinimumSectionSize(300)
        header.setDefaultAlignment(Qt.AlignCenter)
        header.setSectionResizeMode(0, QHeaderView.ResizeToContents)
        header.setSectionResizeMode(1, QHeaderView.Stretch)  # QHeaderView.Fixed

    def showContextMenu(self, position):
        item = self.itemAt(position)
        if item is not None and item.parent():
            menu = QMenu(self)  # Setting parent propagates the font style to context menu as well
            tmp_temp = item.parent().text(0)
            svc_name = item.text(0)
            svc_text = item.text(1)

            if tmp_temp is not None:
                for lnk_name, lnk_head in self.data[tmp_temp].items():
                    apply_action = menu.addAction(lnk_head)
                    apply_action.triggered.connect(lambda _, lnk_name=lnk_name: self.applyAction(lnk_name, svc_name, svc_text))

            menu.exec_(self.viewport().mapToGlobal(position))

    def applyAction(self, lnk_name, svc_name, svc_text):
        f_addr = self.wgt.pick[0]
        module_path = 'linker'
        module = plg_utils.import_path_simple(module_path)

        linker_func = getattr(module, lnk_name)
        linker_func(f_addr, svc_text, svc_name)  # self.env_desc,

    def loadData(self, prompt_data, linker_data):
        for ppt_temp, ppt_desc in prompt_data.items():
            for lnk_func in ppt_desc['link']:
                self.data[ppt_temp][lnk_func] = linker_data[lnk_func]

    def retranslateUi(self):
        self.setHeaderLabels([i18n("Template"), i18n("Response Text")])

    def resizeEvent(self, event):
        super().resizeEvent(event)
        data_col_width = self.header().sectionSize(1)  # self.columnWidth(1)
        # Update the delegate with the new column width
        delegate = self.itemDelegate()
        if isinstance(delegate, PaddedItemDelegate):
            delegate.col_width = data_col_width

        super().resizeEvent(event)


class ContextTable(ReadOnlyTableWidget):
    def __init__(self, parent=None):
        super(ContextTable, self).__init__(parent)
        self.gui = parent
        self.contents = "context"
        self.setColumnCount(2)
        self.setHorizontalHeaderLabels(["Template", "Assembled Text"])
        self.setSelectionBehavior(QTableWidget.SelectRows)
        self.installEventFilter(self)
        self.setHorizontalHeader(CustomHeader(Qt.Horizontal))
        self.setVerticalHeader(CustomHeader(Qt.Vertical))
        # self.horizontalHeader().setSectionResizeMode(QHeaderView.ResizeToContents)  # QHeaderView.Stretch
        self.horizontalHeader().setMinimumSectionSize(300)
        self.horizontalHeader().setSectionResizeMode(0, QHeaderView.ResizeToContents)
        self.horizontalHeader().setSectionResizeMode(1, QHeaderView.Stretch)

    def eventFilter(self, source, event):
        # self.paintEvent(event)
        if source is self and event.type() == QEvent.KeyPress and event.key() == Qt.Key_Delete:
            self.delete_selected_ctx()
            self.delete_selected_row()
            return True  # Event has been handled
        return super().eventFilter(source, event)

    def delete_selected_ctx(self):
        ctx_addr = self.gui.ctx_func['addr']

        selected_rows = self.selectionModel().selectedRows()

        for index in selected_rows:
            row = index.row()
            item = self.item(row, 0)  # Get the item in the first column of the selected row
            if item:  # Make sure the item exists
                template_name = item.text()
                self.gui.job_data['funcs'][ctx_addr]['context'].pop(template_name)

    def delete_selected_row(self):
        # Get the indices of the selected rows
        selected_rows = self.selectionModel().selectedRows()

        for index in selected_rows:
            self.removeRow(index.row())

    def is_unique_in_first_column(self, value):
        """Check if the given value is unique in the first column."""
        for row in range(self.rowCount()):
            if self.item(row, 0) and self.item(row, 0).text() == value:
                return False
        return True

    def addRow(self, data):
        if self.is_unique_in_first_column(data[0]):
            row_count = self.rowCount()
            self.insertRow(row_count)

            for col in range(len(data)):
                item = QTableWidgetItem(data[col])
                self.setItem(row_count, col, item)



class CustomSpacer(QSpacerItem):
    def __init__(self, width=40):
        super().__init__(width, 26, QSizePolicy.Fixed, QSizePolicy.Minimum)

class CustomVBoxLayout(QVBoxLayout):
    def __init__(self, h=0, v=0, s=0, parent=None):
        super().__init__(parent)
        self.setContentsMargins(h, v, h, v)
        self.setSpacing(s)

class CustomHBoxLayout(QHBoxLayout):
    def __init__(self, h=0, v=0, s=0, parent=None):
        super().__init__(parent)
        self.setContentsMargins(h, v, h, v)
        self.setSpacing(s)

class HeadButton(QPushButton):
    def __init__(self, parent=None):
        super().__init__(parent)
        self.setMinimumSize(QSize(200, 30))
        self.setCursor(QCursor(Qt.ArrowCursor))
        self.setProperty("class", "head")

    def lookActive(self):
        self.setCursor(QCursor(Qt.PointingHandCursor))

    def lookInactive(self):
        self.setCursor(QCursor(Qt.ArrowCursor))

class HorizontalSeparator(QWidget):
    def __init__(self, parent=None):
        super().__init__(parent)

        layout = CustomHBoxLayout()
        layout.addItem(CustomSpacer(20))

        separator = QFrame()
        separator.setFrameShape(QFrame.HLine)
        separator.setFrameShadow(QFrame.Sunken)
        layout.addWidget(separator)

        layout.addItem(CustomSpacer(20))
        self.setLayout(layout)

class ProgressButton(QWidget):
    clicked = Signal()

    def __init__(self, parent=None):
        super().__init__(parent)

        layout = CustomVBoxLayout()
        self._bar = ProgressIndicator()
        btn = QPushButton()
        btn.setMinimumSize(QSize(200, 30))
        btn.setCheckable(False)
        btn.setCursor(QCursor(Qt.PointingHandCursor))
        btn.setEnabled(False)
        self._btn = btn
        layout.addWidget(self._bar)
        layout.addWidget(self._btn)
        self.setLayout(layout)

        self._btn.clicked.connect(self.clicked.emit)

    def setEnabled(self, flag):
        self._btn.setEnabled(flag)

    def setText(self, text):
        self._btn.setText(text)

    def updateProgress(self, state):
        self._bar.updateProgress(state)


class ProviderTool(QWidget):
    svcUpdate = Signal(object)

    def __init__(self, name, env_desc, parent=None):
        super(ProviderTool, self).__init__(parent)
        self.gui = parent
        self.env_desc = env_desc
        self.is_stats_shown = False
        self.is_calc = False
        self.is_free_tier = True
        layout = self.initLayout()
        self.setLayout(layout)
        self.retranslateUi()
        self.bindComponents()
        self.dial = lm_utils.LmDialer()
        self.configure(self.dial.cfg_data)

    def bindComponents(self):
        self._cwSelect._select.model().dataChanged.connect(self.updateUi)
        self._swButton.clicked.connect(self.sendTasks)

    def assemblePrompt(self, fn_c, fn_i, fn_q):
        fn_m = plg_utils.get_fenced(plg_utils.strip_comments(fn_c))
        pt_t = "Analyze the following C function: {}".format(fn_m)
        pt_t += " ".join(fn_i)
        pt_t += "\n{}".format(fn_q)
        return pt_t

    def sendTasks(self):
        """
        Assembles contexts and queries into prompts.
        Sends prompts to service endpoints.

        svc_name: service name
        pvd_name: provider name
        mdl_name: model name
        """

        pick = self.gui.ui.taskView.model().pick
        link = self.gui.ui.taskView.model().link
        bank = self.gui.ui.promptWidget.bank
        bags = self.gui.ui.promptWidget.bags

        svc_names = self._cwSelect.getData()
        is_svc_issue = False
        for svc_name in svc_names:
            pvd_name, mdl_name = svc_name.split('/')
            svc_data = self.cfg_desc['providers'][pvd_name]
            dialer = lm_utils.LmDialer()  # svc_data
            dialer.set_data(pvd_name, mdl_name)
            for tid, dsc in link['tasks'].items():
                for func_addr in dsc['order']:
                    func_desc = link['funcs'][func_addr]  # tasks, views, links
                    # if not func_desc['pseudo']:
                    #     func_desc['pseudo'] = str(idaapi.decompile(func_addr))
                    pseudo = str(idaapi.decompile(func_addr))

                    base_bag_ref = func_desc['links']['base']
                    base_reg_refs = bank['bags']['context'][base_bag_ref]

                    quiz_bag_ref = func_desc['links']['quiz']
                    quiz_reg_refs = bank['bags']['prompt'][quiz_bag_ref]

                    context = []
                    for ref in base_reg_refs:
                        ctx_temp, ctx_text = bank['regs']['context'][ref]
                        context.append(ctx_text)

                    reg_keys = []
                    for ref in quiz_reg_refs:
                        qry_temp, qry_text = bank['regs']['prompt'][ref]
                        func_cmpl = self.assemblePrompt(pseudo, context, qry_text)

                        llm_res = dialer.postData(func_cmpl, is_dry=False)
                        if 'error' in llm_res:
                            msg_text = plg_utils.get_error_text(llm_res['message'])
                            usr_res = QMessageBox.question(
                                None,
                                "Unexpected response from LM",
                                "{}: {}.\nExclude this service for this set of tasks?".format(svc_name, msg_text),
                                QMessageBox.Yes | QMessageBox.No
                            )
                            if usr_res == QMessageBox.Yes:
                                is_svc_issue = True
                                break

                        if llm_res[0] in ['{', '[']:
                            llm_res = json.dumps(json.loads(llm_res), separators=(',', ':'))

                        res = {
                            'tmp': qry_temp,
                            'txt': llm_res,
                            'svc': svc_name
                        }

                        # self._table.addRow((*res.values(),))
                        # self.gui.ui.promptWidget.tabResults.updateScreen((*res.values(),))

                        res_hash = plg_utils.get_dict_sha1_trunc(res)
                        bank['regs']['results'][res_hash] = (*res.values(),)

                        reg_keys.append(res_hash)

                    # ---
                    regs = []
                    bag_key = self.gui.ui.taskView.model().link['funcs'][func_addr]['links']['done']
                    if bag_key:
                        regs = bank['bags']['results'][bag_key]
                        bank['bags']['results'].pop(bag_key)

                    reg_keys.extend(regs)
                    bag_key = plg_utils.get_str_list_sha1_trunc(reg_keys)
                    bank['bags']['results'][bag_key] = reg_keys
                    self.gui.ui.taskView.model().link['funcs'][func_addr]['links']['done'] = bag_key

                    if len(list(pick['funcs'].keys())) and func_addr == list(pick['funcs'].keys())[0]:
                        bags['done'] = bag_key

                    # Break task function iteration.
                    if is_svc_issue == True:
                        break

                # Break task iteration.
                if is_svc_issue == True:
                    break
            # Skip service that produced an error.
            if is_svc_issue == True:
                is_svc_issue = False
                continue

        # switch to 'results' tab
        self.gui.ui.promptWidget.setCurrentIndex(2)
        self.gui.ui.promptWidget.onTabChanged(self.gui.ui.promptWidget.currentIndex())



    def showStats(self):
        if not self.is_stats_shown:
            self._swTable.show()
            self._swButton.clicked.disconnect(self.sendTasks)
            self._swButton.clicked.connect(self.calcTokens)
        else:
            self._swTable.hide()
            self._swButton.clicked.disconnect(self.calcTokens)
            self._swButton.clicked.connect(self.sendTasks)
            self._swButton.setEnabled(True)

        self.is_stats_shown = not self.is_stats_shown

        self.retranslateUi()

    def prepareTable(self, is_data):

        stat = defaultdict(lambda: {
            'tokens': 0,
            'price': 0
        })
        svc_set = self._cwSelect.getData()

        if not is_data:
            for i, s_name in enumerate(svc_set):
                stat[s_name]['tokens'] = "-"
                stat[s_name]['price'] = "-"
        else:
            func_reg = []
            job_data = self.gui.job_data
            for task_desc in job_data['tasks'].values():
                for func_addr, func_flag in task_desc['funcs'].items():
                    if (not func_flag == False
                        and func_addr not in func_reg
                        and not job_data['funcs'][func_addr]['pseudo']):
                        func_reg.append(func_addr)
                        job_data['funcs'][func_addr]['pseudo'] = str(idaapi.decompile(func_addr))
                        # self._swButton.updateProgress(int((len(func_reg) / job_data['stats']['calc']) * 100))

            for func_addr in func_reg:
                text_sum = ""
                text_sum += " ".join(job_data['funcs'][func_addr]['context'].values())
                text_sum += " ".join(job_data['funcs'][func_addr]['prompt'].values())

                for i, s_name in enumerate(svc_set):
                    p_name, m_name = s_name.split('/')

                    tokens = self.calcTokentCount(p_name, m_name, text_sum)
                    price = tokens / 1000000 * self.cfg_desc["providers"][p_name]['models'][m_name]['price_in']
                    stat[s_name]['tokens'] += tokens
                    stat[s_name]['price'] += price

        self._swTable.clearContents()
        self._swTable.setRowCount(len(svc_set))

        for row, svc in enumerate(svc_set):
            self._swTable.setItem(row, 0, QTableWidgetItem(svc))

            item1 = QTableWidgetItem(str(stat[svc]['tokens']))
            item1.setTextAlignment(Qt.AlignRight | Qt.AlignVCenter)
            self._swTable.setItem(row, 1, item1)

            a = plg_utils.f_float(stat[svc]['price'])
            item2 = QTableWidgetItem(a)
            item2.setTextAlignment(Qt.AlignRight | Qt.AlignVCenter)
            self._swTable.setItem(row, 2, item2)

    def initLayout(self):
        layout = CustomVBoxLayout()
        self._head = HeadButton()
        layout.addWidget(self._head)
        self._body = self.getBody()
        layout.addWidget(self._body)
        return layout

    def getStats(self):
        widget = QTableWidget()
        widget.setColumnCount(3)
        widget.setHorizontalHeaderLabels(["Provider", "Tokens", "Price"])
        widget.verticalHeader().setVisible(False)
        widget.horizontalHeader().setSectionResizeMode(QHeaderView.Stretch)
        widget.setSelectionBehavior(QTableWidget.SelectRows)
        widget.setShowGrid(False)
        widget.setSelectionMode(QTableWidget.NoSelection)  # Disable selection
        widget.setFocusPolicy(Qt.NoFocus)  # Disable focus
        # widget.horizontalHeader().setMinimumSectionSize(256)
        # widget.horizontalHeader().setSectionResizeMode(0, QHeaderView.ResizeToContents)
        # widget.horizontalHeader().setSectionResizeMode(1, QHeaderView.Stretch)
        first_column_idx = 0
        widget.horizontalHeader().setSectionResizeMode(first_column_idx, QHeaderView.Fixed)
        widget.horizontalHeader().resizeSection(first_column_idx, 240)

        widget.setStyleSheet(
            """
            QTableWidget {
                border: none;
                background-color: transparent;
            }
            QTableWidget::item {
                border: none;
                padding: 0px;
                margin: 0px;
            }
            """
        )

        return widget

    def getBody(self):
        frame = QGroupBox()
        self.vlFiltersGroup = CustomVBoxLayout(20, 15, 20, frame)
        self._swTable = self.getStats()
        self.vlFiltersGroup.addWidget(self._swTable)
        self._swTable.hide()
        self._cwSelect = FilterInputGroup(i18n("Providers"), i18n("Pick providers..."), self.env_desc)
        self.vlFiltersGroup.addWidget(self._cwSelect)
        self._swButton = ProgressButton()
        self.vlFiltersGroup.addWidget(self._swButton)
        return frame

    # def get_tk_data():
    #     return "summed up text generated on the base of the input provided by the user"
    # self.ui.wProviderTool.configure(data, get_tk_data)

    def configure(self, cfg_desc, get_fn=None):
        self.cfg_desc = cfg_desc
        self.get_fn = get_fn
        model_list = []

        for pk, pv in cfg_desc["providers"].items():
            for mk, mv in pv['models'].items():
                model_name = "{}/{}".format(pk, mk)
                model_list.append(model_name)
                if mv['price_in'] > 0.0 or mv['price_out'] > 0.0:
                    self.is_free_tier = False
        if self.is_free_tier:
            self._head.lookInactive()
        self._cwSelect.addItems(model_list, True)
        self._cwSelect.setText("")

    def calcTokentCount(self, p_name, m_name, text):
        # Token count can be calculated only for the known models.
        if p_name == 'mistral':
            from transformers import AutoTokenizer
            tokenizer = AutoTokenizer.from_pretrained(m_name)
            tokens = tokenizer.tokenize(text)
            return len(tokens)
        elif p_name == 'openai':
            import tiktoken
            tokenizer = tiktoken.encoding_for_model(m_name)
            tokens = tokenizer.encode(text)
            return len(tokens)

        print("the corresponding tokenizer was not found")
        return -1

    def initCalc(self):
        pass

    def updateUi(self):

        self.prepareTable(False)
        svc_set = self._cwSelect.getData()
        svc_len = len(svc_set)
        if svc_len:
            self._swButton.setEnabled(True)
            # if self.gui.ctx_func['addr']:  ??
            self.gui.ui.promptWidget.tabChat.setToolEnabled(True)
        else:
            self._swButton.setEnabled(False)
            self.gui.ui.promptWidget.tabChat.setToolEnabled(False)

        for i, pr in enumerate(svc_set):
            p_name, m_name = pr.split('/')
            if self.cfg_desc["providers"][p_name]['models'][m_name]['price_in'] > 0:
                self.is_calc = True
                self.makeCalcEnabled()
                break
        else:
            # Disable calc in case either there are no selected services
            # or if paid tier was not found among selected services.
            self.is_calc = False
            self.makeCalcDisabled()

        self.svcUpdate.emit(svc_set)

    def calcTokens(self):
        self.prepareTable(True)
        #TODO: total price

    def makeCalcEnabled(self):
        self._head.lookActive()
        self._head.clicked.connect(self.showStats)

    def makeCalcDisabled(self):
        self._head.lookInactive()
        self._head.clicked.disconnect(self.showStats)


    def setFuncCount(selfm):
        pass

    def retranslateUi(self):
        # self.func_label.setText(i18n("Functions:"))
        # self.task_label.setText(i18n("Tasks:"))
        # self.token_label.setText(i18n("Tokens:"))
        # self.price_label.setText(i18n("Price:"))
        # self.total_label.setText(i18n("Total Price $:"))
        if self.is_stats_shown:
            self._head.setText(i18n("Pricing"))
            self._swButton.setText(i18n("Calculate"))
        else:
            self._head.setText(i18n("Providers"))
            self._swButton.setText(i18n("Submit"))



class CustomSeparator(QWidget):
    def __init__(self, s=0, parent=None):
        super(CustomSeparator, self).__init__(parent)
        layout = self.genLayout(s)
        self.setLayout(layout)

    def genLayout(self, s):
        sep = QFrame()
        sep.setFrameShape(QFrame.VLine)
        sep.setFrameShadow(QFrame.Sunken)
        layout = QHBoxLayout()
        layout.setSpacing(0)
        layout.addItem(CustomSpacer(s))
        layout.addWidget(sep)
        layout.addItem(CustomSpacer(s))
        return layout


class LabelButton(QPushButton):
    def __init__(self, w=75, parent=None):
        super(LabelButton, self).__init__(parent)
        self.setMinimumSize(QSize(w, 30))
        self.setMaximumSize(QSize(w, 30))
        self.setCheckable(False)
        self.setAutoExclusive(False)


class ComboBoxWidget(QWidget):
    # itemOpSignal = Signal(QListWidgetItem)
    # selectSignal = Signal(QListWidgetItem)

    def __init__(self, text, is_editable):
        super().__init__()
        self.text = text
        self.is_editable = is_editable
        self.initUi()
        self.retranslateUi()
        self.decorateUi()

    def initUi(self):
        self.horizontalLayout = QHBoxLayout(self)
        self._text = QPushButton(self.text)

        sizePolicy = QSizePolicy(QSizePolicy.Minimum, QSizePolicy.Preferred)
        self._text.setSizePolicy(sizePolicy)
        # self._text.clicked.connect(lambda: self.selectSignal.emit(self.item))

        # self.btn_change = self.initSignalBtn(QIcon(':/idaxlm/icon_change.png'))
        self.btn_remove = self.initSignalBtn(QIcon(':/idaxlm/icon_remove.png'))

        # spacerItem = QSpacerItem(0, 0, QSizePolicy.Expanding, QSizePolicy.Minimum)

        self._text.setStyleSheet("text-align: left; padding-left: 10px;}")
        self.horizontalLayout.addWidget(self._text)
        # self.horizontalLayout.addItem(spacerItem)
        if self.is_editable:
            # self.horizontalLayout.addWidget(self.btn_change)
            self.horizontalLayout.addWidget(self.btn_remove)

    def initSignalBtn(self, icon):
        btn = QToolButton()
        btn.setIcon(icon)
        btn.setAutoRaise(True)
        # btn.clicked.connect(lambda: self.itemOpSignal.emit(handler))
        return btn

    def decorateUi(self):
        self._text.setProperty('class','select-text')

    def retranslateUi(self):
        # self.btn_change.setToolTip(i18n("Change"))
        self.btn_remove.setToolTip(i18n("Remove"))


class CustomCombo(QComboBox):
    NEW_LABEL = '<new_template>'
    itemRemoved = Signal(str)

    def __init__(self, parent=None):
        super(CustomCombo, self).__init__(parent)
        self.setObjectName("select-editable")
        self.listw = QListWidget(self)
        self.setModel(self.listw.model())
        self.setView(self.listw)
        self.currentIndexChanged.connect(self.onIndexChanged)

        sizePolicy = QSizePolicy(QSizePolicy.Expanding, QSizePolicy.Fixed)
        sizePolicy.setHorizontalStretch(0)
        sizePolicy.setVerticalStretch(0)
        sizePolicy.setHeightForWidth(self.sizePolicy().hasHeightForWidth())
        self.setSizePolicy(sizePolicy)
        self.setAcceptDrops(True)
        self.setEditable(True)
        self.setSizeAdjustPolicy(QComboBox.AdjustToMinimumContentsLength)
        self.view().setFixedWidth(300)

        # self.lineEdit().editingFinished.connect(self.on_editing_finished)
        # self.currentIndexChanged.connect(self.on_index_changed)

    def refreshInputs(self, items):
        self.clear()
        items.append(CustomCombo.NEW_LABEL)
        for num, text in enumerate(items):
            listwitem = QListWidgetItem(self.listw)
            # listwitem.setToolTip(text)
            is_editable = text != CustomCombo.NEW_LABEL
            itemWidget = ComboBoxWidget(text, is_editable)
            itemWidget._text.clicked.connect(lambda _, l=listwitem: self.handleButtonClick(l))
            itemWidget.btn_remove.clicked.connect(lambda _, l=listwitem: self.removeCombo(l))
            # itemWidget.itemOpSignal.connect(self.removeCombo)
            # itemWidget.selectSignal.connect(self.selectItem)

            if num % 2 == 0:
                listwitem.setBackground(QColor(255, 255, 255))
            else:
                listwitem.setBackground(QColor(237, 243, 254))
            listwitem.setSizeHint(itemWidget.sizeHint())
            listwitem.setFlags(listwitem.flags() | Qt.ItemIsSelectable | Qt.ItemIsEnabled)  # ??
            self.listw.addItem(listwitem)
            self.listw.setItemWidget(listwitem, itemWidget)

    def refreshBackColors(self):
        for row in range(self.view().count()):
            if row % 2 == 0:
                self.view().item(row).setBackground(QColor(255, 255, 255))
            else:
                self.view().item(row).setBackground(QColor(237, 243, 254))

    def onIndexChanged(self, index):
        # As the name states it is fired only(!) when index is changed.abs
        # If the same item was selected from the list the call will not be triggered.
        listWidgetItem = self.listw.item(index)
        retrieved_widget = self.listw.itemWidget(listWidgetItem)
        if retrieved_widget:  # !!
            widget_text = retrieved_widget._text.text()
            if widget_text == CustomCombo.NEW_LABEL:
                widget_text = widget_text[1:-1]
                # self.setEditable(True)
                # self.lineEdit().setFocus()
            # else:
                # self.setEditable(False)
            self.setCurrentText(widget_text)
            self.lineEdit().setText(widget_text)

    def handleButtonClick(self, item):
        view = self.view()
        index = view.indexFromItem(item)
        retrieved_widget = self.listw.itemWidget(item)
        widget_text = retrieved_widget._text.text()
        if widget_text == CustomCombo.NEW_LABEL:
            widget_text = widget_text[1:-1]
            # self.setEditable(True)
            # self.lineEdit().setFocus()
        # else:
            # self.setEditable(False)
        # self.setCurrentText(widget_text)
        # self.lineEdit().setText(widget_text)
        # self.setCurrentIndex(index.row())
        # current_index = self.currentIndex()
        # Close the popup to simulate re-selection
        self.hidePopup()

        # Re-select the item
        self.setCurrentIndex(index.row())

        # Manually trigger the onIndexChanged method or emit the signal
        self.onIndexChanged(index.row())
        # self.currentIndexChanged.emit(0)

    # def selectItem(self, listwidgetItem):
    #     view = self.view()
    #     index = view.indexFromItem(listwidgetItem)
    #     retrieved_widget = self.listw.itemWidget(listwidgetItem)
    #     # self.setCurrentIndex(index.row())
    #     widget_text = retrieved_widget._text.text()
    #     if widget_text == CustomCombo.NEW_LABEL:
    #         widget_text = widget_text[1:-1]
    #         # self.setEditable(True)
    #         # self.lineEdit().setFocus()
    #     # else:
    #         # self.setEditable(False)
    #     self.setCurrentText(widget_text)
    #     self.lineEdit().setText(widget_text)

    def removeCombo(self, item):
        view = self.view()
        index = view.indexFromItem(item)
        retrieved_widget = self.listw.itemWidget(item)
        widget_text = retrieved_widget._text.text()
        view.takeItem(index.row())
        self.refreshBackColors()
        self.showPopup()
        self.itemRemoved.emit(widget_text)

    def count(self):
        return self.view().count()

    # def on_index_changed(self, index):
    #     # Check if 'New Data' is selected
    #     if self.itemText(index) == "<new template>":
    #         self.lineEdit().setText("")  # Clear text to allow user input
    #         self.lineEdit().setFocus()  # Focus on the line edit to type new data

    # def on_editing_finished(self):
    #     # Get the text from the QLineEdit when user finishes editing
    #     new_text = self.lineEdit().text()
    #     if new_text and new_text != "<new template>" and new_text not in [self.itemText(i) for i in range(self.count())]:
    #         self.addItem(new_text)
    #         self.setCurrentText(new_text)  # Set the newly added item as selected

class LabelCombo(QWidget):
    def __init__(self, parent=None):
        super(LabelCombo, self).__init__(parent)
        layout = self.genLayout()
        self.setLayout(layout)
        self.decorateUi()

    def genLayout(self):
        layout = QHBoxLayout()
        layout.setSpacing(0)
        label = LabelButton()

        select = CustomCombo()
        select.setMinimumSize(QSize(16777215, 30))
        select.setMaximumSize(QSize(16777215, 30))

        self._label = label
        self._select = select

        layout.addWidget(self._label)
        layout.addWidget(self._select)
        layout.setStretch(0, 2)
        layout.setStretch(1, 5)
        return layout

    def decorateUi(self):
        self._label.setProperty('class','tool-btn edit-head')


class LabelSelect(QWidget):
    def __init__(self, parent=None):
        super(LabelSelect, self).__init__(parent)
        layout = self.genLayout()
        self.setLayout(layout)
        self.decorateUi()

    def genLayout(self):
        layout = QHBoxLayout()
        layout.setSpacing(0)
        label = LabelButton()

        select = QComboBox()
        select.setMinimumSize(QSize(16777215, 30))
        select.setMaximumSize(QSize(16777215, 30))
        select.setEditable(True)

        self._label = label
        self._select = select

        layout.addWidget(self._label)
        layout.addWidget(self._select)
        layout.setStretch(0, 2)
        layout.setStretch(1, 5)
        return layout

    def decorateUi(self):
        self._label.setProperty('class','tool-btn edit-head')


class LabelEdit(QWidget):
    def __init__(self, hint="Type text", parent=None):
        super(LabelEdit, self).__init__(parent)
        layout = self.genLayout()
        self.setLayout(layout)
        self.decorateUi()

    def genLayout(self):
        layout = QHBoxLayout()
        layout.setSpacing(0)
        label = LabelButton()
        self._label = label

        edit = QLineEdit()
        edit.setMinimumSize(QSize(16777215, 30))
        edit.setMaximumSize(QSize(16777215, 30))
        self._edit = edit

        layout.addWidget(self._label)
        layout.addWidget(self._edit)
        layout.setStretch(0, 2)
        layout.setStretch(1, 5)
        return layout

    def text(self):
        return self._edit.text()

    def decorateUi(self):
        self._label.setProperty('class','tool-btn edit-head')


class ContextTool(QWidget):
    def __init__(self, parent=None, env_desc=None):
        super(ContextTool, self).__init__(parent)
        self.env_desc = env_desc
        self.wgt = parent
        layout = self.genLayout()
        self.setLayout(layout)
        self.retranslateUi()
        self._cwContextTemplate._select.currentIndexChanged.connect(self.schemaChanged)
        self.selected_template = ""

    def genLayout(self):
        layout = CustomHBoxLayout()

        self._cwContextTemplate = LabelCombo()
        self._cwContextPicker = LabelSelect()
        self._cwContextHandle = LabelEdit()
        self._cwContextMutant = LabelSelect()
        self._cwContextPrompt = LabelEdit()
        self._sep_picker = CustomSeparator()
        self._sep_mutant = CustomSeparator()


        if not int(os.getenv('PLUGIN_EXPERT_MODE', 0)):
            self._sep_picker.hide()
            self._cwContextPicker.hide()
            self._sep_mutant.hide()
            self._cwContextMutant.hide()

        # Assemble layout
        layout.addWidget(self._cwContextTemplate)
        layout.addWidget(self._sep_picker)
        layout.addWidget(self._cwContextPicker)
        layout.addWidget(CustomSeparator())
        layout.addWidget(self._cwContextHandle)
        layout.addWidget(self._sep_mutant)
        layout.addWidget(self._cwContextMutant)
        layout.addWidget(CustomSeparator())
        layout.addWidget(self._cwContextPrompt)
        layout.addItem(CustomSpacer(30))
        layout.addWidget(self.initLmContextAdd())
        return layout

    def initLmContextAdd(self):
        submit = QPushButton()
        set_fixed_size(submit, 90, 30)
        # submit.setCheckable(False)
        submit.setCursor(CustomCursor())
        submit.setText(i18n("Submit"))
        submit.setEnabled(False)
        self._submit = submit
        return submit

    def getValues(self):
        _temp = self._cwContextTemplate._select.currentText()
        # _picker = self._cwContextPicker._select.currentText()
        _handle = self._cwContextHandle._edit.text()
        _mutant = self._cwContextMutant._select.currentText()
        _prompt = self._cwContextPrompt._edit.text()
        return {'temp': _temp, 'handle': _handle, 'mutant': _mutant, 'prompt': _prompt}

    def setSubmitHandler(self, handler):
        self._submit.clicked.connect(handler)

    def retranslateUi(self):
        self._cwContextTemplate._label.setText(i18n("Template"))
        self._cwContextPicker._label.setText(i18n("Picker"))
        self._cwContextHandle._label.setText(i18n("Handle"))
        self._cwContextMutant._label.setText(i18n("Mutant"))
        self._cwContextPrompt._label.setText(i18n("Prompt"))

        self._cwContextTemplate._select.lineEdit().setPlaceholderText(i18n("template pick"))
        self._cwContextPicker._select.lineEdit().setPlaceholderText(i18n("picker func"))
        self._cwContextHandle._edit.setPlaceholderText(i18n("picked value"))
        self._cwContextMutant._select.lineEdit().setPlaceholderText(i18n("mutant func"))
        self._cwContextPrompt._edit.setPlaceholderText(i18n("context prompt"))

    def loadData(self, schemas, pickers, mutants):
        self.schemas = schemas  # data/context.json
        self._cwContextTemplate._select.refreshInputs(list(schemas.keys()))
        self._cwContextPicker._select.addItems(pickers)
        self._cwContextMutant._select.addItems(mutants)
        self._cwContextTemplate._select.setCurrentText('')
        self._cwContextPicker._select.setCurrentText('')
        self._cwContextMutant._select.setCurrentText('')

    def get_handle_text(self, picker_data):
        handle_text = ""
        if plg_utils.is_items_same(picker_data):
            handle_text = str(picker_data[0]) if len(picker_data) else ""
        else:
            handle_text = "<multiple values>"
        return handle_text

    def schemaChanged(self):
        selected_item = self._cwContextTemplate._select.currentText()
        selected_index = self._cwContextTemplate._select.currentIndex()

        if selected_item and selected_item != CustomCombo.NEW_LABEL[1:-1]:
            self._cwContextPrompt._edit.setText(self.schemas[selected_item]['prompt'])
            picker_name = self.schemas[selected_item]['picker']

            module_path = 'picker'  # os.path.join(os.path.dirname(__file__), 'data/picker')
            module = plg_utils.import_path_simple(module_path)
            picker_data = []
            # try:
            picker_func = getattr(module, picker_name)
            for func_addr in self.wgt.pick:
                if picker_name == 'get_idle':
                    data = self.schemas[selected_item]['default']
                    if data:
                        picker_data.append(data)
                    else:
                        picker_data.append("")
                    break
                picker_data.append(picker_func(self.env_desc, func_addr))


            # except AttributeError:
            #     pass  # picker_data == []

            self._cwContextPicker._select.setCurrentText(picker_name)

            handle_text = self.get_handle_text(picker_data)

            self._cwContextHandle._edit.setText(handle_text)
            self._cwContextHandle._edit.setPlaceholderText(self.schemas[selected_item]['dummy'])

            self._cwContextMutant._select.setCurrentText(self.schemas[selected_item]['mutant'])
            self._cwContextHandle._edit.setCursorPosition(0)
            self._cwContextPrompt._edit.setCursorPosition(0)
        else:
            self._cwContextTemplate._select.lineEdit().setText(selected_item)
            self._cwContextPicker._select.setCurrentText("")
            self._cwContextHandle._edit.setText("")
            self._cwContextMutant._select.setCurrentText("")
            self._cwContextPrompt._edit.setText("")

        # meth = plg_utils.import_function('idaxlm.data.pickers', 'get_hello')
        # meth()


class PromptWidget(QTabWidget):
    def __init__(self, parent=None, env=None):
        super(PromptWidget, self).__init__(parent)
        self.gui = parent
        self.env = env
        self.pick = set()
        self.bags = {
            'base': None,
            'quiz': None,
            'chat': None,
            'done': None,
        }

        self.setStyleSheet("""
            QTabBar::tab:last {
                border: none;
                background: rgb(240, 240, 240);
                padding: 0 18px 0 18px;
            }
        """)

        self.tabs = {
            'context': ('Context', 'Context'),
            'prompt': ('Prompt', 'Prompt'),
            'results': ('Results', 'Results'),
            'chat': ('Chat', 'Chat'),
            'ghost': ('', 'Ghost'),
        }

        self.bank = {
            'bags': {
                'context': defaultdict(set),
                'prompt': defaultdict(set),
                'results': defaultdict(set),
                'chat': defaultdict(set)
            },
            'regs': {
                'context': {},
                'prompt': {},
                'results': {},
                'chat': {}
            }
        }

        self.services = []

        for tab_title, cls_token in self.tabs.values():
            tab_var_name = "tab{}".format(cls_token)
            tab_cls_name = "{}Tab".format(cls_token)

            tab_cls = globals()[tab_cls_name]
            tab_obj = tab_cls(self, env=self.env)

            setattr(self, tab_var_name, tab_obj)
            self.addTab(tab_obj, tab_title)

        self.tab_num = self.count()
        self.tab_prev = 0
        self.setEnabled(False)

        # self.tabBarClicked.connect(self.handle_tab_click)
        self.currentChanged.connect(self.onTabChanged)

    def get_common_bag_key(self, storage):
        keys = set()
        for func_addr in self.pick:
            key = self.gui.ui.taskView.model().link['funcs'][func_addr]['links'][storage]
            keys.add(key)
            if len(keys) > 1:  # Not all storage values are equal.
                return None
        else:
            # All storage values are equal and set/not set.
            return next(iter(keys)) if self.pick else ''

    def updateContext(self, pick=None):
        self.pick = list(pick['funcs'].keys())  # pick in `tasks` and `prompt` tools are different
        select_len = len(self.pick)
        to_enable = bool(select_len)
        lbl_color = QColor(0, 128, 64) if select_len else QColor(120, 120, 120)
        tab_idx = self.tab_num - 1
        self.setTabText(tab_idx, pick['name'])
        self.tabBar().setTabTextColor(tab_idx, lbl_color)
        #
        self.setEnabled(to_enable)
        #
        self.clear_tables()

        self.bags['base'] = self.get_common_bag_key('base')
        self.bags['quiz'] = self.get_common_bag_key('quiz')
        self.bags['chat'] = self.get_common_bag_key('chat')
        self.bags['done'] = self.get_common_bag_key('done')

        self.setTabEnabled(0, self.bags['base'] != None)  # len(self.pick)
        self.setTabEnabled(1, self.bags['quiz'] != None)  # len(self.pick)

        if self.bags['base'] == None and self.bags['quiz'] == None:
            self.setCurrentIndex(0)
            self.setEnabled(False)

        if self.currentIndex() in [2, 3] and select_len > 1:
            self.setCurrentIndex(0)

        self.setTabEnabled(2, select_len == 1)
        self.setTabEnabled(3, select_len == 1)
        #
        self.tabContext.setToolEnabled(to_enable)
        self.tabPrompt.setToolEnabled(to_enable)
        self.tabResults.setToolEnabled(select_len == 1)
        self.tabChat.setToolEnabled(select_len == 1)
        #
        self.onTabChanged(self.currentIndex())
        self.tabContext._tool.schemaChanged()
        #

    def clear_tables(self):
        self.tabContext._table.clearContents()
        self.tabContext._table.setRowCount(0)
        self.tabPrompt._table.clearContents()
        self.tabPrompt._table.setRowCount(0)
        self.tabResults._screen.clear()
        self.tabChat.table.clear()

    def onTabChanged(self, index):
        if index == self.tab_num - 1:
            current_index = self.currentIndex()
            self.setCurrentIndex(self.tab_prev)
            return

        tab_names = list(self.tabs.keys())
        if len(tab_names) - 1 < index:
            print("Unknown tab index: {}".format(index))
            return

        self.tab_prev = index
        tab_name = tab_names[index]

        if tab_name == 'chat':
            self.tabChat.table.clear()
            key = self.bags['chat']
            if key != '':
                for reg in self.bank['bags']['chat'][key]:
                    qas = self.bank['regs']['chat'][reg]
                    for action in qas:
                        self.tabChat.messageSend(action['owner'], action['text'])

        elif tab_name == 'context':
            self.tabContext._table.clearContents()
            self.tabContext._table.setRowCount(0)
            key = self.bags['base']
            if key != '':
                for reg in self.bank['bags']['context'][key]:
                    temp, text = self.bank['regs']['context'][reg]
                    self.tabContext._table.addRow((temp, text))

        elif tab_name == 'prompt':
            self.tabPrompt._table.clearContents()
            self.tabPrompt._table.setRowCount(0)
            key = self.bags['quiz']
            if key != '':
                for reg in self.bank['bags']['prompt'][key]:
                    temp, text = self.bank['regs']['prompt'][reg]
                    self.tabPrompt._table.addRow((temp, text))

        elif tab_name == 'results':
            self.tabResults._screen.clear()
            key = self.bags['done']

            if key != '':
                for reg in self.bank['bags']['results'][key]:
                    res = self.bank['regs']['results'][reg]
                    self.tabResults.updateScreen(res)
                    # self.tabPrompt._table.addRow((temp, text))

            ## self.ui.promptWidget.tabResults.updateScreen(self.job_data['funcs'][ctx_addr]['results'])

    def updateControls(self, ctx):
        pass
        # self._tool._cwContextTemplate._select.currentIndexChanged.emit(0)
        #         #
        #         self._table.clearContents()
        #         self._table.setRowCount(0)
        #         for temp, text in job_data['funcs'][func_addr]['context'].items():
        #             self._table.addRow((temp, text))
        #         #
        #         self._table.clearContents()
        #         self._table.setRowCount(0)
        #         for temp, text in job_data['funcs'][func_addr]['prompt'].items():
        #             self._table.addRow((temp, text))
        #         #
        #         self.ui.promptWidget.tabResults.updateScreen(job_data['funcs'][func_addr]['results'])

        #         self.ui.promptWidget.tabChat.table.clear()
        #         for desc in job_data['funcs'][func_addr]['chat']:
        #             self.ui.promptWidget.tabChat.messageSend(desc['owner'], desc['text'])

        #         if self.ui.wProviderTool._cwSelect.getData():
        #             self.ui.promptWidget.tabChat.setToolEnabled(True)
        #         else:
        #             self.ui.promptWidget.tabChat.setToolEnabled(False)

    def updateUi(self, pick):
        """Task context based widget UI update"""
        self.updateContext(pick)
        self.updateControls(pick)

    def updateSvc(self, services):
        self.services = services


class PromptTool(QWidget):
    def __init__(self, parent=None):
        super(PromptTool, self).__init__(parent)
        layout = self.genLayout()
        self.setLayout(layout)
        self.retranslateUi()
        self._cwPromptTemplate._select.currentIndexChanged.connect(self.contextTypeChanged)
        self.selected_template = ""

    def genLayout(self):
        layout = CustomHBoxLayout()

        self._cwPromptTemplate = LabelCombo()
        self._cwPromptText = LabelEdit()
        self._cwPromptType = LabelEdit()
        self._cwPromptCase = LabelEdit()
        self._cwPromptSafe = LabelEdit()

        # Assemble layout
        layout.addWidget(self._cwPromptTemplate)
        layout.addWidget(CustomSeparator())
        layout.addWidget(self._cwPromptText)
        layout.addWidget(CustomSeparator())
        layout.addWidget(self._cwPromptType)
        layout.addWidget(CustomSeparator())
        layout.addWidget(self._cwPromptCase)
        layout.addWidget(CustomSeparator())
        layout.addWidget(self._cwPromptSafe)
        layout.addItem(CustomSpacer(30))
        layout.addWidget(self.initLmPromptAdd())
        return layout

    def initLmPromptAdd(self):
        submit = QPushButton()
        set_fixed_size(submit, 90, 30)
        # submit.setCheckable(False)
        submit.setCursor(CustomCursor())
        submit.setEnabled(False)
        submit.setText(i18n("Submit"))
        self._submit = submit
        return submit

    def loadData(self, data):
        self.data = data
        self._cwPromptTemplate._select.refreshInputs(list(data.keys()))
        self._cwPromptTemplate._select.setCurrentText('')

    def contextTypeChanged(self):
        selected_item = self._cwPromptTemplate._select.currentText()
        selected_index = self._cwPromptTemplate._select.currentIndex()

        self.selected_template = selected_item
        if selected_item and selected_item != CustomCombo.NEW_LABEL[1:-1]:
            self._cwPromptText._edit.setText(self.data[selected_item]['text'])
            self._cwPromptType._edit.setText(self.data[selected_item]['type'])
            self._cwPromptCase._edit.setText(self.data[selected_item]['case'])
            self._cwPromptSafe._edit.setText(self.data[selected_item]['safe'])
            self._cwPromptText._edit.setCursorPosition(0)
            self._cwPromptType._edit.setCursorPosition(0)
            self._cwPromptCase._edit.setCursorPosition(0)
            self._cwPromptSafe._edit.setCursorPosition(0)
            self._submit.setEnabled(True)
        else:
            self._cwPromptTemplate._select.lineEdit().setText(selected_item)
            self._cwPromptText._edit.setText("")
            self._cwPromptType._edit.setText("")
            self._cwPromptCase._edit.setText("")
            self._cwPromptSafe._edit.setText("")

    def getValues(self):
        _temp = self._cwPromptTemplate._select.currentText()
        _text = self._cwPromptText._edit.text()
        _type = self._cwPromptType._edit.text()
        _case = self._cwPromptCase._edit.text()
        _safe = self._cwPromptSafe._edit.text()
        _comp = " ".join(part for part in [_text, _type, _case, _safe] if part)
        # {'temp': _temp, 'text': _text, 'type': _type, 'case': _case, 'safe': _safe, 'comp': _comp}
        return {'temp': _temp, 'comp': _comp}

    def setSubmitHandler(self, handler):
        self._submit.clicked.connect(handler)

    def retranslateUi(self):
        self._cwPromptTemplate._label.setText(i18n("Template"))
        self._cwPromptText._label.setText(i18n("Text"))
        self._cwPromptType._label.setText(i18n("Type"))
        self._cwPromptCase._label.setText(i18n("Case"))
        self._cwPromptSafe._label.setText(i18n("Safe"))

        self._cwPromptTemplate._select.lineEdit().setPlaceholderText(i18n("template pick"))
        self._cwPromptText._edit.setPlaceholderText(i18n("prompt text"))
        self._cwPromptType._edit.setPlaceholderText(i18n("return format"))
        self._cwPromptCase._edit.setPlaceholderText(i18n("example case"))
        self._cwPromptSafe._edit.setPlaceholderText(i18n("safety measures"))

        self._cwPromptSafe._label.setToolTip(i18n("Prompt part preventing LM amend the structure of response"))


class ResultsTool(QWidget):
    def __init__(self, parent=None):
        super(ResultsTool, self).__init__(parent)


# ':/idaxlm/icon_remove.png'


class CustomCompleter(QCompleter):
    removeCompletion = Signal(str)

    def __init__(self, completions=[], parent=None):
        super().__init__(parent)
        self.list_widget = QListWidget()
        self.setPopup(self.list_widget)
        self.model = QStringListModel(completions)
        self.setCaseSensitivity(Qt.CaseInsensitive)
        self.setFilterMode(Qt.MatchContains)
        self.setModel(self.model)
        self.populate_popup(completions)
        self.popup().hide()

    def populate_popup(self, completions):
        self.list_widget.clear()
        for completion in completions:
            item = QListWidgetItem(self.list_widget)
            item_widget = self.create_item_widget(completion)
            item.setSizeHint(item_widget.sizeHint())
            self.list_widget.setItemWidget(item, item_widget)

    def initSignalBtn(self, icon):
        btn = QToolButton()
        btn.setIcon(icon)
        btn.setAutoRaise(True)
        # btn.clicked.connect(lambda: self.itemOpSignal.emit(handler))
        return btn

    def create_item_widget(self, text):
        # Create a custom widget with an HBoxLayout containing text and a button
        widget = QWidget()
        layout = QHBoxLayout(widget)
        layout.setContentsMargins(5, 5, 5, 5)

        _text = QLabel(text)
        # _text = QPushButton(text)
        sizePolicy = QSizePolicy(QSizePolicy.Minimum, QSizePolicy.Preferred)
        _text.setSizePolicy(sizePolicy)
        btn_remove = self.initSignalBtn(QIcon(':/idaxlm/icon_remove.png'))
        btn_remove.clicked.connect(lambda: self.removeCompletion.emit(text))
        _text.setStyleSheet("text-align: left; padding-left: 10px; background-color: transparent; border: none;}")
        layout.addWidget(_text)
        layout.addStretch()
        layout.addWidget(btn_remove)

        # button = QPushButton("Action")
        # button.setFixedSize(80, 30)

        # layout.addWidget(label)
        # layout.addWidget(button)

        return widget

    def get_completions(self):
        return self.model.stringList()

    def add_completion(self, prompt):
        completions = self.get_completions()
        if prompt not in completions:
            completions.append(prompt)
            self.model.setStringList(completions)

        self.populate_popup(completions)


class ExpandingTextEdit(QTextEdit):
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)

        # Sample completions
        completions = self.load_suggestions()

        # Custom Completer
        self.completer = CustomCompleter(completions, self)
        self.completer.setCompletionMode(QCompleter.PopupCompletion)

        # Connect QTextEdit and QCompleter
        self.textChanged.connect(self.update_completer)
        self.completer.setWidget(self)
        self.completer.activated.connect(self.insert_completion)

        single_line_height = 30  # self.fontMetrics().height()
        self.max_height = single_line_height * 5 + 10

        self.setVerticalScrollBarPolicy(Qt.ScrollBarAlwaysOff)
        self.set_custom_line_height(17)
        self.setFixedHeight(single_line_height)  # single_line_height + 10
        self.textChanged.connect(self.adjustHeight)
        self.completer.removeCompletion.connect(self.handle_remove)

    def handle_remove(self, text):
        cat, qry = text.split('/')
        string_list = self.completer.get_completions()
        if text in string_list:
            string_list.remove(text)
        self.completer.model.setStringList(string_list)
        self.completer.populate_popup(string_list)
        # self.completer.popup().hide()
        # self.completer.popup().show()
        self.completer.complete()

    def set_custom_line_height(self, height):
        cursor = self.textCursor()
        block_format = cursor.blockFormat()

        # Set the line height in points. Line distance is relative to the font size.
        block_format.setLineHeight(height, QTextBlockFormat.FixedHeight)

        cursor.setBlockFormat(block_format)

        # Apply formatting to all the text
        cursor.select(QTextCursor.Document)
        self.setTextCursor(cursor)

    def adjustHeight(self):
        docHeight = int(self.document().size().height())
        if docHeight + 10 <= self.max_height:
            self.setFixedHeight(docHeight + 5)
            self.setVerticalScrollBarPolicy(Qt.ScrollBarAlwaysOff)
        else:
            self.setFixedHeight(self.max_height)
            self.setVerticalScrollBarPolicy(Qt.ScrollBarAsNeeded)

        # Filter suggestions based on input text
        # self.update_popup_suggestions()

    def update_completer(self):
        cursor = self.textCursor()
        cursor.select(cursor.WordUnderCursor)
        current_text = cursor.selectedText()

        if current_text.strip():
            self.completer.complete()

    def insert_completion(self, completion):
        self.clear()
        cursor = self.textCursor()
        cursor.select(cursor.WordUnderCursor)
        cat, qry = completion.split('/')
        cursor.insertText(qry)

    def update_suggestions(self, prompt, is_dump=False):
        if '/' not in prompt:
            prompt = "common/{}".format(prompt)

        self.completer.add_completion(prompt)

        data = defaultdict(set)
        for comp in self.completer.get_completions():
            cat, req = comp.split('/')
            data[cat].add(req)

        # Convert sets to lists for JSON serialization
        data = {cat: list(reqs) for cat, reqs in data.items()}

        plg_utils.set_json('data/chat.json', dict(data))
        self.load_suggestions()

    def load_suggestions(self):
        completion_desc =  plg_utils.get_json('data/chat.json')
        completions = []
        for cat, rqs in completion_desc.items():
            recs = ["{}/{}".format(cat, req) for req in rqs]
            completions.extend(recs)
        return completions


class ChatTool(QWidget):
    def __init__(self, parent=None):
        super(ChatTool, self).__init__(parent)
        self.gui = parent
        layout = self.genLayout(parent)
        self.setLayout(layout)
        self.retranslateUi()

    def genLayout(self, parent):
        layout = CustomVBoxLayout()
        l1 = CustomHBoxLayout(0, 4)
        l2 = CustomHBoxLayout(0, 2)

        self._chatPrompt = ExpandingTextEdit()
        l1.addWidget(self._chatPrompt)

        l2.addStretch(1)
        self._submit = self.genSendButton()
        l2.addWidget(self._submit)

        layout.addLayout(l1)
        layout.addLayout(l2)
        return layout

    def genSendButton(self):
        submit = QPushButton()
        submit.setEnabled(False)
        set_fixed_size(submit, 90, 30)
        # submit.setCheckable(False)
        submit.setCursor(CustomCursor())
        submit.setText(i18n("Submit"))
        return submit

    def retranslateUi(self):
        pass
        # self._cwMsgText._label.setText(i18n("Text"))

class CheckableComboBox(QComboBox):
    def __init__(self):
        super(CheckableComboBox, self).__init__()
        self.setEditable(True)
        self.lineEdit().setReadOnly(True)
        self.closeOnLineEditClick = False
        self.lineEdit().installEventFilter(self)
        self.view().viewport().installEventFilter(self)
        self.model().dataChanged.connect(self.updateLineEditField)
        self.itemDelegate = QStyledItemDelegate(self)
        self.setItemDelegate(self.itemDelegate)

    def hidePopup(self):
        super(CheckableComboBox, self).hidePopup()
        self.startTimer(100)

    def addItem(self, entry, userData=None):
        text, colr = entry
        item = QStandardItem()
        item.setText(text)
        if not userData is None:
            item.setData(userData)
        if not colr is None:
            item.setBackground(QColor(*colr))
        item.setFlags(Qt.ItemIsEnabled | Qt.ItemIsUserCheckable)
        item.setData(Qt.Unchecked, Qt.CheckStateRole)
        self.model().appendRow(item)

    def chgItem(self, row, text):
        if row == -1:
            self.addItem((text, None))
        else:
            item = self.model().item(row)
            item.setData(text, role=Qt.DisplayRole)

    def removeItem(self, row):
        self.model().removeRow(row)

    def sortItems(self):
        self.model().sort(0)

    def eventFilter(self, widget, event):
        if widget == self.lineEdit():
            if event.type() == QEvent.MouseButtonRelease:
                if self.closeOnLineEditClick:
                    self.hidePopup()
                else:
                    self.showPopup()
                return True
            return super(CheckableComboBox, self).eventFilter(widget, event)
        if widget == self.view().viewport():
            if event.type() == QEvent.MouseButtonRelease:
                indx = self.view().indexAt(event.pos())
                item = self.model().item(indx.row())

                if item.checkState() == Qt.Checked:
                    item.setCheckState(Qt.Unchecked)
                else:
                    item.setCheckState(Qt.Checked)
                return True
            return super(CheckableComboBox, self).eventFilter(widget, event)

    def updateLineEditField(self):
        text_container = []
        for i in range(self.model().rowCount()):
            if self.model().item(i).checkState() == Qt.Checked:
                text_container.append(self.model().item(i).text())
            text_string = '; '.join(text_container)
            self.lineEdit().setText(text_string)
            self.lineEdit().setCursorPosition(0)

    def getData(self):
        return self.lineEdit().text()

    def clearData(self):
        self.clear()


class FrameLayout(QWidget):
    def __init__(self, parent=None, title=None, env=None):
        self.env_desc = env
        QWidget.__init__(self, parent=parent)

        self._is_collasped = True
        self._title_frame = None
        self._content, self._content_layout = (None, None)

        title_frame = self.initTitleFrame(title, self._is_collasped)
        content_widget = self.initContent(self._is_collasped)

        self._main_v_layout = QVBoxLayout(self)
        self._main_v_layout.addWidget(title_frame)
        self._main_v_layout.addWidget(content_widget)

        self.initCollapsable()

    def initTitleFrame(self, title, collapsed):
        self._title_frame = self.TitleFrame(
            title=title,
            collapsed=collapsed,
            env=self.env_desc)
        return self._title_frame

    def initContent(self, collapsed):
        self._content = QWidget()
        self._content_layout = QVBoxLayout()

        self._content.setLayout(self._content_layout)
        self._content.setVisible(not collapsed)

        return self._content

    def addWidget(self, widget):
        self._content_layout.addWidget(widget)

    def initCollapsable(self):
        self._title_frame.clicked.connect(self.toggleCollapsed)

    def toggleCollapsed(self):
        self._content.setVisible(self._is_collasped)
        self._is_collasped = not self._is_collasped
        self._title_frame._arrow.setArrow(int(self._is_collasped))


    class TitleFrame(QFrame):

        clicked = Signal()
        def __init__(self, parent=None, title="", collapsed=False, env=None):
            QFrame.__init__(self, parent=parent)
            self.env_desc = env
            self.setMinimumHeight(24)
            self.move(QPoint(24, 0))

            self._hlayout = QHBoxLayout(self)
            self._hlayout.setContentsMargins(0, 0, 0, 0)
            self._hlayout.setSpacing(0)

            self._arrow = None
            self._title = None

            self._hlayout.addWidget(self.initArrow(collapsed))
            self._hlayout.addWidget(self.initTitle(title))

        def initArrow(self, collapsed):
            self._arrow = FrameLayout.Arrow(collapsed=collapsed, env=self.env_desc)
            return self._arrow

        def initTitle(self, title=None):
            self._title = QLabel(title)
            self._title.setMinimumHeight(24)
            self._title.move(QPoint(24, 0))

            return self._title

        def mousePressEvent(self, event):
            self.clicked.emit()
            return super(FrameLayout.TitleFrame, self).mousePressEvent(event)


    class Arrow(QFrame):
        def __init__(self, parent=None, collapsed=False, env=None):
            QFrame.__init__(self, parent=parent)
            self.env_desc = env
            self.setMaximumSize(24, 24)

            # horizontal == 0
            ha_point1 = QPointF(7.0, 8.0)
            ha_point2 = QPointF(17.0, 8.0)
            ha_point3 = QPointF(12.0, 13.0)
            self._arrow_horizontal = (ha_point1, ha_point2, ha_point3)
            # vertical == 1
            va_point1 = QPointF(8.0, 7.0)
            va_point2 = QPointF(13.0, 12.0)
            va_point3 = QPointF(8.0, 17.0)
            self._arrow_vertical = (va_point1, va_point2, va_point3)
            # arrow
            self._arrow = None
            self.setArrow(int(collapsed))

        def setArrow(self, arrow_dir):
            if arrow_dir:
                self._arrow = self._arrow_vertical
            else:
                self._arrow = self._arrow_horizontal

        def paintEvent(self, event):
            painter = QPainter()
            painter.begin(self)
            painter.setBrush(QColor(192, 192, 192))
            painter.setPen(QColor(64, 64, 64))
            if self.env_desc.lib_qt == 'pyqt5':
                painter.drawPolygon(*self._arrow)
            else:  # 'pyside'
                painter.drawPolygon(self._arrow)
            painter.end()
