import collections
import hashlib
import html
import importlib
import importlib.util
import inspect
import json
import os
import re
import struct
import sys
import textwrap
import zlib
#
from markdown_it import MarkdownIt
from pygments import highlight
from pygments.formatters import HtmlFormatter
from pygments.lexers import get_lexer_by_name


class PluginPath():
    def __init__(self, path):
        self.path = path

    def __enter__(self):
        sys.path.insert(0, self.path)

    def __exit__(self, exc_type, exc_value, traceback):
        try:
            sys.path.remove(self.path)
        except ValueError:
            pass

class RgbColor:
    def __init__(self, color_ref, invert=False, color_nam='unknown'):
        ver_py = sys.version_info.major
        if (ver_py == 2 and
            any(isinstance(color_ref, t) for t in (int, long)) and
            color_ref <= 0xFFFFFFFF):
            self.r, self.g, self.b = self.get_from_tuple(int(color_ref))
        elif (ver_py == 3 and
            isinstance(color_ref, int) and
            color_ref <= 0xFFFFFFFF):
            if invert:
                color_ref = self.invert_rgb_to_bgr(color_ref)
            self.r, self.g, self.b = self.get_from_tuple(color_ref)
        elif isinstance(color_ref, tuple) and len(color_ref) == 3:
            self.r, self.g, self.b = color_ref
        elif isinstance(color_ref, str):
            self.r, self.g, self.b = self.get_from_str(color_ref)
        else:
            raise ValueError("Invalid init value")
        self.name = color_nam

    def invert_rgb_to_bgr(self, color):
        # Extract the red, green, and blue components
        red = (color >> 16) & 0xFF    # Extract the red component
        green = (color >> 8) & 0xFF   # Extract the green component
        blue = color & 0xFF           # Extract the blue component

        # Reassemble them in the order BBGGRR
        inverted_color = (blue << 16) | (green << 8) | red
        return inverted_color

    def get_from_tuple(self, rgb_int):
        r = (rgb_int >> 16) & 255
        g = (rgb_int >> 8) & 255
        b = rgb_int & 255
        return (r, g, b)

    def get_from_str(self, color_ref):
        rgb_pat = r'rgb\((\d{1,3}),(\d{1,3}),(\d{1,3})\)'
        match = re.search(rgb_pat, color_ref)
        if match:
            r, g, b =  map(int, match.groups())
            if not all(0 <= c <= 255 for c in (r, g, b)):
                raise ValueError("Invalid color component values")
            return (r, g, b)
        else:
            raise ValueError("Invalid 'rgb(r,g,b)' string format")

    def get_to_str(self):
        return "rgb({},{},{})".format(self.r, self.g, self.b)

    def get_to_int(self):
        return (self.r << 16) + (self.g << 8) + self.b

    def __eq__(self, b):
        if isinstance(b, RgbColor):
            return self.r == b.r and self.g == b.g and self.b == b.b
        return False

    def __ne__(self, other):
        return not self.__eq__(other)

    def __str__(self):
        return self.get_to_str()

def from_hex(hex_sv):
    return int(hex_sv, base=16)

def f_float(val):
    if isinstance(val, float):
        return format(val, '.4f').rstrip('0').rstrip('.')
    else:
        return val

def invert_dict(original_dict):
    inverted_dict = {}
    for key, value in original_dict.items():
        if value not in inverted_dict:
            inverted_dict[value] = []
        inverted_dict[value].append(key)
    return inverted_dict

def add_prefix(func_name, func_pref, is_shadow=False):
    dlim_vars = ['%', '_']
    dlim_char = dlim_vars[int(is_shadow)]
    pref_norm = func_pref.strip('_')
    if is_shadow == False:
        dlim_anti = dlim_vars[int(not is_shadow)]
        if dlim_anti in pref_norm:
            pref_norm = pref_norm.replace(dlim_anti, dlim_char)
    func_name_new = '{}{}{}'.format(pref_norm, dlim_char, func_name)
    return func_name_new

def get_folder_tree(root_folder):
    folder_structure = {}
    for folder_name in os.listdir(root_folder):
        folder_path = os.path.join(root_folder, folder_name)
        if os.path.isdir(folder_path):
            folder_structure[folder_name] = get_folder_tree(folder_path)
        else:
            folder_structure[folder_name] = "file"
    return folder_structure

def get_ordered_folder_tree(root_folder):
    return collections.OrderedDict(sorted(get_folder_tree(root_folder).items()))

def import_path_simple(mod_name):
    # Dynamic module import; Python 2 compatible.
    dyn_mod_path = os.path.join(os.path.dirname(__file__), '..', 'data')
    sys.path.append(dyn_mod_path)
    module = __import__(mod_name)
    sys.path.pop()
    return module

def import_dyn_module(mod_name):
    dyn_mod_path = os.path.join(os.path.dirname(__file__), '..', 'data', '{}.py'.format(mod_name))
    spec = importlib.util.spec_from_file_location(mod_name, dyn_mod_path)
    module = importlib.util.module_from_spec(spec)
    spec.loader.exec_module(module)
    return module


def get_funcs(file_name, filter=""):
    try:
        # Construct the full path to the file
        module_path = os.path.join(os.path.dirname(__file__), '..', file_name)

        # Get the module name (without .py extension)
        module_name = os.path.splitext(os.path.basename(module_path))[0]

        # Dynamically load the module from the file
        spec = importlib.util.spec_from_file_location(module_name, module_path)
        module = importlib.util.module_from_spec(spec)
        spec.loader.exec_module(module)

        # List to store function names
        function_names = []

        # Inspect the module to find functions
        for name, obj in inspect.getmembers(module, inspect.isfunction):
            if obj.__module__ == module_name:  # Ensure the function is defined in this module
                if name.startswith(filter):
                    function_names.append(name)

        return function_names

    except FileNotFoundError:
        print("Error: The file '{}' was not found.".format(file_name))
        return None
    except (ImportError, IOError) as e:
        print("Error: Could not import or access the file '{}'. {}".format(file_name, str(e)))
        return None


def get_json(file_name):
    try:
        data_path = os.path.join(os.path.dirname(__file__), '..', file_name)
        with open(data_path, 'r') as f:
            return json.load(f)
    except FileNotFoundError:
        print("Error: The file '{}' was not found.".format(file_name))
        return None
    except json.JSONDecodeError:
        print("Error: The file '{}' contains invalid JSON.".format(file_name))
        return None

def set_json(file_name, data):
    try:
        data_path = os.path.join(os.path.dirname(__file__), '..', file_name)
        with open(data_path, 'w') as f:
            json.dump(data, f, indent=4)
        print("Successfully wrote data to '{}'.".format(file_name))
    except IOError as e:
        print("Error: Could not write to file '{}'. {}".format(file_name, str(e)))
    except TypeError as e:
        print("Error: The data provided is not JSON serializable. {}".format(str(e)))

def import_function(module_name, function_name):
    module = importlib.import_module(module_name)
    func = getattr(module, function_name)
    return func

def get_int_list_pack(lst):
    byte_data = b''
    for num in lst:
        if 0 <= num <= 0xffffffff:
            # Pack as a 4-byte integer.
            byte_data += struct.pack('I', num)
        elif 0 <= num <= 0xffffffffffffffff:
            # Pack as an 8-byte integer.
            byte_data += struct.pack('Q', num)
        else:
            raise ValueError("Integer out of supported range")
    return byte_data

def get_list_crc32(lst):
    byte_data = get_int_list_pack(lst)
    return zlib.crc32(byte_data) & 0xffffffff

def get_dict_sha1(dic):
    dump = json.dumps(dic, sort_keys=True)  # , separators=(',', ':')
    sha1_hash = hashlib.sha1(dump.encode('utf-8')).hexdigest()
    return sha1_hash

def get_dict_sha1_trunc(dic):
    return get_dict_sha1(dic)[:8]

def get_int_list_sha1(lst):
    byte_data = get_int_list_pack(lst)
    sha1_hash = hashlib.sha1(byte_data).hexdigest()
    return sha1_hash

def get_int_list_sha1_trunc(lst):
    return get_int_list_sha1(lst)[:8]

def get_str_list_sha1_trunc(lst):
    return get_str_list_sha1(lst)[:8]

def get_clu_crc32(funcs):
    clu_order = sorted(funcs)
    return get_list_crc32(clu_order)

def get_str_md5(input_string):
    md5 = hashlib.md5()
    md5.update(input_string.encode('utf-8'))
    return md5.hexdigest()

def get_str_sha1(string):
    sha1 = hashlib.sha1()
    sha1.update(string.encode('utf-8'))
    return sha1.hexdigest()

def get_str_sha1_trunc(string):
    return get_str_sha1(string)[:8]

def get_list_md5(lst):
    byte_data = get_list_pack(lst)
    md5 = hashlib.md5()
    md5.update(byte_data)
    return md5.hexdigest()

def get_str_list_sha1(strings):
    concatenated = ''.join(strings)
    sha1 = hashlib.sha1(concatenated.encode('utf-8')).hexdigest()
    return sha1

def highlight_code(text_html):
    def highlight_code(match):
        formatter = HtmlFormatter(lineseparator="<br />",
                        prestyles="white-space:pre-wrap; font-family: 'Fira Code Medium';",
                        noclasses=True,
                        nobackground=True,
                        style="solarized-light")

        lexer_name = match.group(1)
        code = match.group(2)
        code = html.unescape(code)
        lexer = get_lexer_by_name(lexer_name, stripnl=False, ensurenl=False)
        highlighted_code = highlight(code, lexer, formatter)
        return '<pre>{}</pre>'.format(highlighted_code)

    code_block_pattern = re.compile(r'<pre><code class="language-(.*?)">(.+?)</code></pre>', re.DOTALL)
    return code_block_pattern.sub(highlight_code, text_html)

def md_to_html(text_md):
    md = MarkdownIt()
    return md.render(text_md)

def render_md(text_md):
    text_html = md_to_html(text_md)
    return highlight_code(text_html)

def format_code(text):
    return "\n".join(textwrap.wrap(text, 80, replace_whitespace=False)).strip()

def split(strng, sep, pos):
    strng = strng.split(sep)
    return sep.join(strng[:pos]), sep.join(strng[pos:])

def get_fenced(code):
    return "\n```\n{}\n```\n".format(code)

def strip_comments(text):
    lines = text.splitlines()
    f_lines = [l for l in lines if not l.strip().startswith('//')]
    return '\n'.join(f_lines)

def get_error_text(msg):
    try:
        obj = json.loads(msg)

        if 'message' in obj:
            return str(obj['message'])
        if 'error' in obj and 'message' in obj['error']:
            return str(obj['error']['message'])
        else:
            return json.dumps(obj)
    except:
        return str(msg)

def get_list_repr(lst):
    return "{} {}".format(", ".join(list(lst)[:3]), '...' if len(lst) > 3 else '')

def is_items_same(lst):
    return all(x == lst[0] for x in lst)
