#!/bin/bash

##########################################################
# IdaPython Plugin Setup Script
# Copyright (c) 2024
# Sergejs 'HRLM' Harlamovs <harlamism@gmail.com>
# Licensed under the MIT License. All rights reserved.
##########################################################

# Debug output flag
is_debug=true

print_banner() {
    local banner_file="${1:-banner.txt}"
    if [ -f "$banner_file" ]; then
        cat "$banner_file"
    else
        echo "Banner file not found: $banner_file"
    fi
}

echo_debug() {
    if [ "$is_debug" = true ]; then
        echo "DEBUG: $1"
    fi
}

get_plg_dir_std() {
    local script_path="$1"
    if [ ! -e "$script_path" ]; then
        echo "Error: Path does not exist."
        return 1
    fi
    local repo_name=$(basename "$script_path")
    local dir_name=$(echo "$repo_name" | tr '[:upper:]' '[:lower:]')
    echo "$dir_name"
}

get_plg_fmt() {
    local path="$1"

    # Extract the version number (X.Y) from the pattern 'idapro-X.Y'.
    # Assuming 'idapro-' is the static part and the version is separated by a hyphen.
    if [[ "$path" =~ idapro-([0-9]+\.[0-9]+) ]]; then
        version="${BASH_REMATCH[1]}"

        # Split version into major and minor part.
        major_version="${version%%.*}"
        minor_version="${version##*.}"

        if (( major_version >= 9 )); then
            return 2
        else
            return 1
        fi
    else
        return 0
    fi
}

# Function to process an array of paths and check their versions
process_paths() {
    local paths=("$@")  # Accept the array of paths as arguments

    local all_above_9=1  # Set to 1 (true) if all paths return 2
    local all_below_9=1  # Set to 1 (true) if all paths return 1
    local found_version=0  # Set to 1 (true) if a version was found

    for path in "${paths[@]}"; do
        get_plg_fmt "$path"  # Call the version checking function
        result=$?

        if [[ $result -eq 2 ]]; then
            all_below_9=0  # If a path is 9.0 or above, not all are below 9.0
            found_version=1  # Version was found
        elif [[ $result -eq 1 ]]; then
            all_above_9=0  # If a path is below 9.0, not all are above 9.0
            found_version=1  # Version was found
        elif [[ $result -eq 0 ]]; then
            all_above_9=0  # If no version, it's not all above 9.0
            all_below_9=0  # If no version, it's not all below 9.0
        fi
    done

    # Return the result based on the flags
    if [[ $all_above_9 -eq 1 ]]; then
        return 2  # All paths are 9.0 or above
    elif [[ $all_below_9 -eq 1 ]]; then
        return 1  # All paths are below 9.0
    else
        if [[ $found_version -eq 1 ]]; then
            return 0  # Mixed results
        else
            return 0  # No versions found in any path
        fi
    fi
}

get_os_name() {
    local os_type=$(uname)
    case "$os_type" in
        Darwin)
            echo "macOS"
            ;;
        Linux)
            echo "Linux"
            ;;
        *)
            echo "Unknown"
            ;;
    esac
}

get_pat_paths_sfx() {
    local s_dir="$1"
    local s_pat="$2"
    local s_sfx="$3"

    s_dir=~  # $(realpath "$s_dir")
    SEARCH_DIR=~

    # Find directories and append suffix
    while IFS= read -r path; do
        dirs+=("${path}${s_sfx}")
    done < <(find $s_dir -maxdepth 1 -type d -name "$s_pat")

    # dirs=($(find $SEARCH_DIR -maxdepth 1 -type d -name 'idapro-*' | while read dir; do echo "${dir}/plugins"; done))

    # Print directories as newline-separated values.
    # This makes it easier to capture the output.
    printf "%s\n" "${dirs[@]}"
}

echo_act_menu() {
    echo "1 - Install files"
    echo "2 - Install Symlinks"
    echo "3 - Remove plugin"
}

echo_fmt_menu() {
    echo "1 - Old plugin format"
    echo "2 - New plugin format"
}

get_act_num() {
    # Prompt the user for install action
    read -p "Choose an action: " act_num
    echo "$act_num"
}

get_fmt_num() {
    # Prompt the user for plugin format
    read -p "Choose a format: " fmt_num
    echo "$fmt_num"
}

perform_action() {
    local ins_act="$1"
    local ins_src="$2"
    local ins_dst="$3"
    local ins_fmt="$4"

    if [ "$ins_fmt" -eq 0 ]; then
        echo_fmt_menu
        plg_fmt=$(get_fmt_num)
    fi

    case $ins_act in
        1)
            # Copy folder with files from the script directory to the selected directory
            if [ -d "$ins_src" ]; then
                cp -r "$ins_src" "$ins_dst"
                echo "Copied $ins_src to $ins_dst"
            else
                echo "Source folder $ins_src does not exist."
            fi
        ;;
        2)
            # Create symlink with the same source destination paths
            if [ -d "$ins_src" ]; then
                ln -s "$ins_src" "$ins_dst"
                echo "Created symlink from $ins_src to $selected_dir/$ins_dst"
            else
                echo "Source folder $ins_src does not exist."
            fi
            ;;
        3)
            # Remove folder and symlink with name '$plugin_folder' if they exist
            if [ -d "$ins_dst" ]; then
                rm -rf "$ins_dst"
                echo "Removed folder $ins_dst"
            fi
            if [ -L "$ins_dst" ]; then
                rm "$ins_dst"
                echo "Removed symlink $ins_dst"
            fi
            ;;
        *)
            echo "Invalid action."
            ;;
    esac
}

install_deps() {
    local repo_path="$1"

    read -p "Should the dependencies be installed (Y/N)? " is_deps

    if [[ "$is_deps" == "Y" || "$is_deps" == "y" ]]; then
        if [ -f "$repo_path/requirements.txt" ]; then
            python3 -m pip install -r "$repo_path/requirements.txt"
        else
            echo "requirements.txt not found in $repo_path"
        fi
    else
        echo "Skipping dependency installation."
    fi
}


main() {
    # OS-Neutral Variables
    # Repository path
    script_path=$(dirname "$(realpath "$0")")
    # Plugin directory
    # plugin_dir="dir_name"
    if [ -z "$plugin_dir" ]; then
        plugin_dir=$(get_plg_dir_std "$script_path")
    fi
    plugin_path="$script_path/$plugin_dir"
    # New plugin format IDA
    new_fmt_ver="9.0"
    plg_sfx="/plugins"

    echo_debug "Repository path: $script_path"
    echo_debug "Plugin dir: $plugin_dir"

    print_banner "banner.txt"
    echo

    os_name=$(get_os_name)
    case "$os_name" in
        macOS)
            echo "The script is running on macOS."
            # macOS-specific actions
            ;;
        Linux)
            echo_debug "The script is running on Linux."
            # Linux-specific actions
            mapfile -t plg_paths < <(get_pat_paths_sfx ~ "idapro-*" "$plg_sfx")
            if [ ${#plg_paths[@]} -eq 0 ]; then
                echo "\"IDA Pro\" is not installed or is in a non-standard location."
                read -p "Enter the path to \"IDA Pro\": (N - exit): " ida_path
                ins_path="${ida_path}${plg_sfx}"
                plg_fmt=$(get_plg_fmt "$ins_path")
            else
                plg_paths+=("$HOME/.idapro/plugins")
                fmt_mix=$(process_paths "${plg_paths[@]}")

                # Display the numbered menu
                for i in "${!plg_paths[@]}"; do
                    echo "$((i + 1)) - ${plg_paths[$i]}"
                done

                echo
                read -p "Select the installation path (0 - exit): " path_num
                if [ "$path_num" -eq 0 ]; then
                    exit 0
                fi
                # Validate the user's input
                if ! [[ "$path_num" =~ ^[0-9]+$ ]] || [ "$path_num" -lt 1 ] || [ "$path_num" -gt ${#plg_paths[@]} ]; then
                    echo "Invalid selection."
                    exit 1
                fi
                ins_path="${plg_paths[$((path_num - 1))]}"
                plg_fmt=$(get_plg_fmt "$ins_path")
                if [ "$plg_fmt" -eq 0 ]; then
                    plg_fmt="$fmt_mix"
                fi
            fi
            echo_debug "Installation path: $ins_path"
            echo_debug "Plugin format: $plg_fmt"
            echo_act_menu
            act_num=$(get_act_num)

            perform_action "$act_num" "$plugin_path" "$ins_path" "$plg_fmt"
            install_deps "$script_path"

            ;;
        *)
            echo "The script is running on an unknown OS."
            # Actions for unknown OS
            ;;
    esac
}

main
