#pragma once

#include "PrecompiledHeader.h"

// enable msg() calls about parsed structures
//#define UNWIND_PRINT
// enable queuing autoanalysis
#define UNWIND_MARK_CODE
// enable stats about new code found
//#define UNWIND_MARK_CODE_STATS

typedef unsigned int uint;
typedef uint8_t u8;
typedef uint16_t u16;
typedef uint32_t u32;
typedef uint64_t u64;
typedef int8_t s8;
typedef int16_t s16;
typedef int32_t s32;
typedef int64_t s64;


class IDAInfo
{
	enum CPUType : uint
	{
		UNKNOWN,
		AMD64,
		X86,
		ARM = 4
	};

	ea_t image_base;
	CPUType processor;

	DECLARE_UNCOPYABLE(IDAInfo)

public:
	IDAInfo()
	{
		image_base = 0;
		processor = UNKNOWN;
	}

	ea_t GetImageBase() const { return image_base; }
	bool IsAMD64() const { return processor == AMD64; }
	bool IsARM() const { return processor == ARM; }
	bool IsWindows() const { return inf.filetype == f_PE; }

	void Update()
	{
		image_base = get_imagebase();

		processor = UNKNOWN;

		if (std::string(inf.procName) == "ARM")
			processor = ARM;
		else if (std::string(inf.procName) == "metapc")
			processor = inf.is_64bit() ? AMD64 : X86;
	}
} idainfo;

class CodeMarker
{
	static void set_range_internal(ea_t b, ea_t const e, bool const is_func)
	{
#ifdef UNWIND_MARK_CODE
		auto begin = b;
		auto end = e ? e : begin;

		if (idainfo.IsARM())
		{
			begin &= ~1;
			end &= ~1;
		}

		// check to see if the range needs to be converted to 'undefined' first
		auto cur_ea = begin;
		uint len = 0;
		do
		{
			if (isCode(get_flags_novalue(cur_ea)))
				break;
			uint cur_len = decode_insn(cur_ea);
			if (cur_len == 0)
				break;
			else
			{
				cur_ea += cur_len;
				len += cur_len;
			}
		} while (cur_ea < end);

		if (len != 0)
			do_unknown_range(begin, len, DOUNK_SIMPLE);

		//msg("add %a - %a%s\n", begin, end, is_func ? " (func)" : "");

		// plan to make code
		if (is_func)
			auto_make_proc(begin);
		else
			auto_make_code(begin);

		if (e)
		{
			// plan to reanalyze
			noUsed(begin, end);
			// plan to analyze
			auto_mark_range(begin, end, AU_FINAL);
		}
#endif
	}

public:
	static void set_range(ea_t const begin, ea_t const end = 0, bool const is_func = false)
	{
		set_range_internal(begin, end, is_func);
	}

	static void set_range(ea_t const begin, bool const is_func = false, ea_t const end = 0)
	{
		set_range_internal(begin, end, is_func);
	}
};

inline bool is_exec(ea_t const ea)
{
	auto s = getseg(ea);
	return s && (s->perm & SEGPERM_EXEC);
}

template<uint index, uint num_bits = 1, typename T = u32>
struct Bitfield
{
	T data;
	static T const mask = (1ull << num_bits) - 1ull;
	
	template<typename T2>
	Bitfield &operator=(T2 rhs)
	{
		data = (data & ~(mask << index)) | (((num_bits > 1) ? rhs & mask : !!value) << index);
		return *this;
	}

	operator T() const { return (data >> index) & mask; }
};

struct rva_t
{
	enum addr_type
	{
		relative,
		absolute
	};
	
	ea_t rva;

	rva_t() : rva() { }

	rva_t(ea_t const address, addr_type const type = relative)
		: rva(address)
	{
		if (type == absolute)
			rva -= idainfo.GetImageBase();
	}

	u32 read() const { return get_32bit(get()); }

	u32 read_inc(uint const amount = 4)
	{
		auto value = read();
		rva += amount;
		return value;
	}

	operator ea_t() const { return idainfo.GetImageBase() + rva; }
	rva_t &operator &=(int const &&rhs) { rva &= rhs; return *this; }
	
	// This must be used when you want the absolute address,
	// but are using the instance in an non-typesafe manner
	// (ie, passing to a variadic function)
	ea_t get() const { return (ea_t)*this; }
};

struct SEHScopeTable
{
	bool valid;

	struct SEHScope
	{
		// first instruction in the __try scope
		rva_t begin;
		// instruction *after the end* of the __try scope
		rva_t end;
		// either the __except filter function rva (including __finally)
		// or an immediate value such as 1 (EXCEPTION_EXECUTE_HANDLER)
		rva_t handler;
		// first instruction in the __except scope
		rva_t jump_target;
	};

	u32 count;
	std::vector<SEHScope> table;

	SEHScopeTable() : valid(), count(), table() {}

	SEHScopeTable(rva_t address)
	{
		count = address.read_inc();
		
		for (u32 i = 0; i < count; ++i)
		{
			SEHScope scope;
			scope.begin = address.read_inc();
			scope.end = address.read_inc();
			scope.handler = address.read_inc();
			scope.jump_target = address.read_inc();
			table.push_back(scope);

			valid = (scope.begin.rva != 0 && scope.end.rva != 0 && scope.handler.rva != 0) &&
				(scope.begin.rva < scope.end.rva) &&
				is_exec(scope.begin) && is_exec(scope.end);
			
			if (!valid)
			{
				table.clear();
				break;
			}
		}

		if (is_valid())
			mark_exception_data();
	}

	bool is_valid() const { return valid; }

	void mark_exception_data() const
	{
		for (u32 i = 0; i < table.size(); ++i)
		{
			auto const it = &table[i];

			// Mark as code

			CodeMarker::set_range(it->begin, it->end);

			if (it->handler.rva != 1)
				CodeMarker::set_range(it->handler, true);

			if (it->jump_target.rva != 0)
				CodeMarker::set_range(it->jump_target, false);

			// Add comments

			std::string scope_num = [&i]() -> std::string
			{
				std::stringstream ss;
				ss << i;
				return ss.str();
			}();

			// IDA needs the effective address for commenting thumb code...
			SEHScope comment = *it;
			if (idainfo.IsARM())
			{
				comment.begin &= ~1;
				comment.end &= ~1;
				comment.handler &= ~1;
				comment.jump_target &= ~1;
			}

			add_long_cmt(comment.begin, true, ("__try_" + scope_num + " {").c_str());
			add_long_cmt(comment.end, true, ("} //__try_" + scope_num).c_str());

			std::stringstream try_begin;
			try_begin << std::hex << it->begin;

			if (it->handler.rva == 1 && it->jump_target.rva != 0)
			{
				// only EXCEPTION_EXECUTE_HANDLER is encoded immediately this way
				set_cmt(comment.jump_target,
					("__except_" + scope_num + "(EXCEPTION_EXECUTE_HANDLER): " + try_begin.str()).c_str(), false);
			}
			else if (it->jump_target.rva == 0)
			{
				// execution will continue from the __try scope into the __finally
				std::stringstream finally_cmt;
				finally_cmt << std::hex << it->handler;
				set_cmt(comment.end, std::string("__finally_" + scope_num + "(" + finally_cmt.str() + ")").c_str(), false);
				set_cmt(comment.handler, std::string("__finally_" + scope_num + ": " + try_begin.str()).c_str(), false);
			}
			else
			{
				std::stringstream filter;
				filter << std::hex << it->handler;
				set_cmt(comment.jump_target, std::string("__except_" + scope_num + "(" + filter.str() + ")").c_str(), false);

				std::stringstream except;
				except << std::hex << it->jump_target;
				set_cmt(comment.handler, std::string("except_filter_" + scope_num + ": " + except.str()).c_str(), false);
			}
		}
	}

	void print() const
	{
		msg("seh_scopes: %x\n", count);
		std::for_each(table.begin(), table.end(), [](SEHScope scope)
		{
			msg("begin %a end %a handler %a jump_target %a\n",
				scope.begin.get(),
				scope.end.get(),
				(scope.handler.rva == 1) ? 1 : scope.handler.get(),
				(scope.jump_target.rva == 0) ? 0 : scope.jump_target.get());
		});
	}
};

struct FuncInfo
{
	struct TryBlockMap
	{
		struct HandlerType
		{
			enum
			{
				ADJ_CONST = 1,
				ADJ_VOLATILE = 2,
				ADJ_UNALIGNED = 4,
				ADJ_REFERENCE = 8,
				ADJ_RESUMABLE = 16,
				ADJ_STDDOTDOT = 64,
				ADJ_COMPLUS = 0x80000000
			};
			u32 adjectives;
			rva_t type;
			s32 catch_obj;
			rva_t handler;
			// amd64 only
			u32 frame;
		};

		struct TryBlockMapEntry
		{
			s32 try_low;
			s32 try_high;
			s32 catch_high;
			s32 num_catches;
			rva_t handler_list;
			typedef std::vector<HandlerType> HandlerList;
			HandlerList handlers;
		};

		std::vector<TryBlockMapEntry> map;

		TryBlockMap() : map() {}

		TryBlockMap(rva_t address, u32 const count)
		{
			for (u32 i = 0; i < count; ++i)
			{
				TryBlockMapEntry entry;
				entry.try_low = address.read_inc();
				entry.try_high = address.read_inc();
				entry.catch_high = address.read_inc();
				entry.num_catches = address.read_inc();
				entry.handler_list = address.read_inc();

				// rva_t::read_inc() modifies the underlying rva,
				// so we must use a copy
				rva_t handler_list = entry.handler_list;

				for (s32 num_catch = 0; num_catch < entry.num_catches; ++num_catch)
				{
					HandlerType handler;
					handler.adjectives = handler_list.read_inc();
					handler.type = handler_list.read_inc();
					handler.catch_obj = handler_list.read_inc();
					handler.handler = handler_list.read_inc();
					if (idainfo.IsAMD64())
						handler.frame = handler_list.read_inc();
					entry.handlers.push_back(handler);
				}

				map.push_back(entry);
			}
		}

		void print() const
		{
			std::for_each(map.begin(), map.end(), [](TryBlockMapEntry entry)
			{
				msg("try_low %i try_high %i catch_high %i num_catches %i handler_list %a\n",
					entry.try_low, entry.try_high, entry.catch_high, entry.num_catches, entry.handler_list.get());

				std::for_each(entry.handlers.begin(), entry.handlers.end(), [](HandlerType handler)
				{
					msg("adjectives %x type %a catch_obj %x handler %a",
						handler.adjectives, handler.type.get(), handler.catch_obj, handler.handler.get());
					if (idainfo.IsAMD64())
						msg(" frame %8x", handler.frame);
					msg("\n");
				});
			});
		}
	};

	struct UnwindMap
	{
		typedef std::pair<s32, rva_t> UnwindMapEntry;
		std::vector<UnwindMapEntry> map;

		UnwindMap() : map() {}

		UnwindMap(rva_t address, u32 const count)
		{
			for (u32 i = 0; i < count; ++i)
			{
				s32 to_state = address.read_inc();
				rva_t action = address.read_inc();
				map.push_back(std::make_pair(to_state, action));
			}
		}

		void print() const
		{
			std::for_each(map.begin(), map.end(), [](UnwindMapEntry entry)
			{
				msg("to_state %i action ", entry.first);
				if (entry.second.rva == 0)
					msg("<none>");
				else
					msg("%a", entry.second.get());
				msg("\n");
			});
		}
	};

	struct IpToStateMap
	{
		typedef std::pair<rva_t, s32> IpToStateMapEntry;
		std::vector<IpToStateMapEntry> map;

		IpToStateMap() : map() {}

		IpToStateMap(rva_t address, u32 const count)
		{
			for (u32 i = 0; i < count; ++i)
			{
				rva_t ip = address.read_inc();
				s32 state = address.read_inc();
				map.push_back(std::make_pair(ip, state));
			}
		}

		void print() const
		{
			std::for_each(map.begin(), map.end(), [](IpToStateMapEntry entry)
			{
				msg("ip %a : %i\n", entry.first.get(), entry.second);
			});
		}
	};

	union
	{
		enum : u32 { header_magic = 0x19930522 };
		enum BBTFlags { unique = 1 };
		u32 header;
		Bitfield<0, 29> magic;
		Bitfield<29, 3> bbt_flags;
	};

	u32 max_state;
	rva_t unwindmap_rva;
	u32 num_try_blocks;
	rva_t try_blockmap_rva;
	u32 num_ip_to_statemap_entries;
	rva_t ip_to_statemap_rva;
	s32 unwind_help;
	rva_t es_typelist;
	u32 eh_flags;

	UnwindMap unwind_map;
	TryBlockMap try_block_map;
	IpToStateMap ip_state_map;

	typedef std::map<s32, std::string> state_chain_t;

	FuncInfo()
		: max_state()
		, unwindmap_rva()
		, num_try_blocks()
		, try_blockmap_rva()
		, num_ip_to_statemap_entries()
		, ip_to_statemap_rva()
		, unwind_help()
		, es_typelist()
		, eh_flags()
		, unwind_map()
		, try_block_map()
		, ip_state_map()
	{}

	FuncInfo(rva_t address)
	{
		header = address.read_inc();

		if (!is_valid())
		{
			//msg("wrong magic(%8x), expected %8x\n", (u32)magic, header_magic);
			return;
		}

		max_state = address.read_inc();
		unwindmap_rva = address.read_inc();
		num_try_blocks = address.read_inc();
		try_blockmap_rva = address.read_inc();
		num_ip_to_statemap_entries = address.read_inc();
		ip_to_statemap_rva = address.read_inc();
		unwind_help = address.read_inc();
		es_typelist = address.read_inc();
		eh_flags = address.read_inc();

		unwind_map = UnwindMap(unwindmap_rva, max_state);
		try_block_map = TryBlockMap(try_blockmap_rva, num_try_blocks);
		ip_state_map = IpToStateMap(ip_to_statemap_rva, num_ip_to_statemap_entries);

		mark_exception_data();
	}

	bool is_valid() const { return magic == header_magic; }

	void mark_exception_data() const
	{
		// Mark as code

		for (auto i = unwind_map.map.cbegin(); i != unwind_map.map.cend(); ++i)
			if (i->second.rva != 0)
				CodeMarker::set_range(i->second, false);
		for (auto i = try_block_map.map.cbegin(); i != try_block_map.map.cend(); ++i)
			for (auto j = i->handlers.cbegin(); j != i->handlers.cend(); ++j)
				CodeMarker::set_range(j->handler, false);
		for (auto i = ip_state_map.map.cbegin(); i != ip_state_map.map.cend(); ++i)
				CodeMarker::set_range(i->first, false);

		// build the unwind chains

		state_chain_t states;
		for (auto it = ip_state_map.map.cbegin(); it != ip_state_map.map.cend(); ++it)
		{
			std::stringstream this_state;
			s32 cur_state = it->second;

			if (cur_state == -1 || states.count(cur_state))
				// already have a graph for this starting state
				continue;

			while (cur_state != -1)
			{
				auto entry = &unwind_map.map[cur_state];
				if (entry->second.rva != 0)
					this_state << std::hex << entry->second << (entry->first == -1 ? "" : ", ");
				cur_state = entry->first;
			}

			if (!this_state.str().empty())
				states[it->second] = this_state.str();
		}

		// Add comments for the try blocks and unwind regions

		u32 try_num = 0;

		for (auto tryblock = try_block_map.map.cbegin(); tryblock != try_block_map.map.cend(); ++tryblock)
		{
			for (auto ip_state = ip_state_map.map.cbegin(); ip_state != ip_state_map.map.cend(); ++ip_state)
			{
				if (ip_state->second >= tryblock->try_low && ip_state->second <= tryblock->try_high)
				{
					// this ip is covered by this try block

					std::stringstream try_cmt;
					try_cmt << "try_" << try_num << " {";
					
					// add the unwind action if it exists
					if (states.count(ip_state->second))
						try_cmt << " <" << states[ip_state->second] << ">";
					
					add_long_cmt(ip_state->first, true, try_cmt.str().c_str());

					// find the ending ip
					ea_t tryblock_end;
					get_block_end(ip_state, tryblock_end);

					if (tryblock_end != BADADDR)
					{
						std::stringstream catch_cmt;
						// give the first line the ending brace
						catch_cmt << "} ";

						for (auto j = tryblock->handlers.cbegin(); j != tryblock->handlers.cend(); ++j)
						{
							catch_cmt << "catch_" << try_num << " (";
							if (j->adjectives & j->ADJ_STDDOTDOT)
								catch_cmt << "...";
							else
								catch_cmt << std::hex << j->type;
							catch_cmt << ")";

							if (j->handler.rva != 0)
								catch_cmt << " <" << std::hex << j->handler << ">";

							add_long_cmt(tryblock_end, true, catch_cmt.str().c_str());
							
							catch_cmt.str("");
						}
					}
					else
					{
						msg("Couldn't find end of c++ try block beginning at %a\n", ip_state->first);
					}

					try_num++;
				}
				else
				{
					// check to see if there are unwind actions for this ip
					mark_unwind_region(ip_state, try_num, states);
				}
			}
		}

		// If there are no try blocks, make sure to mark any unwind regions

		if (try_block_map.map.empty())
		{
			u32 unwind_num = 0;
			
			for (auto ip_state = ip_state_map.map.cbegin(); ip_state != ip_state_map.map.cend(); ++ip_state)
			{
				mark_unwind_region(ip_state, unwind_num, states);
			}
		}
	}

	// returns the next ip in ip2state map, or the function end.
	// returns BADADDR if it fails
	void get_block_end(
		std::vector<IpToStateMap::IpToStateMapEntry>::const_iterator &ip_state,
		ea_t &end) const
	{
		// find the ending ip
		end = BADADDR;
		if (ip_state + 1 != ip_state_map.map.cend())
		{
			end = (ip_state + 1)->first;
		}
		else
		{
			func_t *func = get_prev_func(ip_state->first);
			area_t limits;
			if (get_func_limits(func, &limits))
				end = limits.endEA;
		}
	}

	void mark_unwind_region(
		std::vector<IpToStateMap::IpToStateMapEntry>::const_iterator &ip_state,
		u32 &num, state_chain_t &states) const
	{
		// check to see if it's state requires an unwind action

		if (states.count(ip_state->second))
		{
			std::stringstream unwind_cmt;
			unwind_cmt << "unwind_" << num << " { <" << states[ip_state->second] << ">";
			add_long_cmt(ip_state->first, true, unwind_cmt.str().c_str());

			ea_t unwind_end;
			get_block_end(ip_state, unwind_end);

			if (unwind_end != BADADDR)
			{
				unwind_cmt.str("");
				unwind_cmt << "} // unwind_" << num;
				add_long_cmt(unwind_end, true, unwind_cmt.str().c_str());
			}

			num++;
		}
	}

	void print() const
	{
		msg("max_state %x unwindmap %a num_try_blocks %x try_blockmap %a\n"
			"num_ip_to_statemap_entries %x ip_to_statemap %a unwind_help %i\n"
			"es_typelist %a eh_flags %8x\n",
			max_state, unwindmap_rva.get(), num_try_blocks, try_blockmap_rva.get(),
			num_ip_to_statemap_entries, ip_to_statemap_rva.get(), unwind_help,
			es_typelist.get(), eh_flags);
		unwind_map.print();
		try_block_map.print();
		ip_state_map.print();
	}
};

struct lang_specific_data
{
	s32 gs_cookie_offset;

	SEHScopeTable seh_table;

	u32 gs_handlercheck_value;

	rva_t func_info_addr;
	FuncInfo function_info;

	enum LangType
	{
		NOT_FOUND,
		SEH,		// __C_specific_handler
		Cxx,		// __CxxFrameHandler
		GSCheck = 4,// __GSHandlerCheck

		GS_cookie_valid = 0x100
	};
	uint type;

	lang_specific_data() : type(NOT_FOUND) {}

	lang_specific_data(rva_t address)
	{
		determine_lang_type(address);

		parse(address);
	}
	
	// Fills in gs_cookie_offset if detected
	void determine_lang_type(rva_t &address)
	{
		type = NOT_FOUND;
		s32 peek_word = address.read();

		// if the first word is negative, it's probably a GS cookie offset
		if (peek_word < 0)
		{
			gs_cookie_offset = peek_word;
			type = type | GS_cookie_valid;

			// TODO doesn't exist on arm, does it exist on amd64?
			if (idainfo.IsAMD64() && (gs_cookie_offset & 4))
			{
				msg("gs_cookie_offset @ %a (%8x) HAS FLAG!\n",
					address.get(), gs_cookie_offset);
			}

			address.rva += sizeof(gs_cookie_offset);
			peek_word = address.read();
		}

		// if the first actual word is small, assume it is an SEH scope count  
		if (peek_word > 0 && peek_word <= 0xff)
		{
			type |= SEH;
		}
		// if the first word is a plausible aligned RVA, assume it is a C++ EH
		else if (peek_word > 0 && (peek_word & 3) == 0)
		{
			type |= Cxx;
		}
	}

	void parse(rva_t &address)
	{
		switch (type)
		{
		case SEH:
			seh_table = SEHScopeTable(rva_t(address, rva_t::absolute));
			if (!seh_table.is_valid())
			{
				seh_table = SEHScopeTable();

				type &= GS_cookie_valid;
				type |= GSCheck;

				parse(address);
			}
			break;
		case Cxx:
			func_info_addr = address.read();
			function_info = FuncInfo(func_info_addr);
			if (!function_info.is_valid())
			{
				func_info_addr = 0;
				function_info = FuncInfo();

				type &= GS_cookie_valid;
				type |= GSCheck;

				parse(address);
			}
			break;
		case GSCheck:
			gs_handlercheck_value = address.read();
			break;
		default:
			type = NOT_FOUND;
			break;
		}
	}

	void print() const
	{
		if (type & GS_cookie_valid)
			msg("gs_cookie_offset %i\n", gs_cookie_offset);

		switch (type & ~GS_cookie_valid)
		{
		case SEH:
			seh_table.print();
			break;
		case Cxx:
			msg("C++ EH: %a\n", func_info_addr.get());
			function_info.print();
			break;
		case GSCheck:
			msg("gs_handlercheck_value %x\n", gs_handlercheck_value);
			break;
		default:
			break;
		}
	}
};

struct handler_info
{
	rva_t exception_handler_rva;
	lang_specific_data lang_data;

	handler_info() {}

	handler_info(rva_t address)
	{
		exception_handler_rva = rva_t(address, rva_t::absolute).read();
		address.rva += sizeof(u32);

		lang_data = lang_specific_data(address);

		CodeMarker::set_range(exception_handler_rva, true);
	}

	void print() const
	{
		msg("exception_handler %a\n", exception_handler_rva.get());
		lang_data.print();
	}
};

////////////////////////////////////////////////////////////////////////////////
// ARM

struct pdata_record
{
	enum { record_size = sizeof(u32) * 2 };

	rva_t func_rva;

	union
	{
		ea_t exception_info_rva;

		Bitfield< 0,  2, ea_t> flag;
		Bitfield< 2, 11, ea_t> func_length;
		Bitfield<13,  2, ea_t> ret;
		Bitfield<15,  1, ea_t> H;
		Bitfield<16,  3, ea_t> reg;
		Bitfield<19,  1, ea_t> R;
		Bitfield<20,  1, ea_t> L;
		Bitfield<21,  1, ea_t> C;
		Bitfield<22, 10, ea_t> stack_adjust;
	};

	pdata_record() : func_rva(), exception_info_rva() {}

	pdata_record(rva_t address)
	{
		func_rva = address.read_inc();
		exception_info_rva = address.read();
	}

	bool is_valid() const
	{
		return func_rva.rva != 0 && exception_info_rva != 0;
	}
};

struct xdata_record
{
	// 1
	struct
	{
		union
		{
			u32 _1;

			Bitfield<0, 18> func_length;
			Bitfield<18, 2> version;
			Bitfield<20> X;
			Bitfield<21> E;
			Bitfield<22> F;
			Bitfield<23, 5> epilog_count;
			Bitfield<28, 4> code_words;
		};

		union
		{
			u32 _2;

			Bitfield<0, 16> epilog_count_extended;
			Bitfield<16, 8> code_words_extended;
			Bitfield<24, 8> reserved;
		};
	};

	// 2
	union epilog_scope
	{
		u32 _3;

		Bitfield<0, 18> epilog_start_offset;
		Bitfield<18, 2> reserved;
		Bitfield<20, 4> condition;
		Bitfield<24, 8> epilog_start_index;

		epilog_scope(rva_t const address)
		{
			_3 = address.read();
		}
	};
	std::vector<epilog_scope> epilog_scopes;

	// 3
	typedef u8 unwind_code;
	std::vector<unwind_code> unwind_codes;

	// 4
	handler_info handler_data;

	bool has_extension_word() const
	{
		return epilog_count == 0 && code_words == 0;
	}

	u32 real_epilog_count() const
	{
		if (has_extension_word() && E == 0)
			return epilog_count_extended;
		else
			return epilog_count;
	}
	
	// "codeword" is 32bits (4 codes)
	u32 real_code_count() const
	{
		if (has_extension_word())
			return code_words_extended * sizeof(u32);
		else
			return code_words * sizeof(u32);
	}

	xdata_record() {}

	xdata_record(rva_t address)
	{
		// header
		_1 = address.read_inc();

		if (has_extension_word())
		{
			_2 = address.read_inc();
		}

		// epilog scopes
		if (E == 0)
		{
			for (uint i = 0; i < real_epilog_count(); ++i)
			{
				epilog_scopes.push_back(address);
				address.rva += sizeof(u32);
			}
		}

		// unwind codes
		for (uint i = 0; i < real_code_count(); ++i)
		{
			unwind_codes.push_back(static_cast<u8>(address.read_inc(1)));
		}

		// exception handler information
		if (X == 1)
		{
			handler_data = handler_info(address);
		}

		//msg("(ended at addr %a)\n", address.get());
	}
};

struct exception_record_arm
{
	pdata_record pdata;
	xdata_record xdata;

	enum { record_size = 2 * sizeof(u32) };

	exception_record_arm(ea_t address)
	{
		pdata = pdata_record(rva_t(address, rva_t::absolute));

		rva_t exception_info_addr = rva_t(pdata.exception_info_rva);
		if (pdata.flag == 0)
		{
			xdata = xdata_record(exception_info_addr);
		}

		ea_t func_length = (pdata.flag == 0) ? (ea_t)xdata.func_length : pdata.func_length;
		CodeMarker::set_range(pdata.func_rva, pdata.func_rva + func_length, true);
		
#ifdef UNWIND_PRINT
		msg("pdata @ %a: func_rva %a ", address, pdata.func_rva.get());
		if (pdata.flag == 0)
			msg("exception_info_rva %a", exception_info_addr.get());
		else
			msg("stack_adjust %3x C %x L %x R %x reg %x H %x ret %x func_length %3x flag %x",
			(ea_t)pdata.stack_adjust,(ea_t)pdata.C,(ea_t)pdata.L,(ea_t)pdata.R,(ea_t)pdata.reg,
			(ea_t)pdata.H,(ea_t)pdata.ret,(ea_t)pdata.func_length*2,(ea_t)pdata.flag);
		msg("\n");

		if (pdata.flag == 0)
		{
			msg("xdata @ %a ", exception_info_addr.get());
			msg("code_words %2x epilog_count %2x F %x E %x X %x version %x func_length %5x\n",
				(u32)xdata.code_words,(u32)xdata.epilog_count,(u32)xdata.F,(u32)xdata.E,(u32)xdata.X,
				(u32)xdata.version,(u32)xdata.func_length * 2);
			if (xdata.E != 0)
				msg("%x is the only epilog index\n", (u32)xdata.epilog_count);
			std::for_each(xdata.epilog_scopes.cbegin(), xdata.epilog_scopes.cend(),
				[](xdata_record::epilog_scope const &scope)
			{
				msg("epilog_start_index %2x condition %x reserved %x epilog_start_offset %3x\n",
					(u32)scope.epilog_start_index,(u32)scope.condition,(u32)scope.reserved,
					(u32)scope.epilog_start_offset * 2);
			});
			std::for_each(xdata.unwind_codes.cbegin(), xdata.unwind_codes.cend(),
				[](xdata_record::unwind_code const &ucode)
			{
				msg("unwind code %x\n", ucode);
			});
			if (xdata.X == 1)
				xdata.handler_data.print();
		}
#endif
	}
};

////////////////////////////////////////////////////////////////////////////////
// AMD64

struct runtime_function
{
	rva_t begin;
	rva_t end;
	rva_t unwind;

	enum { indirect = 1 };

	runtime_function() : begin(), end(), unwind() {}

	runtime_function(rva_t address)
	{
		begin = address.read_inc();
		end = address.read_inc();
		unwind = address.read_inc();
	}

	bool is_valid() const
	{
		return begin.rva != 0 && end.rva != 0 && unwind.rva != 0;
	}
};

struct unwind_info
{
	struct unwind_code
	{
		enum OpCode
		{
			PUSH_NONVOL,
			ALLOC_LARGE,
			ALLOC_SMALL,
			SET_FPREG,
			SAVE_NONVOL,
			SAVE_NONVOL_FAR,
			SAVE_XMM,
			SAVE_XMM_FAR,
			SAVE_XMM128,
			SAVE_XMM128_FAR,
			SAVE_MACHFRAME
		};

		u8 code_offset;
		union
		{
			u8 opcode;
			Bitfield<0, 4, u8> op;
			Bitfield<4, 4, u8> info;
		};
	};

	union
	{
		enum
		{
			NHANDLER = 0,
			EHANDLER = 1,
			UHANDLER = 2,
			CHAININFO = 4
		};
		u8 header;
		Bitfield<0, 3, u8> version;
		Bitfield<3, 5, u8> flags;
	};
	u8 prolog_size;
	u8 num_codes;
	union
	{
		u8 frame_info;
		Bitfield<0, 4, u8> frame_register;
		Bitfield<4, 4, u8> frame_offset;
	};
	std::vector<unwind_code> unwind_codes;

	handler_info handler_data;
	
	unwind_info() {}

	unwind_info(rva_t function, rva_t address)
	{
		header = static_cast<u8>(address.read_inc(1));
		prolog_size = static_cast<u8>(address.read_inc(1));
		num_codes = static_cast<u8>(address.read_inc(1));
		frame_info = static_cast<u8>(address.read_inc(1));

		for (u8 i = 0; i < num_codes; ++i)
		{
			unwind_code ucode;
			ucode.code_offset = static_cast<u8>(address.read_inc(1));
			ucode.opcode = static_cast<u8>(address.read_inc(1));
			unwind_codes.push_back(ucode);
		}

		// align
		address = (address.rva + 3) & ~3;

		if (flags == CHAININFO)
		{
			unwind_info(function, runtime_function(address).unwind);
		}
		else if (flags & (EHANDLER | UHANDLER))
		{
			handler_data = handler_info(address);

			// If lang specific data wasn't found, make sure to still show a
			// comment about the handler

			if (handler_data.lang_data.type == handler_data.lang_data.NOT_FOUND)
			{
				auto protected_func = get_func(function);
				auto handler_func = get_func(handler_data.exception_handler_rva);
				if (protected_func && handler_func)
				{
					std::stringstream comment;
					qstring fname;

					comment << "Exception Handler: ";
					if (get_func_name2(&fname, handler_data.exception_handler_rva))
						comment << fname.c_str();
					comment << "(" << std::hex << handler_data.exception_handler_rva << ")";
					set_func_cmt(protected_func, comment.str().c_str(), false);

					comment.str("");

					comment << "Exception Handler for: ";
					if (get_func_name2(&fname, function))
						comment << fname.c_str();
					comment << "(" << std::hex << function << ")";
					set_func_cmt(handler_func, comment.str().c_str(), false);
				}
				else
				{
					msg("failed to get fnptrs to comment about exceptions from %a being handled by %a\n",
						function.get(), handler_data.exception_handler_rva.get());
				}
			}
		}
	}
};

struct exception_record_amd64
{
	runtime_function pdata;
	unwind_info xdata;

	enum { record_size = 3 * sizeof(u32) };

	exception_record_amd64(ea_t address)
	{
		pdata = runtime_function(rva_t(address, rva_t::absolute));

		while (pdata.unwind.rva & pdata.indirect)
		{
			pdata.unwind.rva &= ~pdata.indirect;
#ifdef UNWIND_PRINT
			msg("pdata -> %a\n", pdata.unwind.get());
#endif

			CodeMarker::set_range(pdata.begin, pdata.end);

			pdata = runtime_function(pdata.unwind);
		}

		CodeMarker::set_range(pdata.begin, pdata.end);

		xdata = unwind_info(pdata.begin, pdata.unwind);

#ifdef UNWIND_PRINT
		msg("pdata @ %a begin %a end %a unwind %a\n",
			address, pdata.begin.get(), pdata.end.get(), pdata.unwind.get());

		msg("version %x flags %2x prolog_size %2x num_codes %2x frame_register %x frame_offset %x\n",
			xdata.version.operator u8(), xdata.flags.operator u8(),
			xdata.prolog_size,
			xdata.num_codes,
			xdata.frame_register.operator u8(), xdata.frame_offset.operator u8());
		std::for_each(xdata.unwind_codes.begin(), xdata.unwind_codes.end(),
			[](unwind_info::unwind_code const &ucode)
		{
			msg("code_offset %2x op %x info %x\n",
				ucode.code_offset,
				ucode.op.operator u8(), ucode.info.operator u8());
		});
		if (xdata.flags & (xdata.EHANDLER | xdata.UHANDLER))
			xdata.handler_data.print();
#endif
	}
};
