#include "PrecompiledHeader.h"
#include "Types.h"

static u64 coverage_begin = 0;
u64 calc_coverage()
{
	u64 num = 0;
#ifdef UNWIND_MARK_CODE_STATS
	for (auto seg = get_first_seg(); seg != nullptr; seg = get_next_seg(seg->startEA))
	{
		if (seg->perm & SEGPERM_EXEC)
		{
			func_item_iterator_t fii;
			fii.set_range(seg->startEA, seg->endEA);
			do
			{
				ea_t ea = fii.current();
				if (decode_insn(ea))
				{
					++num;
				}
			} while (fii.next_code());
		}
	}
#endif
	return num;
}

int idaapi code_stats(void *, int notification_code, va_list va)
{
	if (notification_code == processor_t::auto_queue_empty)
	{
		auto coverage_end = calc_coverage();
		auto coverage_diff = coverage_end - coverage_begin;
		auto coverage_inc = (coverage_end && coverage_begin)
			? ((static_cast<double>(coverage_end) / coverage_begin) - 1.) * 100.
			: 0;
		// '\r' doesn't work..oh well, spam
		msg("%lli more instructions found (%f%% more[%lli->%lli])\n",
			coverage_diff, coverage_inc, coverage_begin, coverage_end);
	}

	return 0;
}

int idaapi init()
{
	idainfo.Update();

	if (idainfo.IsWindows() && (idainfo.IsAMD64() || idainfo.IsARM()))
		return PLUGIN_KEEP;
	else
		return PLUGIN_SKIP;
}

void idaapi term()
{
#ifdef UNWIND_MARK_CODE_STATS
	unhook_from_notification_point(HT_IDP, code_stats, nullptr);
#endif
}

void idaapi run(int arg)
{
	// find p/xdata segments
	std::vector<segment_t *> pdata_segments;

	for (auto cur_segment = get_first_seg(); cur_segment != nullptr; cur_segment = get_next_seg(cur_segment->startEA))
	{
		auto name = std::vector<char>(MAXSTR);
		auto const size_copied = get_true_segm_name(cur_segment, name.data(), name.size());
		if (size_copied != -1) {
			if (!strcmp(name.data(), ".pdata"))
				pdata_segments.push_back(cur_segment);
		}
	}

	if (coverage_begin == 0)
		coverage_begin = calc_coverage();
	auto time_begin = std::chrono::steady_clock::now();

	if (idainfo.IsARM())
	{
		for (auto it = pdata_segments.cbegin(); it != pdata_segments.cend(); ++it)
		{
			for (auto ea = (*it)->startEA; ea < (*it)->endEA; ea += exception_record_arm::record_size)
			{
				pdata_record pdata(rva_t(ea, rva_t::addr_type::absolute));
				if (!pdata.is_valid())
					break;

				exception_record_arm exception_record(ea);
			}
		}
	}
	else
	{
		for (auto it = pdata_segments.cbegin(); it != pdata_segments.cend(); ++it)
		{
			for (auto ea = (*it)->startEA; ea < (*it)->endEA; ea += exception_record_amd64::record_size)
			{
				runtime_function pdata(rva_t(ea, rva_t::addr_type::absolute));
				if (!pdata.is_valid())
					break;

				exception_record_amd64 exception_record(ea);
			}
		}
	}

	auto time_diff = std::chrono::duration_cast<std::chrono::milliseconds>(std::chrono::steady_clock::now() - time_begin);
	msg("- parsing took %ims\n", time_diff.count());

#ifdef UNWIND_MARK_CODE_STATS
	msg("waiting for ida to finish analysis...\n");
	hook_to_notification_point(HT_IDP, code_stats, nullptr);
#endif
}

plugin_t PLUGIN =
{
	IDP_INTERFACE_VERSION,
	0,								// plugin flags
	init,							// initialize
	term,							// terminate. this pointer may be NULL.
	run,							// invoke plugin
	nullptr,						// long comment about the plugin
	nullptr,						// multiline help about the plugin
	"PE exception data helper",		// the preferred short name of the plugin
	"Ctrl+F1"						// the preferred hotkey to run the plugin
};
